/**
 * Copyright (c) 2002-2014 "Neo Technology,"
 * Network Engine for Objects in Lund AB [http://neotechnology.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.cypher.internal.compiler.v2_1.planner

import org.neo4j.cypher.internal.compiler.v2_1.ast._
import org.neo4j.cypher.internal.compiler.v2_1.spi.TokenContext
import org.neo4j.cypher.internal.compiler.v2_1.LabelId
import org.neo4j.cypher.internal.compiler.v2_1.PropertyKeyId
import org.neo4j.cypher.internal.compiler.v2_1.ast.Query
import org.neo4j.cypher.internal.compiler.v2_1.RelTypeId

class SimpleTokenResolver {
  def resolve(ast: Query)(implicit semanticTable: SemanticTable, tokenContext: TokenContext) {
    val (propertyKeyNames, labelNames, relTypeNames) =
      ast.treeFold((Set.empty[PropertyKeyName], Set.empty[LabelName], Set.empty[RelTypeName]))({
        case token: PropertyKeyName => {
          case ((propKeyNames, labelNames, relTypeNames), c) =>
            (propKeyNames + token, labelNames, relTypeNames)
        }
        case token: LabelName => {
          case ((propKeyNames, labelNames, relTypeNames), _) =>
            (propKeyNames, labelNames + token, relTypeNames)
        }
        case token: RelTypeName => {
          case ((propKeyNames, labelNames, relTypeNames), _) =>
            (propKeyNames, labelNames, relTypeNames + token)
        }
      })

    propertyKeyNames.foreach{ x => propertyKeyId(x.name) }
    labelNames.foreach{ x => labelId(x.name) }
    relTypeNames.foreach{ x => relTypeId(x.name) }
  }

  private def propertyKeyId(name: String)(implicit semanticTable: SemanticTable, tokenContext: TokenContext) {
    tokenContext.getOptPropertyKeyId(name).map(PropertyKeyId) match {
      case Some(id) =>
        semanticTable.resolvedPropertyKeyNames += name -> id
      case None =>
    }
  }

  private def labelId(name: String)(implicit semanticTable: SemanticTable, tokenContext: TokenContext) {
    tokenContext.getOptLabelId(name).map(LabelId) match {
      case Some(id) =>
        semanticTable.resolvedLabelIds += name -> id
      case None =>
    }
  }

  private def relTypeId(name: String)(implicit semanticTable: SemanticTable, tokenContext: TokenContext) {
    tokenContext.getOptRelTypeId(name).map(RelTypeId) match {
      case Some(id) =>
        semanticTable.resolvedRelTypeNames += name -> id
      case None =>
    }
  }
}
