/**
 * Copyright (c) 2002-2014 "Neo Technology,"
 * Network Engine for Objects in Lund AB [http://neotechnology.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.cypher.internal.compiler.v2_1.pipes

import org.neo4j.cypher.internal.compiler.v2_1._
import commands.expressions.{CachedExpression, Identifier, Expression}
import symbols._
import org.neo4j.cypher.internal.compiler.v2_1.PlanDescription.Arguments.KeyNames

object ExtractPipe {
  def apply(source: Pipe, expressions: Map[String, Expression])(implicit pipeMonitor: PipeMonitor): ExtractPipe = source match {
      // If we can merge the two pipes together, do it
    case p: ExtractPipe if expressions.values.forall(_.symbolDependenciesMet(p.source.symbols)) =>
      new ExtractPipe(p.source, p.expressions ++ expressions, true)

    case _              =>
      new ExtractPipe(source, expressions, true)
  }
}

case class ExtractPipe(source: Pipe, expressions: Map[String, Expression], hack_remove_this:Boolean)
                      (implicit pipeMonitor: PipeMonitor) extends PipeWithSource(source, pipeMonitor) {
  val symbols: SymbolTable = {
    val newIdentifiers = expressions.map {
      case (name, expression) => name -> expression.getType(source.symbols)
    }

    source.symbols.add(newIdentifiers)
  }

  /*
  Most of the time, we can execute expressions and put the results straight back into the original execution context.
  Some times, an expression we want to run can overwrite an identifier that already exists in the context. In these
  cases, we need to run the expressions on the original execution context. Here we decide wich one it is we're dealing
  with and hard code the version to use
   */
  val applyExpressions: (ExecutionContext, QueryState) => ExecutionContext = {
    val overwritesAlreadyExistingIdentifiers = expressions.exists {
      case (name, Identifier(originalName)) => name != originalName
      case (name, CachedExpression(originalName, _)) => name != originalName
      case (name, _) => source.symbols.hasIdentifierNamed(name)
    }

    val applyExpressionsOverwritingOriginal = (ctx: ExecutionContext, state: QueryState) => {
      expressions.foreach {
        case (name, expression) =>
          ctx += name -> expression(ctx)(state)
      }
      ctx
    }
    val applyExpressionsWhileKeepingOriginal = (ctx: ExecutionContext, state: QueryState) => {
      val original = ctx.clone()
      expressions.foreach {
        case (name, expression) =>
          ctx += name -> expression(original)(state)
      }
      ctx
    }

    if (overwritesAlreadyExistingIdentifiers)
      applyExpressionsWhileKeepingOriginal
    else
      applyExpressionsOverwritingOriginal
  }

  protected def internalCreateResults(input: Iterator[ExecutionContext], state: QueryState) =
    input.map( ctx => applyExpressions(ctx, state) )

  override def planDescription = {
    val arguments = expressions.map(_._1).toSeq

    new PlanDescriptionImpl(this, "Extract", SingleChild(source.planDescription), Seq(KeyNames(arguments)))
  }
}

