/*
 * Joda Software License, Version 1.0
 *
 *
 * Copyright (c) 2001-03 Stephen Colebourne.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Joda project (http://www.joda.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "Joda" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact licence@joda.org.
 *
 * 5. Products derived from this software may not be called "Joda",
 *    nor may "Joda" appear in their name, without prior written
 *    permission of the Joda project.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE JODA AUTHORS OR THE PROJECT
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Joda project and was originally
 * created by Stephen Colebourne <scolebourne@joda.org>. For more
 * information on the Joda project, please see <http://www.joda.org/>.
 */
package org.joda.time.format;

import java.io.IOException;
import java.io.Writer;
import java.lang.reflect.Method;
import java.text.ParseException;

import org.joda.time.Chronology;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.MutableDateTime;
import org.joda.time.ReadWritableInstant;
import org.joda.time.ReadableInstant;

/**
 * Abstract base class for implementing {@link DateTimePrinter}s,
 * {@link DateTimeParser}s, and {@link DateTimeFormatter}s. This class
 * intentionally does not implement any of those interfaces. You can subclass
 * and implement only the interfaces that you need to.
 * <p>
 * The print methods assume that your subclass has implemented DateTimePrinter or
 * DateTimeFormatter. If not, a ClassCastException is thrown when calling those
 * methods.
 * <p>
 * Likewise, the parse methods assume that your subclass has implemented
 * DateTimeParser or DateTimeFormatter. If not, a ClassCastException is thrown
 * when calling the parse methods.
 *
 * @author Brian S O'Neill
 */
public abstract class AbstractDateTimeFormatter {

    private static Method cInitCauseMethod;

    static {
        // cope with JDK 1.4 enhancements
        Method initCauseMethod = null;
        try {
            initCauseMethod = Throwable.class.getMethod
                ("initCause", new Class[] {Throwable.class});
            
        } catch (NoSuchMethodException ex) {
            // ignore
        } catch (SecurityException ex) {
            // ignore
        }
        cInitCauseMethod = initCauseMethod;
    }

    // Accessed also by AbstractDurationFormatter.
    static String createErrorMessage(String text, int errorPos) {
        int sampleLen = errorPos + 20;
        String sampleText;
        if (text.length() <= sampleLen) {
            sampleText = text;
        } else {
            sampleText = text.substring(0, sampleLen).concat("...");
        }
        
        if (errorPos <= 0) {
            return "Invalid format: \"" + sampleText + '"';
        }
        
        if (errorPos >= text.length()) {
            return "Invalid format: \"" + sampleText + "\" is too short";
        }
        
        return "Invalid format: \"" + sampleText + "\" is malformed at \"" +
            sampleText.substring(errorPos) + '"';
    }

    private static void setCause(ParseException pe, Throwable initCause) {
        if (cInitCauseMethod != null) {
            try {
                cInitCauseMethod.invoke(pe, new Object[]{initCause});
            } catch (Exception e) {
                cInitCauseMethod = null;
            }
        }
    }

    /**
     * Returns the Chronology being used by the formatter, or null if none.
     */
    public abstract Chronology getChronology();

    public void printTo(StringBuffer buf, ReadableInstant instant) {
        long millisUTC = instant.getMillis();
        Chronology chrono;
        if ((chrono = instant.getChronology()) != null) {
            printTo(buf, millisUTC, chrono.getDateTimeZone());
        } else {
            ((DateTimePrinter)this).printTo(buf, millisUTC, null, millisUTC);
        }
    }

    public void printTo(Writer out, ReadableInstant instant) throws IOException {
        long millisUTC = instant.getMillis();
        Chronology chrono;
        if ((chrono = instant.getChronology()) != null) {
            printTo(out, millisUTC, chrono.getDateTimeZone());
        } else {
            ((DateTimePrinter)this).printTo(out, millisUTC, null, millisUTC);
        }
    }

    public void printTo(StringBuffer buf, long millisUTC) {
        printTo(buf, millisUTC, null);
    }

    public void printTo(Writer out, long millisUTC) throws IOException {
        printTo(out, millisUTC, null);
    }

    public void printTo(StringBuffer buf, long millisUTC, DateTimeZone zone) {
        if (zone != null) {
            ((DateTimePrinter)this).printTo
                (buf, millisUTC, zone, millisUTC + zone.getOffset(millisUTC));
        } else {
            ((DateTimePrinter)this).printTo(buf, millisUTC, null, millisUTC);
        }
    }

    public void printTo(Writer out, long millisUTC, DateTimeZone zone) throws IOException {
        if (zone != null) {
            ((DateTimePrinter)this).printTo
                (out, millisUTC, zone, millisUTC + zone.getOffset(millisUTC));
        } else {
            ((DateTimePrinter)this).printTo(out, millisUTC, null, millisUTC);
        }
    }

    public String print(ReadableInstant instant) {
        long millisUTC = instant.getMillis();
        Chronology chrono;
        if ((chrono = instant.getChronology()) != null) {
            return print(millisUTC, chrono.getDateTimeZone());
        } else {
            return print(millisUTC, null, millisUTC);
        }
    }

    public String print(long millisUTC) {
        return print(millisUTC, null);
    }

    public String print(long millisUTC, DateTimeZone zone) {
        if (zone != null) {
            return print
                (millisUTC, zone, millisUTC + zone.getOffset(millisUTC));
        } else {
            return print(millisUTC, null, millisUTC);
        }
    }

    public String print(long millisUTC, DateTimeZone zone, long millisLocal) {
        DateTimePrinter p = (DateTimePrinter)this;
        StringBuffer buf = new StringBuffer(p.estimatePrintedLength());
        p.printTo(buf, millisUTC, zone, millisLocal);
        return buf.toString();
    }

    public int parseInto(ReadWritableInstant instant, String text, int position) {
        DateTimeParser p = (DateTimeParser)this;

        long millis = instant.getMillis();
        Chronology chrono = instant.getChronology();
        if (chrono != null) {
            DateTimeZone zone = chrono.getDateTimeZone();
            if (zone != null) {
                // Move millis to local time.
                millis += zone.getOffset(millis);
            }
        }

        DateTimeParserBucket bucket = createBucket(millis);
        position = p.parseInto(bucket, text, position);
        instant.setMillis(bucket.computeMillis());
        return position;
    }
    
    public long parseMillis(String text) throws ParseException {
        return parseMillis(text, 0);
    }

    public long parseMillis(String text, long millis) throws ParseException {
        DateTimeParser p = (DateTimeParser)this;
        DateTimeParserBucket bucket = createBucket(millis);

        int newPos = p.parseInto(bucket, text, 0);
        if (newPos >= 0) {
            if (newPos >= text.length()) {
                try {
                    return bucket.computeMillis();
                } catch (IllegalArgumentException ex) {
                    ParseException pe = new ParseException(ex.getMessage(), 0);
                    setCause(pe, ex);
                    throw pe;
                }
            }
        } else {
            newPos = ~newPos;
        }

        throw new ParseException(createErrorMessage(text, newPos), newPos);
    }

    public DateTime parseDateTime(String text) throws ParseException {
        return new DateTime(parseMillis(text), getChronology());
    }

    public MutableDateTime parseMutableDateTime(String text) throws ParseException {
        return new MutableDateTime(parseMillis(text), getChronology());
    }

    private DateTimeParserBucket createBucket(long millis) {
        DateTimeParserBucket bucket = new DateTimeParserBucket(millis);
        Chronology chrono = getChronology();
        if (chrono != null) {
            DateTimeZone zone = chrono.getDateTimeZone();
            if (zone != null) {
                bucket.setDateTimeZone(zone);
            }
        }
        return bucket;
    }

}
