/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package org.jfrog.support.rest.model;

import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonNaming;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateUtils;

import javax.validation.constraints.NotNull;
import java.io.IOException;
import java.text.ParseException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;

/**
 * @author Tamir Hadad
 */
@Builder
@Data
@NoArgsConstructor
@AllArgsConstructor
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonNaming(PropertyNamingStrategy.SnakeCaseStrategy.class)
@JsonInclude(JsonInclude.Include.NON_NULL)
public class SupportBundleConfig {
    private String name;
    private String description;
    private String id;
    private Artifactory artifactory;
    private SupportBundleParameters parameters;
    private Integer available;
    @JsonDeserialize(using = DateTimeWithFallbackDeSerializer.class)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = Formats.DATE_TIME_WITH_ZONE)
    private ZonedDateTime created;
    private BundleCreationStatus status;

    @Builder
    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    @JsonIgnoreProperties(ignoreUnknown = true)
    @JsonNaming(PropertyNamingStrategy.SnakeCaseStrategy.class)
    public static class Artifactory {
        @NotNull
        private String serviceId;
        @NotNull
        private String bundleUrl;
    }

    @Slf4j
    static class DateTimeWithFallbackDeSerializer extends JsonDeserializer<ZonedDateTime> {

        @Override
        public ZonedDateTime deserialize(JsonParser jsonParser, DeserializationContext deserializationContext)
                throws IOException {
            try {
                Instant instant = DateUtils.parseDate(jsonParser.getText(), Formats.DATE_TIME_WITH_ZONE).toInstant();
                return ZonedDateTime.ofInstant(instant, ZoneId.systemDefault());
            } catch (ParseException e) {
                log.warn("Error parsing original support bundle creation time: {}. error: {}. will try other parsers",
                        jsonParser.getText(), e.getMessage());
                try {
                    Instant instant = DateUtils.parseDate(jsonParser.getText(), Formats.DATE_TIME_OTHER_KNOWN_PATTERNS)
                            .toInstant();
                    return ZonedDateTime.ofInstant(instant, ZoneId.systemDefault());
                } catch (ParseException ex) {
                    log.warn("Failed parsing original support bundle creation time: {} with other parsers as well, " +
                            "returning default now()", jsonParser.getText());
                    return ZonedDateTime.now();
                }
            }
        }
    }
}
