package org.jfrog.support.common.core.collectors;

import org.jfrog.support.common.LogsFilesMatcher;
import org.jfrog.support.common.config.SystemLogsConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.attribute.FileTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.Optional;

/**
 * By default collects all .log and .zip files in the date range specified  at the {@link SystemLogsConfiguration}
 */
public class DefaultFilesMatcher implements LogsFilesMatcher {
    private final Logger log = LoggerFactory.getLogger(DefaultFilesMatcher.class);

    @Override
    public boolean isMatch(Path filePath, SystemLogsConfiguration configuration) {
        String logsFilesPattern = "glob:**.log";
        String zippedLogsPattern = "glob:**.zip";
        log.debug("Matching file {}", filePath);
        PathMatcher pathMatcher = FileSystems.getDefault().getPathMatcher(logsFilesPattern);
        PathMatcher zippedPathMatcher = FileSystems.getDefault().getPathMatcher(zippedLogsPattern);
        if (pathMatcher.matches(filePath) || zippedPathMatcher.matches(filePath)) {
            long fileModifiedTime = getFileModifiedTimeAsEpoch(filePath);
            long minTime = getMinTimeAsEpoch(configuration);
            long maxTime = getMaxTimeAsEpoch(configuration);
            return isDateBetween(fileModifiedTime, minTime, maxTime);
        }

        log.debug("{} didn't matched any pattern by default collect the file as well.", filePath);
        return true;
    }

    private boolean isDateBetween(long fileModifiedTime, long minTime, long maxTime) {
        return fileModifiedTime >= minTime && fileModifiedTime <= maxTime;
    }

    private long getMaxTimeAsEpoch(SystemLogsConfiguration logsSettings) {
        return logsSettings.getEndDate().toInstant().atZone(ZoneId.systemDefault()).toLocalDate().atTime(23, 59, 59)
                .toEpochSecond(ZoneOffset.UTC) * 1000;
    }

    private long getMinTimeAsEpoch(SystemLogsConfiguration logsSettings) {
        return logsSettings.getStartDate().toInstant().atZone(ZoneId.systemDefault()).toLocalDate().atTime(0, 0, 0).toEpochSecond(ZoneOffset.UTC) * 1000;
    }

    private Long getFileModifiedTimeAsEpoch(Path path) {
        try {
            FileTime lastModifiedTime = Files.getLastModifiedTime(path);
            log.debug("File's last modified time: {}", lastModifiedTime);
            return Optional.ofNullable(lastModifiedTime)
                    .map(FileTime::toMillis)
                    .orElse(0L);
        } catch (IOException e) {
            log.warn("Cannot get file's last modified time: {}", e.getMessage());
            log.debug("Exception:", e);
            return 0L;
        }
    }
}
