/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package org.jfrog.support.common.core;

import com.google.common.base.Charsets;
import com.google.common.io.Files;
import org.jfrog.support.common.CollectConfiguration;
import org.jfrog.support.common.core.exceptions.ContentCollectionExceptionException;
import org.jfrog.support.common.utils.StringBuilderWrapper;

import java.io.File;
import java.io.IOException;

/**
 * Provides content generic services where all
 * items in the context requires same treatment
 *
 * @author Michael Pasternak
 */
public abstract class AbstractGenericContentCollector<T extends CollectConfiguration> extends AbstractContentCollector<T> {

    /**
     * @param contentName a name for specific sub-folder
     */
    protected AbstractGenericContentCollector(String contentName) {
        super(contentName);
    }

    /**
     * Collects security info
     *
     * @param configuration {@link CollectConfiguration}
     * @param tmpDir output dir
     *
     * @return result
     */
    protected final boolean doCollect(T configuration, File tmpDir) {
        if (configuration.isEnabled()) {
            try {
                StringBuilderWrapper content = doProduceContent(configuration);
                Files.write(content, getOutputFile(tmpDir), Charsets.UTF_8);
                getLog().info("Collection of " + getContentName() + " was successfully accomplished");
                return true;
            } catch (IOException | InstantiationException |
                    IllegalAccessException | ContentCollectionExceptionException e) {
                getLog().error("Collecting " + getContentName() + " has failed, - " + e.getMessage());
                getLog().debug("Cause: {}", e);
            }
        } else {
            getLog().debug("Content collection of " + getContentName() +" is disabled");
        }
        return false;
    }

    /**
     * @throws {@link ContentCollectionExceptionException} when no
     * content is available
     */
    protected StringBuilderWrapper failure() {
        throw new ContentCollectionExceptionException(
                "No content was collected for " + getContentName()
        );
    }

    /**
     * @throws {@link ContentCollectionExceptionException} when no
     * content is available
     */
    protected StringBuilderWrapper failure(Exception e) {
        throw new ContentCollectionExceptionException(
                "No content was collected for " + getContentName() +
                " - " + e.getMessage()
        );
    }

    /**
     * Produces content and returns it wrapped with {@link StringBuilderWrapper}
     *
     * @return {@link StringBuilderWrapper}
     *
     * @throws IOException
     * @throws InstantiationException
     * @throws IllegalAccessException
     */
    abstract protected StringBuilderWrapper doProduceContent(T configuration)
            throws IOException, InstantiationException, IllegalAccessException;
}
