/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package org.jfrog.support.common.core.collectors.system.info;

import com.fasterxml.jackson.core.util.DefaultIndenter;
import com.fasterxml.jackson.core.util.DefaultPrettyPrinter;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import com.google.common.collect.Multimap;
import org.jfrog.support.common.config.SystemInfoConfiguration;
import org.jfrog.support.common.core.AbstractGenericContentCollector;
import org.jfrog.support.common.core.exceptions.BundleConfigurationException;
import org.jfrog.support.common.utils.StringBuilderWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Collection;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * System info collector
 *
 * @author Michael Pasternak
 */
public class SystemInfoCollector extends AbstractGenericContentCollector<SystemInfoConfiguration> {
    private static final Logger log = LoggerFactory.getLogger(SystemInfoCollector.class);
    private static final ObjectWriter instance = initObjectMapper();

    public SystemInfoCollector() {
        super("system");
    }

    private static ObjectWriter initObjectMapper() {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.enable(SerializationFeature.WRITE_SINGLE_ELEM_ARRAYS_UNWRAPPED)
                .enable(SerializationFeature.INDENT_OUTPUT)
                .enable(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS)
                .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
                .registerModule(new ParameterNamesModule());
        DefaultPrettyPrinter prettyPrinter = new DefaultPrettyPrinter();
        prettyPrinter.indentArraysWith(DefaultIndenter.SYSTEM_LINEFEED_INSTANCE);
        return objectMapper.writer(prettyPrinter);
    }

    /**
     * Produces content and returns it wrapped with {@link StringBuilderWrapper}
     *
     * @param configuration the runtime configuration
     * @return {@link StringBuilderWrapper}
     *
     * @throws IOException
     * @throws InstantiationException
     * @throws IllegalAccessException
     */
    @Override
    protected StringBuilderWrapper doProduceContent(SystemInfoConfiguration configuration)
            throws InstantiationException,
            IllegalAccessException, IOException {
        Map<String, Multimap<String, String>> info = createInfoWriter().getInfoMap();
        if (!info.isEmpty()) {
            addProductCustomData(info);
            Map<String, Map<String, Collection<String>>> toJson = info.entrySet().stream()
                    .collect(Collectors.toMap(Map.Entry::getKey, entry -> entry.getValue().asMap()));
            return new StringBuilderWrapper(instance.writeValueAsString(toJson));
        } else {
            getLog().debug("No content was fetched from CommonInfoWriter");
        }
        return failure();
    }

    protected void addProductCustomData(Map<String, Multimap<String, String>> info) {
    }

    private CommonInfoWriter createInfoWriter() {
        return new CommonInfoWriter();
    }

    @Override
    public Logger getLog() {
        return log;
    }

    /**
     * Makes sure configuration is valid
     *
     * @param configuration configuration to check
     * @throws BundleConfigurationException if configuration is invalid
     */
    @Override
    protected void doEnsureConfiguration(SystemInfoConfiguration configuration)
            throws BundleConfigurationException {
    }

    /**
     * @return The filename to be used
     */
    @Override
    protected String getFileName() {
       return "system-info.json";
    }

}
