/*
 *
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2016 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.common;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.URI;
import java.net.URISyntaxException;

/**
 * @author yoavl
 */
public abstract class ResourceUtils {

    private static final Logger log = LoggerFactory.getLogger(ResourceUtils.class);

    private ResourceUtils() {
        // utility class
    }

    /**
     * Returns the specified resource input stream. Throws an exception if the resource is not found in the class path.
     *
     * @param path The resource path
     * @return The classpath resource input stream
     *
     * @see Class#getResourceAsStream(java.lang.String)
     */
    public static InputStream getResource(String path) {
        InputStream is = ResourceUtils.class.getResourceAsStream(path);
        assertResourceNotNull(path, is);
        return is;
    }

    /**
     * Returns the specified resource file handle. Throws an exception if the resource is not found in the class path
     * or if the path doesn't represent a file.
     *
     * @param path The resource file path
     * @return The file at the specified path
     *
     * @see Class#getResourceAsStream(java.lang.String)
     */
    public static File getResourceAsFile(String path) {
        try {
            URI uri = ResourceUtils.class.getResource(path).toURI();
            assertResourceNotNull(path, uri);
            return new File(uri);
        } catch (URISyntaxException e) {
            throw new RuntimeException("Error in finding file by URI for path='" + path + "'", e);
        } catch (Exception e) {
            // TODO: Handle and remove it.
            log.error("Failed to get Resource '{}' ", path, e);
            throw e;
        }
    }

    /**
     * Returns the specified resource content as UTF-8 string. Throws an exception if the resource is not found
     * in the class path.
     *
     * @param path The resource content as UTF-8 string
     * @return The classpath resource input stream
     *
     * @see Class#getResourceAsStream(java.lang.String)
     */
    public static String getResourceAsString(String path) {
        try (InputStream is = getResource(path)) {
            return IOUtils.toString(is, "utf-8");
        } catch (IOException e) {
            throw new RuntimeException("Failed to transform resource at '" + path + "'to string", e);
        }
    }

    public static boolean resourceExists(String path) {
        return ResourceUtils.class.getResource(path) != null;
    }

    public static void copyResource(String resourcePath, File outputFile) throws IOException {
        copyResource(resourcePath, outputFile, null, null);
    }

    public static void copyResource(String resourcePath, File outputFile, Class clazz) throws IOException {
        copyResource(resourcePath, outputFile, null, clazz);
    }

    public static void copyResource(String resourcePath, File outputFile, InputStreamManipulator manipulator,
            Class clazz) throws IOException {
        OutputStream os = null;
        try {
            os = new BufferedOutputStream(FileUtils.openOutputStream(outputFile));
            copyResource(resourcePath, os, manipulator, clazz);
        } finally {
            IOUtils.closeQuietly(os);
        }
    }

    public static void copyResource(String resourcePath, OutputStream outputStream, InputStreamManipulator manipulator,
            Class clazz) throws IOException {
        InputStream origInputStream = null;
        InputStream usedInputStream = null;
        try {
            origInputStream = clazz != null ?
                    clazz.getResourceAsStream(resourcePath) :
                    Thread.currentThread().getContextClassLoader().getResourceAsStream(resourcePath);
            assertResourceNotNull(resourcePath, origInputStream);
            if (manipulator != null) {
                InputStream mip = manipulator.manipulate(origInputStream);
                if (mip == null) {
                    throw new RuntimeException("Received a null stream from stream manipulation");
                }
                usedInputStream = mip;
            } else {
                usedInputStream = origInputStream;
            }
            IOUtils.copy(usedInputStream, outputStream);
        } finally {
            IOUtils.closeQuietly(outputStream);
            IOUtils.closeQuietly(usedInputStream);
            IOUtils.closeQuietly(origInputStream);
        }
    }

    public interface InputStreamManipulator {
        InputStream manipulate(InputStream origStream) throws IOException;
    }

    private static void assertResourceNotNull(String resourcePath, Object resourceHandle) {
        if (resourceHandle == null) {
            throw new IllegalArgumentException("Could not find the classpath resource at: " + resourcePath + ".");
        }
    }

}