package org.jfrog.common;


import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import com.github.fge.jsonpatch.JsonPatchException;
import com.github.fge.jsonpatch.mergepatch.JsonMergePatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;

/**
 * JSON related utilities
 *
 * @author Yinon Avraham.
 */
public class JsonUtils extends MapperUtilsBase {
    private static final Logger log = LoggerFactory.getLogger(JsonUtils.class);

    private static final JsonUtils instance = new JsonUtils(JsonInclude.Include.NON_NULL);
    private static final JsonUtils instanceWithAll = new JsonUtils(JsonInclude.Include.ALWAYS);


    protected JsonUtils(JsonInclude.Include serializationInclusion) {
        super(new ObjectMapper() {{
            setSerializationInclusion(serializationInclusion);
            configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            registerModule(new ParameterNamesModule());
        }});
    }

    protected JsonUtils(ObjectMapper objectMapper) { super(objectMapper); }

    public static synchronized JsonUtils getInstance() {
        return instance;
    }

    public static synchronized JsonUtils getInstanceSerializingNulls() {
        return instanceWithAll;
    }

    public static String toJsonMerge(Object entity, Set<String> fieldsToInclude) {
        JsonNode tree = JsonUtils.getInstanceSerializingNulls().valueToTree(entity);

        for (Iterator<Map.Entry<String, JsonNode>> i = tree.fields(); i.hasNext(); ) {
            Map.Entry entry = i.next();
            if (!fieldsToInclude.contains(entry.getKey())) {
                i.remove();
            }
        }
        return JsonUtils.getInstanceSerializingNulls().valueToString(tree);
    }

    public static synchronized JsonUtils createInstance(ObjectMapper objectMapper) {
        return new JsonUtils(objectMapper);
    }

    public static <I, T extends I> I jsonMerge(I originalModel, String patch, Class<T> modelType,
            @Nullable Consumer<JsonNode> validatePatchResult) {
        ObjectMapper mapper = new ObjectMapper()
                .enable(SerializationFeature.INDENT_OUTPUT);
        try {
            JsonNode entity = mapper.valueToTree(originalModel);
            JsonMergePatch patchTool = mapper.readValue(patch, JsonMergePatch.class);
            JsonNode modifiedEntity = patchTool.apply(entity);
            if (validatePatchResult != null) {
                validatePatchResult.accept(modifiedEntity);
            }
            return mapper.treeToValue(modifiedEntity, modelType);
        } catch (IOException | JsonPatchException e) {
            String message = "Failed to parse json patch content";
            log.trace(message, e);
            throw new JsonMergeException(message);
        }
    }
}
