package org.jfrog.common;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.*;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.Reader;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;

/**
 * @author Saffi Hartal 18-02-14
 * Implements a stream parsing with jackason fasterxml
 * Don't implement toString!!
 *
 * That a copy of the JsonParserStream - but adapted to faster xml jackason.
 * It is intended as a replacement for the JsonParserStream at the point of time when we upgrade from it to faster xml jackson
 *
 * Don't use it without proper testing and taking that into consideration.
 *
 * The SuppressWarnings should be removed once it replaces the JSonParserStream.
 */
@JsonDeserialize(using = JsonParserFastStream.Deserializer.class)
@SuppressWarnings({"WeakerAccess", "Duplicates"})
public abstract class JsonParserFastStream {
    private static final Logger log = LoggerFactory.getLogger(JsonParserFastStream.class);
    private final JsonParser jp;
    ArrayList<AbstractMap.SimpleEntry<String, Object>> fields = new ArrayList<>();

    private JsonParserFastStream(JsonParser jp) {
        this.jp = jp;
    }

    /**
     * parse with reader
     * one can later use convert to class.the JsoneNode
     *
     * @param reader Reader
     * @return JsonParserStream
     */
    public static JsonParserFastStream parse(Reader reader) throws IOException {
        return Helper.parse(reader).obj;
    }

    public abstract Stream<JsonNode> getArray(String name, Consumer<Exception> onFinish) throws IOException;

    public abstract Stream<JsonNode> getArray(Consumer<Exception> onFinish) throws IOException;

    public abstract Stream<Map.Entry<String, JsonNode>> getFields(Consumer<Exception> onFinish) throws IOException;

    Stream<JsonNode> parseArray(String name, Consumer<Exception> onFinish) throws IOException {

        Consumer<JsonParserFastStream> consumeFields = ((x) -> {
            while (jp.hasCurrentToken() && jp.getCurrentToken().equals(JsonToken.FIELD_NAME)) {
                try {
                    String currentName = jp.getCurrentName();

                    if (currentName.equals(name)) {
                        break;
                    }
                    jp.nextToken();

                    Object val = null;
                    if (jp.getCurrentToken().equals(JsonToken.VALUE_STRING)) {
                        val = jp.getText();
                    } else if (jp.getCurrentToken().equals(JsonToken.VALUE_NUMBER_INT)) {
                        val = jp.getIntValue();
                    } else if (jp.getCurrentToken().equals(JsonToken.VALUE_NUMBER_FLOAT)) {
                        val = jp.getFloatValue();
                    } else if (jp.getCurrentToken().equals(JsonToken.START_OBJECT)) {
                        val = jp.readValueAsTree();
                    } else if (jp.getCurrentToken().equals(JsonToken.VALUE_NULL)) {
                        val = null;
                    } else if (jp.getCurrentToken().equals(JsonToken.VALUE_TRUE)) {
                        val = true;
                    } else if (jp.getCurrentToken().equals(JsonToken.VALUE_FALSE)) {
                        val = false;
                    }
                    fields.add(
                            new AbstractMap.SimpleEntry<>(
                                    currentName, val));

                    jp.nextToken();

                } catch (IOException e) {
                    onFinish.accept(e);
                    break;
                }
            }
        });

        Consumer<JsonParserFastStream> arrayEnded = ((x) -> {
            consumeFields.accept(null);
            if (jp.getCurrentToken().equals(JsonToken.END_OBJECT)) {
                // wow!
                onFinish.accept(null);
                return;
            }
            log.trace("Done");
        });

        Function<JsonParserFastStream, Stream<JsonNode>> readObject = ((x) -> {
            try {
                jp.nextToken();
                consumeFields.accept(null);
                if (jp.getCurrentToken().equals(JsonToken.FIELD_NAME)) {
                    jp.nextToken();
                    if (jp.getCurrentToken().equals(JsonToken.START_ARRAY)) {
                        // ok - start
                        return parseArray(arrayEnded, onFinish);

                    } else {
                        String msg = "parse array required";
                        log.trace(msg);
                        onFinish.accept(new RuntimeException(msg));
                        return null;
                    }
                }
            } catch (JsonParseException e) {
                log.trace("parse ", e);
                onFinish.accept(e);
                return null;

            } catch (IOException e) {
                String msg = "Failed parseFields";

                if (log.isDebugEnabled()) {
                    log.debug(msg + ", fields read: {}", fields.toString());
                }
                onFinish.accept(e);
                return null;

            }
            onFinish.accept(new RuntimeException("unknown"));
            return null;
        });


        return readObject.apply(null);
    }

    Stream<JsonNode> parseArray(Consumer<Exception> onFinish) throws IOException {
        return parseArray((x) -> {
        }, onFinish);
    }

    Stream<JsonNode> parseArray(Consumer<JsonParserFastStream> afterResults, Consumer<Exception> onFinish) throws IOException {
        Function<Integer, JsonNode> generate = it -> {
            try {
                jp.nextToken();
                if (jp.getCurrentToken().equals(JsonToken.END_ARRAY)) {
                    // done
                    jp.nextToken();
                    afterResults.accept(null);
                    return null;
                }
                return jp.readValueAsTree();
            } catch (IOException ex) {
                if (log.isDebugEnabled()) {
                    log.debug("Failed parseFields, fields read: {}", fields.toString(), ex);
                }
                onFinish.accept(ex);
            }
            return null;
        };
        return StreamSupportUtils.generateTillNull(generate)
                .onClose(() ->
                        onFinish.accept(null));
    }


    Stream<Map.Entry<String, JsonNode>> parseFields(Consumer<Exception> onFinish) throws IOException {

        Function<JsonParserFastStream, Stream<Map.Entry<String, JsonNode>>> readResults = ((x) -> {
            try {
                jp.nextToken();
                if (jp.getCurrentToken().equals(JsonToken.FIELD_NAME)) {
                    Function<Integer, Map.Entry<String, JsonNode>> generate = it -> {
                        try {
                            if (jp.getCurrentToken().equals(JsonToken.END_OBJECT)) {
                                // done
                                jp.nextToken();
                                onFinish.accept(null);
                                return null;
                            }
                            final String currentName = jp.getCurrentName();
                            jp.nextToken();

                            Map.Entry<String, JsonNode> res = new AbstractMap.SimpleEntry<>(
                                    currentName,
                                    jp.readValueAsTree());
                            jp.nextToken();

                            return res;

                        } catch (IOException ex) {
                            if (log.isDebugEnabled()) {
                                log.debug("Failed parseFields, fields read: {}", fields.toString(), ex);
                            }
                            onFinish.accept(ex);
                        }
                        return null;
                    };
                    return StreamSupportUtils.generateTillNull(generate)
                            .onClose(() -> onFinish.accept(null));
                } else {
                    String msg = "Failed parseFields, not a field name";
                    if (log.isDebugEnabled()) {
                        log.debug(msg + ", fields read: {}", fields.toString());
                    }
                    onFinish.accept(new RuntimeException(msg));
                    return null;
                }
            } catch (IOException e) {
                String msg = "Failed parseFields";

                if (log.isDebugEnabled()) {
                    log.debug(msg + ", fields read: {}", fields.toString());
                }
                onFinish.accept(e);
                return null;
            }
        });

        Stream<Map.Entry<String, JsonNode>> result = readResults.apply(null);
        if (result == null) {
            return Stream.empty();
        }
        return result;
    }

    static class Deserializer extends JsonDeserializer<JsonParserFastStream> {
        @Override
        public JsonParserFastStream deserialize(JsonParser jp, DeserializationContext ctxt) throws IOException {
            return new JsonParserFastStream(jp) {
                @Override
                public Stream<JsonNode> getArray(String name, Consumer<Exception> onFinish) throws IOException {
                    return parseArray(name, onFinish)
                            .onClose(() -> IOUtils.closeQuietly(jp));
                }

                @Override
                public Stream<JsonNode> getArray(Consumer<Exception> onFinish) throws IOException {
                    return parseArray(onFinish)
                            .onClose(() -> IOUtils.closeQuietly(jp));
                }

                @Override
                public Stream<Map.Entry<String, JsonNode>> getFields(Consumer<Exception> onFinish) throws IOException {
                    return parseFields(onFinish)
                            .onClose(() -> IOUtils.closeQuietly(jp));
                }
            };
        }
    }

    public static class JsonContext {
        public final ObjectMapper mapper;
        public final JsonFactory factory;

        public JsonContext() {
            mapper = new ObjectMapper();
            factory = new JsonFactory(mapper);
        }

        public JsonParserFastStream parse(Reader reader) throws IOException {
            return factory.createParser(reader).readValueAs(JsonParserFastStream.class);
        }

        public <T> T toValue(JsonNode node, TypeReference<T> valueTypeRef) throws IOException {
            Class<T> cls = getClassFromTypeRef(valueTypeRef);
            return mapper.treeToValue(node, cls);
        }

        @SuppressWarnings("unused")
        public <T> JsonNode toJsonNode(T value) {
            return mapper.valueToTree(value);
        }

        @SuppressWarnings("unchecked")
        private <T> Class<T> getClassFromTypeRef(TypeReference<T> valueTypeRef) {
            JavaType type = mapper.getTypeFactory().constructType(valueTypeRef);
            return (Class<T>) type.getRawClass();
        }
    }

    /**
     * for converting streams to Typed object and holding mapper
     */
    public static class Helper {
        final JsonParserFastStream obj;
        final JsonContext context;

        Helper(JsonContext context, JsonParserFastStream obj) {
            this.context = context;
            this.obj = obj;
        }

        static Helper parse(Reader reader) throws IOException {
            JsonContext context = new JsonContext();
            return parse(context, reader);
        }

        private static Helper parse(JsonContext context, Reader reader) throws IOException {
            return new Helper(context, context.parse(reader));
        }

        public <T> Stream<T> getArray(String name, TypeReference<T> valueTypeRef, Consumer<Exception> onFinish) throws IOException {
            Stream<JsonNode> jsonNodeStream = obj.getArray(name, onFinish);
            return mapTo(jsonNodeStream, valueTypeRef, onFinish)
                    .onClose(jsonNodeStream::close);
        }

        @SuppressWarnings("unused")
        public <T> Stream<T> getArray(TypeReference<T> valueTypeRef, Consumer<Exception> onFinish) throws IOException {
            Stream<JsonNode> jsonNodeStream = obj.getArray(onFinish);
            return mapTo(jsonNodeStream, valueTypeRef, onFinish)
                    .onClose(jsonNodeStream::close);
        }

        private <T> Stream<T> mapTo(Stream<JsonNode> jsonNodeStream, TypeReference<T> valueTypeRef, Consumer<Exception> onFinish) {
            Class<T> cls = context.getClassFromTypeRef(valueTypeRef);
            return jsonNodeStream.map(it -> {
                try {
                    return context.mapper.treeToValue(it, cls);
                } catch (IOException e) {
                    onFinish.accept(e);
                    return null;
                }
            }).filter(Objects::nonNull);
        }

    }


}
