package org.jfrog.common;

import org.apache.commons.lang3.text.StrSubstitutor;
import org.codehaus.plexus.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.slf4j.event.Level;

import javax.annotation.concurrent.ThreadSafe;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;

/**
 * Configurable Retry Utility
 *
 * Support retying some code until success or "give up".
 * Configuration defined by RetryOptions, @see RetryOptions for details.
 *
 * @author Noam Shemesh
 */
@ThreadSafe
public abstract class ExecutionUtils {
    private static final Logger log = LoggerFactory.getLogger(ExecutionUtils.class);
    private static final Map<Level, Consumer<String>> logLevelTrace = Map.of(Level.DEBUG, log::debug
            , Level.INFO, log::info
            , Level.WARN, log::warn);

    private ExecutionUtils() {
    }

    /**
     *
     */
    public static <T> CompletableFuture<T> retry(Retryable<T> funcToRetry, RetryOptions retryOptions) {
        return retry(funcToRetry, retryOptions, Executors.newCachedThreadPool());
    }

    /**
     * Executing a function and retrying again up to <pre>retryOptions.numberOfRetries</pre> attempts.
     * A retry will happen only when a RetryException is thrown. Every other exception will be finish the execution.
     *
     * @param funcToRetry  The function to execute and retry if necessary
     * @param retryOptions Configure the retry attempts and other parameters
     * @param threadPool   Use this thread pool instead of {@link Executors#newCachedThreadPool()}
     * @param <T>          Type of the value to return from the executable function
     * @return Value of the successful execution
     */
    public static <T> CompletableFuture<T> retry(Retryable<T> funcToRetry, RetryOptions retryOptions,
            ExecutorService threadPool) {
        CompletableFuture<T> future = new CompletableFuture<>();
        Runnable task = generateExecutionRunnable(funcToRetry, retryOptions, threadPool, future, 0,
                System.currentTimeMillis());
        threadPool.submit(task);
        return future;
    }

    private static <T> Runnable generateExecutionRunnable(Retryable<T> funcToRetry, RetryOptions retryOptions,
            ExecutorService threadPool, CompletableFuture<T> future, int executedRetries, long startTime) {
        return addTraceId(() -> {
            try {
                handleFunctionExecution(funcToRetry, retryOptions, threadPool, future, executedRetries, startTime);
            } catch (Throwable e) {
                handleStopError(future, "code exception", retryOptions.getFailedLog(), e);
            }
        });
    }

    private static Runnable addTraceId(Runnable r) {
        String traceId = MDC.get("uber-trace-id");
        if (StringUtils.isNotBlank(traceId)) {
            return new MDCRunnableDecorator(getTraceIdFromHeaderStr(traceId), r);
        }
        return r;
    }

    public static String getTraceIdFromHeaderStr(String headerStr) {
        if (headerStr == null) { //just in case we do not have a headerStr, should not happen but we do not want to crash the system if it does
            log.debug("Got null headerStr for traceId, returning \"\"");
            return "";
        }

        String traceId = headerStr;
        if (headerStr.contains(":")) {
            traceId = headerStr.split(":")[0];
        }
        return traceId;
    }

    private static <T> void handleFunctionExecution(Retryable<T> funcToRetry, RetryOptions retryOptions,
            ExecutorService threadPool, CompletableFuture<T> future, int executedRetries, long startTime)
            throws InterruptedException {
        try {
            future.complete(funcToRetry.tryExecute());
        } catch (RetryException e) {
            if (executedRetries + 1 >= retryOptions.numberOfRetries) {
                handleStopError(future, "exceeded number of attempts (" +
                        retryOptions.numberOfRetries + ")", retryOptions.getFailedLog(), e);
            } else if (retryOptions.getShouldRetry().getAsBoolean()) {
                handleRetry(funcToRetry, retryOptions, threadPool, future, executedRetries, startTime, e);
            } else {
                log.debug("Retry predicate failed, stop retrying");
                future.cancel(false);
            }
        }
    }

    private static <T> void handleRetry(Retryable<T> funcToRetry, RetryOptions retryOptions, ExecutorService threadPool,
            CompletableFuture<T> future, int executedRetries, long startTime, RetryException e)
            throws InterruptedException {

        log(retryOptions, executedRetries, e, startTime);
        Thread.sleep(retryOptions.timeout);
        int timeout = retryOptions.timeout * retryOptions.exponentialBackoffMultiplier;
        if (retryOptions.backoffMaxDelay > 0 && retryOptions.backoffMaxDelay < timeout) {
            timeout = retryOptions.backoffMaxDelay;
        }

        RetryOptions newOptions = RetryOptions.fromOther(retryOptions)
                .timeout(timeout)
                .build();

        threadPool.submit(generateExecutionRunnable(funcToRetry, newOptions, threadPool, future, executedRetries + 1,
                startTime));
    }

    private static void log(RetryOptions retryOptions, int executedRetries, RetryException e,
            long startTime) {
        Level logLevel = shouldLog(retryOptions, executedRetries, startTime) ? retryOptions.logLevel : Level.DEBUG;
        String elapsedTime = TimeUnitFormat
                .getTimeString(System.currentTimeMillis() - startTime, TimeUnit.MILLISECONDS);
        String formattedLog = StrSubstitutor.replace(retryOptions.getRetryLog(), Map.of("RETRY", executedRetries + 1
                , "ELAPSED_TIME", elapsedTime
                , "ERROR", e.getMessage()));

        logLevelTrace.get(logLevel).accept(formattedLog);
    }

    private static boolean shouldLog(RetryOptions retryOptions, int executedRetries, long startTime) {
        return System.currentTimeMillis() > (startTime + retryOptions.getReportDelayMillis())
                && ((executedRetries + 1) % retryOptions.getBulkExceptionReportSize()) == 0;
    }

    private static <T> void handleStopError(CompletableFuture<T> future, String reasonToFinish, String logTrace, Throwable exception) {
        String message = StrSubstitutor
                .replace(logTrace, Map.of("ERROR", exception.getMessage(), "FINISH_REASON", reasonToFinish));
        log.debug(message, exception);
        log.error(message);
        future.completeExceptionally(new ExecutionFailed(message, exception));
    }

    static class MDCRunnableDecorator implements Runnable {
        private String traceId;
        private Runnable runnable;

        public MDCRunnableDecorator(String traceId, Runnable runnable) {
            this.traceId = traceId;
            this.runnable = runnable;
        }

        @Override
        public void run() {
            MDC.put("uber-trace-id", traceId);
            log.trace("starting a thread with traceId: {}", traceId);
            runnable.run();
        }
    }

    /**
     * Configure how to submit code action.
     * Details in comments in code.
     */
    public static class RetryOptions {
        private final int numberOfRetries;
        private final int exponentialBackoffMultiplier;
        private final int timeout;
        private final int backoffMaxDelay;
        private final int bulkExceptionReportSize;
        private final long reportDelayMillis;
        private final Level logLevel;
        private final String retryLog;
        private final String failedLog;
        private final BooleanSupplier shouldRetry;


        private RetryOptions(int numberOfRetries, int exponentialBackoffMultiplier, int timeout, int backoffMaxDelay,
                int bulkExceptionReportSize, long reportDelayMillis, Level logLevel, String retryLog,
                String failedLog, BooleanSupplier shouldRetry) {
            this.numberOfRetries = numberOfRetries;
            this.exponentialBackoffMultiplier = exponentialBackoffMultiplier;
            this.timeout = timeout;
            this.backoffMaxDelay = backoffMaxDelay;
            this.bulkExceptionReportSize = bulkExceptionReportSize;
            this.reportDelayMillis = reportDelayMillis;
            this.logLevel = logLevel;
            this.retryLog = retryLog;
            this.failedLog = failedLog;
            this.shouldRetry = shouldRetry;
        }

        public static Builder builder() {
            return new Builder();
        }

        public static Builder fromOther(RetryOptions retryOptions) {
            return new Builder()
                    .numberOfRetries(retryOptions.numberOfRetries)
                    .exponentialBackoffMultiplier(retryOptions.exponentialBackoffMultiplier)
                    .timeout(retryOptions.timeout)
                    .backoffMaxDelay(retryOptions.backoffMaxDelay)
                    .bulkExceptionReportSize(retryOptions.bulkExceptionReportSize)
                    .reportDelayMillis(retryOptions.getReportDelayMillis())
                    .logLevel(retryOptions.logLevel)
                    .retryLog(retryOptions.retryLog)
                    .failedLog(retryOptions.failedLog)
                    .shouldRetry(retryOptions.shouldRetry);
        }

        public int getNumberOfRetries() {
            return numberOfRetries;
        }

        public int getExponentialBackoffMultiplier() {
            return exponentialBackoffMultiplier;
        }

        public int getTimeout() {
            return timeout;
        }

        public int getBackoffMaxDelay() {
            return backoffMaxDelay;
        }

        public int getBulkExceptionReportSize() {
            return bulkExceptionReportSize;
        }

        public long getReportDelayMillis() {
            return reportDelayMillis;
        }

        public Level getLogLevel() {
            return logLevel;
        }

        public String getRetryLog() {
            return retryLog;
        }

        public String getFailedLog() {
            return failedLog;
        }

        public BooleanSupplier getShouldRetry() {
            return shouldRetry;
        }

        public static class Builder {
            private int numberOfRetries = 5;
            private int exponentialBackoffMultiplier = 2;
            private int timeout = 100;
            private int backoffMaxDelay = 1000;
            private int bulkExceptionReportSize = 1; //keep old action of reporting all exceptions (retries)
            private long reportDelayMillis;
            private Level logLevel = Level.WARN;
            private String retryLog = "Retry ${RETRY} Elapsed ${ELAPSED_TIME} failed: ${ERROR}. Trying again";
            private String failedLog = "Last retry failed: ${ERROR}. Not trying again (${FINISH_REASON})";
            private BooleanSupplier shouldRetry = () -> true;


            /**
             * Delay (milliseconds) between retries
             */
            public Builder timeout(int timeout) {
                this.timeout = ArgUtils.requireNonNegative(timeout, "timeout must be non negative");
                return this;
            }

            /**
             * Maximum attempts
             */
            public Builder numberOfRetries(int numberOfRetries) {
                this.numberOfRetries = ArgUtils
                        .requirePositive(numberOfRetries, "numberOfRetries must be non negative");
                return this;
            }

            /**
             * Increase delay time between attempts (factor)
             */
            public Builder exponentialBackoffMultiplier(int exponentialBackoffMultiplier) {
                this.exponentialBackoffMultiplier = ArgUtils
                        .requirePositive(exponentialBackoffMultiplier, "exponentialBackoffMultiplier must be positive");
                return this;
            }

            /**
             * if positive, maximum delay between retries. (Has preference over timeout)
             */
            public Builder backoffMaxDelay(int backoffMaxDelay) {
                this.backoffMaxDelay = ArgUtils
                        .requireNonNegative(backoffMaxDelay, "backoffMaxDelay must be non negative");
                return this;
            }

            /**
             * limit logging retries to once per bulkExceptionReportSize
             */
            public Builder bulkExceptionReportSize(int bulkExceptionReportSize) {
                this.bulkExceptionReportSize = ArgUtils
                        .requireNonNegative(bulkExceptionReportSize, "backoffMaxDelay must be non negative");
                return this;
            }

            /**
             * avoid logging retries until after reportDelayMillis
             */
            public Builder reportDelayMillis(long reportDelayMillis) {
                this.reportDelayMillis = ArgUtils
                        .requireNonNegative(reportDelayMillis, "reportDelayMillis must be non negative");
                return this;
            }

            public Builder logLevel(Level logLevel) {
                this.logLevel = logLevel;
                return this;
            }

            public Builder retryLog(String retryLog) {
                this.retryLog = retryLog;
                return this;
            }

            public Builder failedLog(String failedLog) {
                this.failedLog = failedLog;
                return this;
            }

            public Builder shouldRetry(BooleanSupplier shouldRetry) {
                this.shouldRetry = shouldRetry;
                return this;
            }

            public RetryOptions build() {
                return new RetryOptions(this.numberOfRetries, this.exponentialBackoffMultiplier, this.timeout,
                        this.backoffMaxDelay, this.bulkExceptionReportSize, this.reportDelayMillis, this.logLevel,
                        this.retryLog, this.failedLog, this.shouldRetry);
            }
        }
    }
}
