package org.jfrog.common;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.util.concurrent.Callable;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @author Noam Shemesh
 * @author Dan Feldman
 */
public abstract class ExceptionUtils {
    private static final Logger log = LoggerFactory.getLogger(ExceptionUtils.class);

    private ExceptionUtils() {}

    public static <T> T wrapException(Callable<T> callable) {
        return wrapException(callable, (String) null);
    }

    public static <T, X extends RuntimeException> T wrapException(Callable<T> callable, Class<X> wrapWith) {
        return wrapException(callable, wrapWith, null, null);
    }

    public static <T, X extends RuntimeException> T wrapException(Callable<T> callable, Class<X> wrapWith, @Nullable String err, @Nullable String context) {
        RuntimeException wrapper;
        try {
            return wrapException(callable, context);
        } catch (RuntimeException e) {
            try {
                if (isNotBlank(err)) {
                    wrapper = wrapWith.getDeclaredConstructor(String.class, Throwable.class).newInstance(err, e.getCause());
                } else {
                    wrapper = wrapWith.getDeclaredConstructor(Throwable.class).newInstance(e.getCause());
                }
            } catch (Exception oops) {
                log.debug("Can't use exception {} to wrap {}: ", wrapWith.getSimpleName(), e.getClass().getSimpleName());
                log.debug("", oops);
                wrapper = e;
            }
            throw wrapper;
        }
    }

    public static <T> T wrapException(Callable<T> callable, String context) {
        try {
            return callable.call();
        } catch (Exception e) {
            throw new RuntimeException((isEmpty(context) ? "" : "[" + context + "]: ") + e.getMessage(), e);
        }
    }
}
