/*
 *
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2018 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.common;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.Query;
import java.lang.management.ManagementFactory;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * A helper class that attempts to detect the http port Artifactory is listening on.
 *
 * @author Yossi Shaul
 */
public class TomcatUtils {

    private static final Logger log = LoggerFactory.getLogger(TomcatUtils.class);

    private TomcatUtils() {
    }

    /**
     * Search for a connector in the following order:
     * By scheme and proffered port.
     * By scheme.
     * First in the connectors list.
     * If none was found, throws {@link IllegalStateException}
     * @param scheme - connector scheme
     * @param preferredPort - connector preferredPort
     * @return connector details
     */
    public static ConnectorDetails getConnector(String scheme, int preferredPort) {

        Set<TomcatUtils.ConnectorDetails> connectors = TomcatUtils.getHttpConnectors();
        if (connectors.isEmpty()) {
            throw new IllegalStateException("Could not detect listening port.");
        }
        // prefer the http connector by scheme and the preferredPort
        Optional<ConnectorDetails> httpConnector = connectors.stream()
                .filter(c -> c.getScheme().equalsIgnoreCase(scheme))
                .filter(c -> c.getPort() == preferredPort)
                .findFirst();

        if (!httpConnector.isPresent()) {
            log.trace(
                    "HTTP connector with preferred scheme {} and port {} not found, looking for first connector only by preferred scheme.",
                    scheme, preferredPort);
            httpConnector = connectors.stream()
                    .filter(c -> c.getScheme().equalsIgnoreCase(scheme)).findFirst();
        }

        return httpConnector.orElseGet(() -> {
            log.trace("HTTP connector with preferred scheme {} not found, looking for first connector.", scheme);
            return connectors.iterator().next();
        });
    }

    /**
     * @return Tomcat HTTP/1.1 connectors details or empty set is no such connector found.
     */
    static Set<ConnectorDetails> getHttpConnectors() {
        try {
            // tomcat connectors detector - tomcat publishes the connectors under Tomcat/Connector/PORT(S)
            // the detector will fetch from the first HTTP/1.1 collector
            MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
            Set<ObjectName> connectors = mbs.queryNames(new ObjectName("*:type=Connector,*"),
                    Query.match(Query.attr("protocol"), Query.value("HTTP/1.1")));
            return connectors.stream()
                    .map(c -> new ConnectorDetails(getSchemaAttribute(mbs, c), c.getKeyProperty("port")))
                    .collect(Collectors.toSet());
        } catch (Exception e) {
            log.error("Failed to detect Tomcat connector: {}", e.getMessage());
            log.debug("Failed to detect Tomcat connector", e);
            throw new RuntimeException(e);
        }
    }

    private static String getSchemaAttribute(MBeanServer mbs, ObjectName c) {
        try {
            return mbs.getAttribute(c, "scheme").toString();
        } catch (Exception e) {
            throw new RuntimeException("Failed to parse scheme attribute of connector: " + c);
        }
    }

    public static class ConnectorDetails {
        private String scheme;
        private int port;

        ConnectorDetails(String scheme, String port) {
            this.scheme = StringUtils.isNotBlank(scheme) ? scheme : "http";
            this.port = Integer.parseInt(port);
        }

        /**
         * @param host        - url host
         * @param contextPath - url context path
         * @return url built fro, the connector scheme and port and the provided host and path
         */
        public String buildUrl(String host, String contextPath) {
            StringBuilder urlBuilder = new StringBuilder(getScheme())
                    .append("://")
                    .append(host)
                    .append(":")
                    .append(getPort());

            if (!contextPath.startsWith("/")) {
                urlBuilder.append("/");
            }
            return urlBuilder.append(contextPath).toString();
        }

        /**
         * @return The scheme this connector is configured to use (http or https)
         */
        public String getScheme() {
            return scheme;
        }

        /**
         * @return The port this connector is configured to use
         */
        public int getPort() {
            return port;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ConnectorDetails that = (ConnectorDetails) o;
            return port == that.port && Objects.equals(scheme, that.scheme);
        }

        @Override
        public int hashCode() {
            return Objects.hash(scheme, port);
        }

        @Override
        public String toString() {
            return "ConnectorDetails{scheme='" + scheme + '\'' + ", port=" + port + '}';
        }
    }
}