package org.jfrog.common;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.helpers.MessageFormatter;

import javax.annotation.concurrent.ThreadSafe;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

/**
 * Configurable Retry Utility
 *
 * Support retying some code until success or "give up".
 * Configuration defined by RetryOptions, @see RetryOptions for details.
 *
 * @author Noam Shemesh
 */
@ThreadSafe
public abstract class ExecutionUtils {
    private static final Logger log = LoggerFactory.getLogger(ExecutionUtils.class);

    private ExecutionUtils() {}

    /**
     *
     */
    public static <T> CompletableFuture<T> retry(Retryable<T> funcToRetry, RetryOptions retryOptions) {
        return retry(funcToRetry, retryOptions, Executors.newCachedThreadPool());
    }

    /**
     * Executing a function and retrying again up to <pre>retryOptions.numberOfRetries</pre> attempts.
     * A retry will happen only when a RetryException is thrown. Every other exception will be finish the execution.
     *
     * @param funcToRetry The function to execute and retry if necessary
     * @param retryOptions Configure the retry attempts and other parameters
     * @param threadPool Use this thread pool instead of {@link Executors#newCachedThreadPool()}
     * @param <T> Type of the value to return from the executable function
     * @return Value of the successful execution
     */
    public static <T> CompletableFuture<T> retry(Retryable<T> funcToRetry, RetryOptions retryOptions, ExecutorService threadPool) {
        CompletableFuture<T> future = new CompletableFuture<>();
        Runnable task = generateExecutionRunnable(funcToRetry, retryOptions, threadPool, future, 0,
                System.currentTimeMillis());
        threadPool.submit(task);
        return future;
    }

    private static <T> Runnable generateExecutionRunnable(Retryable<T> funcToRetry, RetryOptions retryOptions,
            ExecutorService threadPool, CompletableFuture<T> future, int executedRetries, long startTime) {
        return () -> {
            try {
                handleFunctionExecution(funcToRetry, retryOptions, threadPool, future, executedRetries, startTime);
            } catch (Throwable e) {
                handleStopError(future, "code exception", e.getMessage(), e);
            }
        };
    }

    private static <T> void handleFunctionExecution(Retryable<T> funcToRetry, RetryOptions retryOptions,
            ExecutorService threadPool, CompletableFuture<T> future, int executedRetries, long startTime)
            throws InterruptedException {
        try {
            future.complete(funcToRetry.tryExecute());
        } catch (RetryException e) {
            if (executedRetries + 1 >= retryOptions.numberOfRetries) {
                handleStopError(future, "exceeded number of attempts (" +
                        retryOptions.numberOfRetries + ")", e.getMessage(), e);
            } else {
                handleRetry(funcToRetry, retryOptions, threadPool, future, executedRetries, startTime, e);
            }
        }
    }

    private static <T> void handleRetry(Retryable<T> funcToRetry, RetryOptions retryOptions, ExecutorService threadPool,
            CompletableFuture<T> future, int executedRetries, long startTime, RetryException e)
            throws InterruptedException {
        if (shouldLog(retryOptions, executedRetries, startTime)) {
            String elapsedTime = TimeUnitFormat
                    .getTimeString(System.currentTimeMillis() - startTime, TimeUnit.MILLISECONDS);
            log.warn("Retry {} Elapsed {} failed: {}. Trying again", executedRetries + 1, elapsedTime, e.getMessage());
        }

        Thread.sleep(retryOptions.timeout);

        int timeout = retryOptions.timeout * retryOptions.exponentialBackoffMultiplier;
        if (retryOptions.backoffMaxDelay > 0 && retryOptions.backoffMaxDelay < timeout) {
            timeout = retryOptions.backoffMaxDelay;
        }

        RetryOptions newOptions = RetryOptions.fromOther(retryOptions)
                .timeout(timeout)
                .build();

        threadPool.submit(generateExecutionRunnable(funcToRetry, newOptions, threadPool, future, executedRetries + 1,
                startTime));
    }

    private static boolean shouldLog(RetryOptions retryOptions, int executedRetries, long startTime) {
        return  System.currentTimeMillis() > (startTime + retryOptions.getReportDelayMillis())
                && ((executedRetries + 1) % retryOptions.getBulkExceptionReportSize()) == 0;
    }

    private static <T> void handleStopError(CompletableFuture<T> future, String exceptionMessage, String reasonToFinish, Throwable exception) {
        String message =
                MessageFormatter.format("Last retry failed: {}. Not trying again ({})", exceptionMessage, reasonToFinish).getMessage();
        log.debug(message, exception);
        log.error(message);
        future.completeExceptionally(new ExecutionFailed(message, exception));
    }


    /**
     * Configure how to submit code action.
     * Details in comments in code.
     */
    public static class RetryOptions {
        private final int numberOfRetries;
        private final int exponentialBackoffMultiplier;
        private final int timeout;
        private final int backoffMaxDelay;
        private final int bulkExceptionReportSize;
        private final long reportDelayMillis;

        private RetryOptions(int numberOfRetries, int exponentialBackoffMultiplier, int timeout, int backoffMaxDelay,
                int bulkExceptionReportSize, long reportDelayMillis) {
            this.numberOfRetries = numberOfRetries;
            this.exponentialBackoffMultiplier = exponentialBackoffMultiplier;
            this.timeout = timeout;
            this.backoffMaxDelay = backoffMaxDelay;
            this.bulkExceptionReportSize = bulkExceptionReportSize;
            this.reportDelayMillis = reportDelayMillis;
        }

        public static Builder builder() {
            return new Builder();
        }

        public static Builder fromOther(RetryOptions retryOptions) {
            return new Builder()
                    .numberOfRetries(retryOptions.numberOfRetries)
                    .exponentialBackoffMultiplier(retryOptions.exponentialBackoffMultiplier)
                    .timeout(retryOptions.timeout)
                    .backoffMaxDelay(retryOptions.backoffMaxDelay)
                    .bulkExceptionReportSize(retryOptions.bulkExceptionReportSize)
                    .reportDelayMillis(retryOptions.getReportDelayMillis());
        }

        public int getNumberOfRetries() {
            return numberOfRetries;
        }

        public int getExponentialBackoffMultiplier() {
            return exponentialBackoffMultiplier;
        }

        public int getTimeout() {
            return timeout;
        }

        public int getBackoffMaxDelay() {
            return backoffMaxDelay;
        }

        public int getBulkExceptionReportSize() { return bulkExceptionReportSize; }

        public long getReportDelayMillis() {
            return reportDelayMillis;
        }

        public static class Builder {
            private int numberOfRetries = 5;
            private int exponentialBackoffMultiplier = 2;
            private int timeout = 100;
            private int backoffMaxDelay = 1000;
            private int bulkExceptionReportSize = 1; //keep old action of reporting all exceptions (retries)
            private long reportDelayMillis;

            /** Delay (milliseconds) between retries */
            public Builder timeout(int timeout) {
                this.timeout = ArgUtils.requireNonNegative(timeout, "timeout must be non negative");
                return this;
            }

            /** Maximum attempts */
            public Builder numberOfRetries(int numberOfRetries) {
                this.numberOfRetries = ArgUtils.requirePositive(numberOfRetries, "numberOfRetries must be non negative");
                return this;
            }

            /** Increase delay time between attempts (factor) */
            public Builder exponentialBackoffMultiplier(int exponentialBackoffMultiplier) {
                this.exponentialBackoffMultiplier = ArgUtils.requirePositive(exponentialBackoffMultiplier, "exponentialBackoffMultiplier must be positive");
                return this;
            }

            /** if positive, maximum delay between retries. (Has preference over timeout) */
            public Builder backoffMaxDelay(int backoffMaxDelay) {
                this.backoffMaxDelay = ArgUtils.requireNonNegative(backoffMaxDelay, "backoffMaxDelay must be non negative");
                return this;
            }

            /** limit logging retries to once per bulkExceptionReportSize */
            public Builder bulkExceptionReportSize(int bulkExceptionReportSize) {
                this.bulkExceptionReportSize = ArgUtils.requireNonNegative(bulkExceptionReportSize, "backoffMaxDelay must be non negative");
                return this;
            }

            /** avoid logging retries until after reportDelayMillis */
            public Builder reportDelayMillis(long reportDelayMillis) {
                this.reportDelayMillis = ArgUtils.requireNonNegative(reportDelayMillis, "reportDelayMillis must be non negative");
                return this;
            }

            public RetryOptions build() {
                return new RetryOptions(this.numberOfRetries, this.exponentialBackoffMultiplier, this.timeout,
                        this.backoffMaxDelay, this.bulkExceptionReportSize, this.reportDelayMillis);
            }
        }
    }
}
