package org.jfrog.client.util;

import lombok.extern.slf4j.Slf4j;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;


/**
 * The logic of NoProxyHostsEvaluator and its tests is based on the code in:
 * https://github.com/jenkinsci/remoting/blob/master/src/main/java/hudson/remoting/Util.java
 */

@Slf4j
public class NoProxyHostsEvaluator {

    private final String noProxyHosts;

    public NoProxyHostsEvaluator(@Nullable String noProxyHosts) {
        this.noProxyHosts = getCleanedNoProxyHostsList(noProxyHosts);
    }

    /**
     * @return String that contains list of hosts to bypass the proxy, separated by comma,
     * examples can be found here: https://www.w3.org/Daemon/User/Proxies/ProxyClients.html
     */
    public String getNoProxyHosts() { return noProxyHosts; }

    /**
     * Check if given URL is in the exclusion list defined by the no_proxy environment variable.
     * On most *NIX system wildcards are not supported but if one top domain is added, all related subdomains will also
     * be ignored. Both "mit.edu" and ".mit.edu" are valid syntax, as well as "*.mit.edu".
     * We also allow adding optional [:port] to the end of the host pattern as shown
     * https://www.w3.org/Daemon/User/Proxies/ProxyClients.html
     *
     * Regexp:
     * - \Q and \E: https://docs.oracle.com/javase/7/docs/api/java/util/regex/Pattern.html
     * - To match IPV4/IPV/FQDN: Regular Expressions Cookbook, 2nd Edition (ISBN: 9781449327453)
     *
     * Warning: this method won't match shortened representation of IPV6 address
     */
    public boolean shouldBypassProxy(@Nonnull String host) {
        String noProxyHostsList = noProxyHosts;
        if (!noProxyHostsList.isEmpty()) {
            // IPV4 and IPV6
            if (host.matches("^(?:[0-9]{1,3}\\.){3}[0-9]{1,3}(:[0-9]{1,5})??$")
                    || host.matches("^(?:[a-fA-F0-9]{1,4}:){7}[a-fA-F0-9]{1,4}(:[0-9]{1,5})??$")) {
                return match(noProxyHostsList, host);
            } else {
                return isHostAndPortMatch(noProxyHostsList, host);
            }
        }
        return false;
    }

    private boolean isHostAndPortMatch(@Nonnull String noProxyHostsList, @Nonnull String host) {
        int depth = 0;
        String originalHost = host;
        // Loop while we have a valid domain name: acme.com
        // We add a safeguard to avoid a case where the host would always be valid because the regex would
        // for example fail to remove subdomains.
        // According to Wikipedia (no RFC defines it), 128 is the max number of subdivision for a valid FQDN:
        // https://en.wikipedia.org/wiki/Subdomain#Overview
        while (host.matches("^([a-z0-9]+(-[a-z0-9]+)*\\.)+[a-z]{2,}(:[0-9]{1,5})??$")
                && depth < 128) {
            ++depth;
            if (match(noProxyHostsList, host)){
                return true;
            } else {
                // Remove first subdomain: sub1.sub2.acme.com -> sub2.acme.com
                host = host.replaceFirst("^[a-z0-9]+(-[a-z0-9]+)*\\.", "");
            }
        }

        String[] noProxyArray = noProxyHostsList.split(",");
        return depth > 0 && suffixMatch(originalHost, noProxyArray);
    }

    private boolean match(@Nonnull String noProxyHostsList, @Nonnull String host) {
        // Check if the no_proxy list contains the host
        if (noProxyHostsList.matches(".*(^|,)\\Q" + host + "\\E($|,).*"))
            return true;

        // Check if host comes with port, but host on no_proxy list without port
        int portSplitIndex = host.indexOf(':');
        return portSplitIndex > -1 && noProxyHostsList.contains(host.substring(0, portSplitIndex))
                && !noProxyHostsList.contains(host.substring(0, portSplitIndex + 1));
    }

    private String getCleanedNoProxyHostsList(@Nullable String noProxyHostsList) {
        if (noProxyHostsList == null){
            return "";
        }
        String updatedString = noProxyHostsList.trim()
                // Remove spaces
                .replaceAll("\\s+", "")
                // Remove asterisks followed by dot
                .replaceAll("\\*\\.", "")
                // Convert .foobar.com to foobar.com after first char
                .replaceAll(",\\.", ",");
        if (updatedString.startsWith(".")) {
            // Convert .foobar.com to foobar.com in the first char
            return updatedString.substring(1);
        }
        return updatedString;
    }

    private boolean suffixMatch(String host, String[] noProxyArray) {
        for (String proxy : noProxyArray) {
            if (!proxy.contains("."))
                continue;
            if (host.endsWith(proxy))
                return true;
        }
        return false;
    }
}
