/*
 *
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2016 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.client.http;

import org.apache.commons.lang.StringUtils;
import org.apache.http.HeaderElement;
import org.apache.http.HeaderElementIterator;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.HttpResponseInterceptor;
import org.apache.http.auth.AuthSchemeProvider;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.NTCredentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.RedirectStrategy;
import org.apache.http.client.config.AuthSchemes;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.ConnectionKeepAliveStrategy;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.DefaultHostnameVerifier;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.auth.SPNegoSchemeFactory;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicHeaderElementIterator;
import org.apache.http.protocol.HTTP;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.ssl.SSLContexts;
import org.jfrog.client.http.auth.ProxyPreemptiveAuthInterceptor;
import org.jfrog.client.http.model.ProxyConfig;
import org.jfrog.client.util.KeyStoreProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.SSLContext;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.UnknownHostException;
import java.security.KeyStore;
import java.security.Principal;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * Base builder for HTTP client.
 *
 * @author Yossi Shaul
 */
public abstract class HttpBuilderBase<T extends HttpBuilderBase> {
    private static final Logger log = LoggerFactory.getLogger(HttpBuilderBase.class);

    protected HttpClientBuilder builder = HttpClients.custom();
    protected RequestConfig.Builder config = RequestConfig.custom();
    protected HttpHost defaultHost;
    protected BasicCredentialsProvider credsProvider;
    protected boolean allowAnyHostAuth;
    // Signifies what auth scheme will be used by the client
    protected JFrogAuthScheme chosenAuthScheme = JFrogAuthScheme.BASIC;
    protected boolean cookieSupportEnabled = false;
    protected HttpHost proxyHost;
    private KeyStoreProvider keyStoreProvider;
    private KeyStoreProvider clientCertKeyStoreProvider;
    private String clientCertAlias;
    private boolean trustSelfSignCert = false;
    protected SSLContextBuilder sslContextBuilder;
    private boolean noHostVerification = false;
    protected int maxConnectionsTotal = DEFAULT_MAX_CONNECTIONS;
    protected int maxConnectionsPerRoute = DEFAULT_MAX_CONNECTIONS;
    private int connectionPoolTimeToLive = CONNECTION_POOL_TIME_TO_LIVE;
    private RequestConfig defaultRequestConfig;
    protected String noProxyHosts;

    private static final String NO_PROXY_HOSTS = "NO_PROXY";

    public HttpBuilderBase() {
        credsProvider = new BasicCredentialsProvider();
        config.setMaxRedirects(20);
    }

    public CloseableHttpClient build() {
        return build(false);
    }

    public CloseableHttpClient build(boolean withParams) {
        String noProxyHostsList = StringUtils.isBlank(noProxyHosts) ?
                System.getenv(NO_PROXY_HOSTS) : noProxyHosts;
        builder.setRoutePlanner(new DefaultHostSpecificProxyRoutePlanner.Builder()
                .defaultHost(defaultHost).proxyHost(proxyHost).noProxyHosts(noProxyHostsList).build());
        PoolingHttpClientConnectionManager connectionMgr = configConnectionManager();
        if (withParams) {
            return new CloseableHttpClientWithParamsDecorator(builder.build(), connectionMgr,
                    chosenAuthScheme == JFrogAuthScheme.SPNEGO, defaultRequestConfig);
        }
        return new CloseableHttpClientDecorator(builder.build(), connectionMgr,
                chosenAuthScheme == JFrogAuthScheme.SPNEGO, defaultRequestConfig);
    }

    private T self() {
        return (T) this;
    }

    /**
     * Sets a set of domains for which the proxy should not be consulted;
     * the contents is a comma-separated list of domain names, with an optional :port part:
     *
     * NOTE: by default localhost will bypass the proxy
     */
    public T noProxyHosts(String noProxyHosts){
        this.noProxyHosts = noProxyHosts;
        return self();
    }

    /**
     * Sets the User-Agent value
     */
    public T userAgent(String userAgent) {
        builder.setUserAgent(userAgent);
        return self();
    }

    /**
     * Disables the automatic gzip compression on read. Once disabled cannot be activated.
     */
    public T disableGzipResponse() {
        builder.disableContentCompression();
        return self();
    }

    /**
     * Sets the host the client works with by default. Uses the default scheme (http) and port (80)
     *
     * @param host The host name
     */
    public T host(String host) {
        return host(host, 80);
    }

    /**
     * Sets the host and port the client works with by default. Uses https if port is 443.
     *
     * @param host The host name
     * @param port The server port
     */
    public T host(String host, int port) {
        String scheme = port != 443 ? "http" : "https";
        return host(host, port, scheme);
    }

    /**
     * Sets the host and port the client works with by default.
     *
     * @param host   The host name
     * @param port   The server port
     * @param scheme The http scheme (http or https)
     */
    public T host(String host, int port, String scheme) {
        if (StringUtils.isNotBlank(host)) {
            defaultHost = new HttpHost(host, port, scheme);
        } else {
            defaultHost = null;
        }
        return self();
    }

    /**
     * Sets the host the client works with by default. This method accepts any valid {@link URL} formatted string.
     * This will extract the schema, host and port to use by default.
     *
     * @throws IllegalArgumentException if the given URL is invalid
     */
    public T hostFromUrl(String urlStr) {
        if (StringUtils.isNotBlank(urlStr)) {
            try {
                URL url = new URL(urlStr);
                defaultHost = new HttpHost(url.getHost(), url.getPort(), url.getProtocol());
            } catch (MalformedURLException e) {
                throw new IllegalArgumentException("Cannot parse the url " + urlStr, e);
            }
        } else {
            defaultHost = null;
        }
        return self();
    }

    public T maxConnectionsPerRoute(int maxConnectionsPerHost) {
        this.maxConnectionsPerRoute = maxConnectionsPerHost;
        return self();
    }

    public T maxTotalConnections(int maxTotalConnections) {
        this.maxConnectionsTotal = maxTotalConnections;
        return self();
    }

    public T connectionTimeout(int connectionTimeout) {
        config.setConnectTimeout(connectionTimeout);
        return self();
    }

    public T socketTimeout(int soTimeout) {
        config.setSocketTimeout(soTimeout);
        return self();
    }

    /**
     * see {@link RequestConfig#isStaleConnectionCheckEnabled()}
     */
    public T staleCheckingEnabled(boolean staleCheckingEnabled) {
        config.setStaleConnectionCheckEnabled(staleCheckingEnabled);
        return self();
    }

    /**
     * Disable request retries on service unavailability.
     */
    public T noRetry() {
        return retry(0, false);
    }

    /**
     * Number of retry attempts. Default is 3 retries.
     *
     * @param retryCount Number of retry attempts. 0 means no retries.
     */
    public T retry(int retryCount) {
        return retry(retryCount, false);
    }

    /**
     * Number of retry attempts. Default is 3 retries.
     *
     * @param retryCount              Number of retry attempts. 0 means no retries.
     * @param requestSentRetryEnabled True if it's safe to to retry non-idempotent requests that have been sent
     */
    public T retry(int retryCount, boolean requestSentRetryEnabled) {
        if (retryCount == 0) {
            builder.disableAutomaticRetries();
        } else {
            builder.setRetryHandler(new DefaultHttpRequestRetryHandler(retryCount, requestSentRetryEnabled));
        }
        return self();
    }

    /**
     * Ignores blank or invalid input
     */
    public T localAddress(String localAddress) {
        if (StringUtils.isNotBlank(localAddress)) {
            try {
                InetAddress address = InetAddress.getByName(localAddress);
                config.setLocalAddress(address);
            } catch (UnknownHostException e) {
                throw new IllegalArgumentException("Invalid local address: " + localAddress, e);
            }
        }
        return self();
    }

    /**
     * How long to keep connections alive for reuse purposes before ditching them
     *
     * @param seconds Time to live in seconds
     */
    public T connectionPoolTTL(int seconds) {
        this.connectionPoolTimeToLive = seconds;
        return self();
    }

    /**
     * Enable cookie management for this client.
     */
    public T enableCookieManagement(boolean enableCookieManagement) {
        if (enableCookieManagement) {
            config.setCookieSpec(CookieSpecs.BROWSER_COMPATIBILITY);
        } else {
            config.setCookieSpec(null);
        }
        this.cookieSupportEnabled = enableCookieManagement;
        return self();
    }

    /**
     * The KeyStore provider for SSL trust
     *
     * @param keyStoreProvider the trust KeyStore provider
     * @return {@link T}
     */
    public T keyStoreProvider(KeyStoreProvider keyStoreProvider) {
        this.keyStoreProvider = keyStoreProvider;
        return self();
    }

    /**
     * The KeyStore provider for client SSL authentication
     *
     * @param keyStoreProvider the client certificate KeyStore provider
     */
    public T clientCertKeyStoreProvider(KeyStoreProvider keyStoreProvider) {
        this.clientCertKeyStoreProvider = keyStoreProvider;
        return self();
    }

    /**
     * The alias of the client certificate to use for client TLS authentication.
     *
     * @param clientCertAliasName The client certificate alias
     */
    public T clientCertAlias(String clientCertAliasName) {
        this.clientCertAlias = clientCertAliasName;
        return self();
    }

    /**
     * Set SPNEGO scheme for kerberos auth.
     *
     * @param useKerberos to activate kerberos executions
     * @return {@link T}
     */
    public T useKerberos(boolean useKerberos) {
        if (useKerberos) {
            Credentials use_jaas_creds = new Credentials() {
                @Override
                public String getPassword() {
                    return null;
                }

                @Override
                public Principal getUserPrincipal() {
                    return null;
                }
            };
            CredentialsProvider credsProvider1 = new BasicCredentialsProvider();
            credsProvider1.setCredentials(new AuthScope(null, -1, null), use_jaas_creds);
            Registry<AuthSchemeProvider> spnegoScheme = RegistryBuilder.<AuthSchemeProvider>create()
                    .register(AuthSchemes.SPNEGO, new SPNegoSchemeFactory(true))
                    .build();
            builder.setDefaultAuthSchemeRegistry(spnegoScheme).setDefaultCredentialsProvider(credsProvider1);
            chosenAuthScheme = JFrogAuthScheme.SPNEGO;
        }
        return self();
    }

    /**
     * @param trustSelfSignCert Trust self signed certificates on SSL handshake
     * @return {@link T}
     */
    public T trustSelfSignCert(boolean trustSelfSignCert) {
        this.trustSelfSignCert = trustSelfSignCert;
        return self();
    }

    /**
     * @param sslContextBuilder SSLContext builder
     * @return {@link T}
     */
    public T sslContextBuilder(SSLContextBuilder sslContextBuilder) {
        this.sslContextBuilder = sslContextBuilder;
        return self();
    }

    /**
     * @param noHostVerification whether host name verification against CA certificate is disabled on SSL handshake
     * @return {@link T}
     */
    public T noHostVerification(boolean noHostVerification) {
        this.noHostVerification = noHostVerification;
        return self();
    }

    /**
     * Ignores null credentials
     */
    public T authentication(UsernamePasswordCredentials creds) {
        if (creds != null) {
            authentication(creds.getUserName(), creds.getPassword());
        }
        return self();
    }

    /**
     * Configures preemptive authentication on this client. Ignores blank username input.
     */
    public T authentication(String username, String password) {
        return authentication(username, password, false);
    }

    /**
     * Configures preemptive authentication on this client. Ignores blank username input.
     */
    public T authentication(String username, String password, boolean allowAnyHost, List<String> additionalAuthScopes) {
        if (StringUtils.isNotBlank(username)) {
            if (defaultHost == null || StringUtils.isBlank(defaultHost.getHostName())) {
                throw new IllegalStateException("Cannot configure authentication when host is not set.");
            }
            this.allowAnyHostAuth = allowAnyHost;
            AuthScope authscope = allowAnyHost ?
                    new AuthScope(AuthScope.ANY_HOST, AuthScope.ANY_PORT, AuthScope.ANY_REALM) :
                    new AuthScope(defaultHost.getHostName(), AuthScope.ANY_PORT, AuthScope.ANY_REALM);
            credsProvider.setCredentials(authscope, new UsernamePasswordCredentials(username, password));
            if (additionalAuthScopes != null) {
                additionalAuthScopes.forEach(scope -> credsProvider
                        .setCredentials(new AuthScope(scope, AuthScope.ANY_PORT, AuthScope.ANY_REALM),
                                new UsernamePasswordCredentials(username, password)));
            }
        }
        return self();
    }

    /**
     * Configures preemptive authentication on this client. Ignores blank username input.
     */
    public T authentication(String username, String password, boolean allowAnyHost) {
        return authentication(username, password, allowAnyHost, null);
    }

    /**
     * Adds a {@link HttpRequestInterceptor} as the first interceptor.
     *
     * @param interceptor The request interceptor to add
     */
    public T addRequestInterceptor(HttpRequestInterceptor interceptor) {
        builder.addInterceptorFirst(interceptor);
        return self();
    }

    /**
     * Adds a {@link HttpRequestInterceptor} as the first interceptor.
     *
     * @param interceptor The response interceptor to add
     */
    public T addResponseInterceptor(HttpResponseInterceptor interceptor) {
        builder.addInterceptorLast(interceptor);
        return self();
    }

    /**
     * Configures a RedirectStrategy to replace the DefaultRedirectStrategy
     */
    public T redirectStrategy(RedirectStrategy redirectStrategy) {
        builder.setRedirectStrategy(redirectStrategy);
        return self();
    }

    /**
     * Configure whether the client should normalize paths also affect redirects
     */
    public T normalize(boolean isNormalize) {
        config.setNormalizeUri(isNormalize);
        return self();
    }

    public T proxy(ProxyConfig proxy) {
        if (proxy == null) {
            return self();
        }
        this.proxyHost = new HttpHost(proxy.getHost(), proxy.getPort());
        ProxyConfigBuilder proxyBuilder = proxy(proxy.getHost(), proxy.getPort());
        if (StringUtils.isNotBlank(proxy.getUsername())) {
            if (proxy.getDomain() == null) {
                proxyBuilder.authentication(proxy.getUsername(), proxy.getPassword());
            } else {
                try {
                    String ntHost = StringUtils.isBlank(proxy.getNtHost()) ? InetAddress.getLocalHost().getHostName() :
                            proxy.getNtHost();
                    proxyBuilder
                            .ntlmAuthentication(proxy.getUsername(), proxy.getPassword(), ntHost, proxy.getDomain());
                } catch (UnknownHostException e) {
                    log.error("Failed to determine required local hostname for NTLM credentials.", e);
                }
            }
            proxyBuilder.redirectToHostProxies(proxy.getRedirectedToHostsList());
        }
        return self();
    }

    public ProxyConfigBuilder proxy(String host, int port) {
        return new ProxyConfigBuilder(host, port);
    }

    public class ProxyConfigBuilder {
        private final String proxyHost;
        private final int proxyPort;
        Credentials creds;

        public ProxyConfigBuilder(String host, int port) {
            this.proxyHost = host;
            this.proxyPort = port;
        }

        public ProxyConfigBuilder authentication(String username, String password) {
            creds = new UsernamePasswordCredentials(username, password);
            //This will demote the NTLM authentication scheme so that the proxy won't barf
            //when we try to give it traditional credentials. If the proxy doesn't do NTLM
            //then this won't hurt it (jcej at tragus dot org)
            List<String> authPrefs = Arrays.asList(AuthSchemes.DIGEST, AuthSchemes.BASIC, AuthSchemes.NTLM);
            config.setProxyPreferredAuthSchemes(authPrefs);

            // preemptive proxy authentication
            builder.addInterceptorFirst(new ProxyPreemptiveAuthInterceptor());
            setProxyCreds(proxyHost, proxyPort);
            return this;
        }

        public ProxyConfigBuilder ntlmAuthentication(String username, String password, String ntHost, String domain) {
            creds = new NTCredentials(username, password, ntHost, domain);
            // preemptive proxy authentication
            builder.addInterceptorFirst(new ProxyPreemptiveAuthInterceptor());
            setProxyCreds(proxyHost, proxyPort);
            return this;
        }

        /**
         * List of hosts names that the proxy server might redirect with and should be trusted with the proxy creds.
         * Only applicable when proxy credentials are configured
         *
         * @param hosts List of host names to provide with proxy credentials.
         */
        public ProxyConfigBuilder redirectToHostProxies(String[] hosts) {
            if (creds != null && hosts != null) {
                for (String hostName : hosts) {
                    setProxyCreds(hostName, proxyPort);
                }
            }
            return this;
        }

        private void setProxyCreds(String host, int port) {
            if (StringUtils.isBlank(proxyHost) || port == 0) {
                throw new IllegalStateException("Proxy host and port must be set before creating authentication");
            }
            credsProvider.setCredentials(new AuthScope(host, port, AuthScope.ANY_REALM), creds);
        }
    }

    public boolean isCookieSupportEnabled() {
        return cookieSupportEnabled;
    }

    /**
     * Produces a {@link ConnectionKeepAliveStrategy}
     *
     * @return keep-alive strategy to be used for connection pool
     */
    public static ConnectionKeepAliveStrategy createConnectionKeepAliveStrategy() {
        return (response, context) -> {
            // Honor 'keep-alive' header
            HeaderElementIterator it = new BasicHeaderElementIterator(
                    response.headerIterator(HTTP.CONN_KEEP_ALIVE));
            while (it.hasNext()) {
                HeaderElement he = it.nextElement();
                String param = he.getName();
                String value = he.getValue();
                if (value != null && param.equalsIgnoreCase("timeout")) {
                    try {
                        return Long.parseLong(value) * 1000;
                    } catch (NumberFormatException ignore) {
                    }
                }
            }
            return 30 * 1000;
        };
    }

    public RequestConfig getDefaultRequestConfig() {
        return defaultRequestConfig;
    }

    public HttpHost getProxyHost() {
        return proxyHost;
    }

    protected PoolingHttpClientConnectionManager configConnectionManager() {
        if (!isCookieSupportEnabled()) {
            builder.disableCookieManagement();
        }
        if (hasCredentials()) {
            builder.setDefaultCredentialsProvider(credsProvider);
        }
        defaultRequestConfig = config.build();
        builder.setDefaultRequestConfig(defaultRequestConfig);

        /**
         * Connection management
         */
        builder.setKeepAliveStrategy(createConnectionKeepAliveStrategy());

        // TODO: [by fsi] Looks like the following methods are ignored
        // since the connection manager is doing this work
        builder.setMaxConnTotal(maxConnectionsTotal);
        builder.setMaxConnPerRoute(maxConnectionsPerRoute);

        PoolingHttpClientConnectionManager connectionMgr = createConnectionMgr();
        builder.setConnectionManager(connectionMgr);
        return connectionMgr;
    }

    /**
     * Creates custom Http Client connection pool to be used by Http Client
     *
     * @return {@link PoolingHttpClientConnectionManager}
     */
    private PoolingHttpClientConnectionManager createConnectionMgr() {
        PoolingHttpClientConnectionManager connectionMgr;

        // prepare SSLContext
        SSLContext sslContext = buildSslContext();
        ConnectionSocketFactory plainsf = PlainConnectionSocketFactory.getSocketFactory();
        // we allow to disable host name verification against CA certificate,
        // notice: in general this is insecure and should be avoided in production,
        // (this type of configuration is useful for development purposes)
        LayeredConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(
                sslContext,
                noHostVerification ? NoopHostnameVerifier.INSTANCE : new DefaultHostnameVerifier()
        );
        Registry<ConnectionSocketFactory> r = RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", plainsf)
                .register("https", sslsf)
                .build();
        connectionMgr = new PoolingHttpClientConnectionManager(r, null, null,
                null, connectionPoolTimeToLive, TimeUnit.SECONDS);

        connectionMgr.setMaxTotal(maxConnectionsTotal);
        connectionMgr.setDefaultMaxPerRoute(maxConnectionsPerRoute);
        HttpHost localhost = new HttpHost("localhost", 80);
        connectionMgr.setMaxPerRoute(new HttpRoute(localhost), maxConnectionsPerRoute);
        return connectionMgr;
    }

    private SSLContext buildSslContext() {
        SSLContext sslContext = null;
        try {
            SSLContextBuilder sslBuilder = sslContextBuilder;
            if (trustSelfSignCert) {
                // trust any self signed certificate
                sslBuilder = sslBuilder != null ? sslBuilder : SSLContexts.custom();
                sslBuilder.loadTrustMaterial(TrustSelfSignedMultiChainStrategy.INSTANCE);
            }
            if (keyStoreProvider != null) {
                KeyStore trustStore = keyStoreProvider.provide();
                sslBuilder = sslBuilder != null ? sslBuilder : SSLContexts.custom();
                sslBuilder.loadTrustMaterial(trustStore, null);
            }
            if (clientCertKeyStoreProvider != null && StringUtils.isNotBlank(clientCertAlias)) {
                sslBuilder = sslBuilder != null ? sslBuilder : SSLContexts.custom();
                KeyStore keyStore = clientCertKeyStoreProvider.provide();
                sslBuilder.loadKeyMaterial(keyStore, clientCertKeyStoreProvider.getPassword(),
                        (aliases, socket) -> clientCertAlias);
            }
            if (sslBuilder != null) {
                sslContext = sslBuilder.build();
            }
        } catch (Exception e) {
            log.error("SSLContexts initiation has failed, " + e.getMessage());
        }
        return sslContext != null ? sslContext : SSLContexts.createDefault();
    }

    private boolean hasCredentials() {
        return credsProvider.getCredentials(AuthScope.ANY) != null;
    }

    protected enum JFrogAuthScheme {
        BASIC, BEARER, SPNEGO
    }

    /**
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * WARNING CONFUSING HTTPCLIENT DOC                                                     *
     * we tested this and the *longer* the timeout is the better it will reuse connections  *
     * and not the opposite as we would expect from reading their convoluted doc            *
     * see RTFACT-13074                                                                     *
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     **/
    public static final int CONNECTION_POOL_TIME_TO_LIVE = 30;
    private static final int DEFAULT_MAX_CONNECTIONS = 50;

}
