package org.jfrog.client.http;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ByteArrayEntity;

import javax.annotation.Nonnull;
import java.net.URI;

import static java.util.Objects.requireNonNull;

/**
 * @author Yinon Avraham.
 */
public class HttpRequest extends HttpEntityEnclosingRequestBase {

    public static final String X_ORIGIN_USERNAME = "X-Origin-Username";
    public static final String X_ORIGIN_USER_ADDRESS = "X-Origin-User-Address";

    private final String method;

    HttpRequest(@Nonnull String method) {
        this.method = requireNonNull(method, "method is required").toUpperCase();
    }

    @Override
    public String getMethod() {
        return method;
    }

    public static HttpRequestBase fromRestRequest(RestRequest originalReq, String baseUrl) {
        return fromRestRequest(originalReq, baseUrl, null);
    }

    public static HttpRequestBase fromRestRequest(RestRequest originalReq, String baseUrl, RequestConfig defaultRequestConfig) {
        HttpRequest httpRequest = new HttpRequest(originalReq.getMethod());
        if (originalReq.getBody() != null) {
            httpRequest.setEntity(new ByteArrayEntity(originalReq.getBody()));
        }
        httpRequest.setURI(createUri(originalReq, baseUrl));

        setSocketTimeoutIfNeeded(originalReq, httpRequest, defaultRequestConfig);
        originalReq.getHeaders().forEach(header -> httpRequest.addHeader(header.getKey(), header.getValue()));
        return httpRequest;
    }

    private static void setSocketTimeoutIfNeeded(RestRequest originalReq, HttpRequest httpRequest, RequestConfig defaultRequestConfig) {
        if (originalReq.getSocketTimeout() == null || originalReq.getSocketTimeout() <= 0) {
            return;
        }

        if (defaultRequestConfig == null) {
            throw new IllegalStateException("Default request config must be provided when setting different timeout");
        }

        httpRequest.setConfig(
                RequestConfig.copy(defaultRequestConfig)
                        .setSocketTimeout(originalReq.getSocketTimeout())
                        .build());
    }

    private static URI createUri(RestRequest originalReq, String baseUrl) {
        String uri = (isFullPath(originalReq) ? "" : baseUrl + '/')
                + StringUtils.removeStart(originalReq.getPath(), "/");
        for (Pair<String, String> param : originalReq.getQueryParams()) {
            uri = QueryParamsBuilder.param(uri, param.getKey(), param.getValue());
        }
        return URI.create(uri);
    }

    private static boolean isFullPath(RestRequest originalReq) {
        return originalReq.getPath().matches("^https?://.*");
    }
}
