package org.jfrog.client.util;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.Objects;

/**
 * @author Shay Bagants
 */
class FileKeyStoreProvider implements KeyStoreProvider {
    private static final Logger log = LoggerFactory.getLogger(FileKeyStoreProvider.class);

    private KeyStore keyStore;
    private String keyStorePassword;

    FileKeyStoreProvider(String keyStoreLocation, String keyStorePassword) throws KeyStoreProviderException {
        if (StringUtils.isBlank(keyStoreLocation)) {
            throw new KeyStoreProviderException("No KeyStore file was provided.");
        }
        if (StringUtils.isBlank(keyStorePassword)) {
            throw new KeyStoreProviderException("Password cannot be empty");
        }
        File keyStoreFile = new File(keyStoreLocation);
        if (!keyStoreFile.exists()) {
            log.error("Unable to find KeyStore at: '{}'", keyStoreFile.getAbsolutePath());
            throw new KeyStoreProviderException("Unable to find KeyStore at: " + keyStoreFile.getAbsolutePath() + "");
        }
        this.keyStorePassword = keyStorePassword;

        try (FileInputStream clientCertKeyStore = new FileInputStream(keyStoreFile)) {
            KeyStore jks = KeyStore.getInstance("JKS");
            jks.load(clientCertKeyStore, keyStorePassword.toCharArray());
            keyStore = jks;
        } catch (KeyStoreException | IOException | CertificateException | NoSuchAlgorithmException e) {
            log.error("Unable to load certificate: {}", e.getMessage());
            log.debug("Unable to load certificate: {}", e);
            throw new KeyStoreProviderException("Unable to load certificate.", e);
        }
    }

    @Override
    public KeyStore provide() throws KeyStoreException {
        return Objects.requireNonNull(keyStore);
    }

    @Override
    public char[] getPassword() {
        return keyStorePassword.toCharArray();
    }
}
