package org.jfrog.common.config.diff;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import javax.annotation.processing.AbstractProcessor;
import javax.annotation.processing.RoundEnvironment;
import javax.annotation.processing.SupportedAnnotationTypes;
import javax.annotation.processing.SupportedSourceVersion;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.ElementFilter;
import javax.lang.model.util.Elements;
import javax.lang.model.util.Types;
import javax.tools.Diagnostic;
import javax.tools.JavaFileObject;
import java.io.IOException;
import java.io.Writer;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Main entry point to the diff engine, when marking a bean with {@link GenerateDiffFunction} annotation, the annotation
 * processor provides this class with the annotated bean and the generator creates a new generated java class that
 * produce a list of diffs for two objects from the same annotated type
 *
 * read more about the diff mechanism here:
 * https://docs.google.com/document/d/1b2sjx9lfkNIMvywETeKQbRS7WhI3N1-xddS8IGjgwMs/edit?usp=sharing
 *
 * @author Noam Shemesh
 */
@SupportedAnnotationTypes("org.jfrog.common.config.diff.GenerateDiffFunction")
@SupportedSourceVersion(SourceVersion.RELEASE_11)
public class DiffProcessor extends AbstractProcessor {

    private Elements elementUtils;
    private Types typeUtils;

    @Override
    public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
        typeUtils = processingEnv.getTypeUtils();
        elementUtils = processingEnv.getElementUtils();

        try {
            Collection<? extends Element> annotatedElements = getAnnotatedElementsIfValid(roundEnv);
            if (annotatedElements == null) {
                return false;
            }

            // Also retrieving all classes under one package
            Map<PackageElement, Set<Element>> packageToClassNames = annotatedElements.stream()
                    .collect(Collectors.groupingBy(this::getPackage))
                    .entrySet().stream()
                    .map(entry -> Pair.of(entry.getKey(),
                            ImmutableSet.<Element>builder()
                                    .addAll(entry.getValue())
                                    .addAll(expandPackageElements(entry.getKey()))
                                    .build()))
                    .collect(Collectors.toMap(Pair::getKey, Pair::getValue));

            validateReferencedClasses(packageToClassNames, annotatedElements);

            packageToClassNames.forEach(this::generateForPackage);
        } catch (FailWithError e) {
            processingEnv.getMessager().printMessage(Diagnostic.Kind.ERROR, e.getMessage(), e.element);
            return false;
        }

        return true;
    }

    private Set<? extends Element> expandPackageElements(PackageElement pack) {
        return pack.getEnclosedElements().stream()
                .filter(elem -> elem.getAnnotation(GenerateDiffFunction.class) != null)
                .collect(Collectors.toSet());
    }

    private Collection<? extends Element> getAnnotatedElementsIfValid(RoundEnvironment roundEnv) {
        if (roundEnv.errorRaised()) {
            return null;
        }
        Collection<? extends Element> annotatedElements = roundEnv.getElementsAnnotatedWith(GenerateDiffFunction.class);
        if (annotatedElements.size() == 0) {
            return null;
        }

        List<Element> notClassElements = annotatedElements.stream()
                .filter(element -> !element.getKind().isInterface() && !element.getKind().isClass())
                .collect(Collectors.toList());

        if (notClassElements.size() > 0) {
            notClassElements.forEach(element -> processingEnv.getMessager().printMessage(Diagnostic.Kind.ERROR,
                    "Only interfaces or classes are supported for @" + GenerateDiffFunction.class.getSimpleName(), element));

            return null;
        }

        return annotatedElements;
    }

    private void validateReferencedClasses(Map<PackageElement, Set<Element>> packageToClassNames,
            Collection<? extends Element> annotatedElements) {
        List<String> elementsNames = annotatedElements.stream()
                .map(elem -> elem.getSimpleName().toString())
                .collect(Collectors.toList());

        Set<String> duplicates = elementsNames.stream().filter(i -> Collections.frequency(elementsNames, i) > 1)
                .collect(Collectors.toSet());
        if (duplicates.size() > 0) {
            throw new FailWithError(
                    "Duplicate class names are found: " + duplicates.toString(), null);
        }

        packageToClassNames.entrySet().stream()
                .map(entry -> Pair.of(entry.getKey(), entry.getValue().stream()
                        .collect(Collectors.toMap(this::elementName, Function.identity()))))
                .sorted(Comparator.comparing(pair -> pair.getLeft().getSimpleName().toString()))
                .forEach(this::validateMissingElementsAnnotations);
    }

    private void validateMissingElementsAnnotations(Pair<PackageElement, Map<String, Element>> packageElements) {
        Map<Element, Set<TypeMirror>> requiredTypes = packageElements.getRight().values().stream()
                .collect(Collectors.toMap(
                        Function.identity(),
                        value -> streamOfNotAtomicNotReferenceFieldsTypes(value).collect(Collectors.toSet())
                ));

        validateAnnotation(packageMissingElements(packageElements.getRight(), requiredTypes));
    }

    private void validateAnnotation(Map<Element, Set<Element>> missingElements) {
        List<Element> notAnnotatedElements = missingElements.keySet().stream()
                .filter(element -> element.getAnnotation(GenerateDiffFunction.class) == null)
                .distinct()
                .collect(Collectors.toList());

        if (notAnnotatedElements.size() > 0) {
            throw new FailWithError("Referenced class" + (notAnnotatedElements.size() > 1 ? "es are" : " is") +
                    " not annotated with @" + GenerateDiffFunction.class.getSimpleName() + ". " + notAnnotatedElements.toString() +
                    ". First missing element referenced by class " + missingElements.get(notAnnotatedElements.get(0)) +
                    hierarchicalRefs(missingElements, missingElements.get(notAnnotatedElements.get(0))),
                    notAnnotatedElements.stream().findFirst().orElse(null));
        }
    }

    private String hierarchicalRefs(Map<Element, Set<Element>> missingElements, Set<Element> elements) {
        if (elements.stream().noneMatch(missingElements::containsKey)) {
            return "";
        }

        Set<Element> referencedElements = elements.stream()
                .flatMap(elem -> missingElements.get(elem).stream())
                .collect(Collectors.toSet());

        return ", that was referenced from " + referencedElements.stream()
                .map(Object::toString)
                .collect(Collectors.joining(", ")) +
                hierarchicalRefs(missingElements, referencedElements);
    }

    private Map<Element, Set<Element>> packageMissingElements(Map<String, Element> packageElements, Map<Element, Set<TypeMirror>> requiredTypes) {
        return requiredTypes.entrySet().stream()
                .flatMap(entry -> (Stream<Pair<Element, TypeMirror>>) entry.getValue().stream().map(value -> Pair.of(entry.getKey(), value)))
                .filter(refAndType -> !isPrimitiveOrSimilar(refAndType.getRight()))
                .flatMap(this::extractTypeArguments)
                .map(refAndType -> pairOfValue(refAndType, extractSpecificType(refAndType.getRight())))
                .filter(refAndType -> !isPrimitiveOrSimilar(refAndType.getRight())) // Again, maybe arrived from map/collection generics
                .map(refAndType -> pairOfValue(refAndType, typeUtils.asElement(refAndType.getRight())))
                .filter(refAndElement -> !packageElements.containsKey(refAndElement.getRight().getSimpleName().toString()))
                .collect(Collectors.toMap(Pair::getRight, pair -> ImmutableSet.of(pair.getLeft()),
                        (pair1, pair2) -> ImmutableSet.<Element>builder().addAll(pair1).addAll(pair2).build()));
    }

    private <T> Pair<Element, T> pairOfValue(Pair<Element, TypeMirror> refAndType, T value) {
        return Pair.of(refAndType.getLeft(), value);
    }

    private Stream<Pair<Element, TypeMirror>> extractTypeArguments(Pair<Element, TypeMirror> refAndType) {
        return (isMap(refAndType.getRight()) || isCollection(refAndType.getRight())) ?
                ((DeclaredType) refAndType.getRight()).getTypeArguments().stream()
                        .map(res -> Pair.of(refAndType.getLeft(), res)) :
                Stream.of(refAndType);
    }

    private void generateForPackage(PackageElement pack, Collection<? extends Element> annotatedElements) {
        String className = DiffFunctions.class.getSimpleName() + "Impl";

        StringBuilder source = new StringBuilder();
        List<String> functionNames = new ArrayList<>();

        boolean withComponentStereotype = annotatedElements.stream()
                .anyMatch(elem -> elem.getAnnotation(GenerateDiffFunction.class).withComponentStereotype());
        String packageName = pack.getQualifiedName().toString();
        addPackageAndImports(source, packageName, annotatedElements, withComponentStereotype);
        addClassDefAndBasicMethods(className, source, withComponentStereotype);
        List<String> fields = annotatedElements.stream()
                .flatMap(element -> addMethodForAnnotatedClass(source, packageName, functionNames, element).stream())
                .distinct()
                .collect(Collectors.toList());
        fields.forEach(field -> source.append("  ").append(field).append("\n"));
        addConstructor(className, source, functionNames);
        source.append("}\n");

        CharSequence filename = pack.getQualifiedName().toString() + "." + className;
        writeJavaSource(source, filename);
    }

    private void addPackageAndImports(StringBuilder source, String packageName, Collection<? extends Element> annotatedElements,
            boolean withComponentStereotype) {
        source.append("package ").append(packageName).append(";\n\n")
                .append("import org.apache.commons.lang3.builder.*;\n")
                .append("import java.util.function.BiFunction;\n")
                .append("import java.util.*;\n")
                .append("import java.util.stream.*;\n")
                .append("import ").append(DiffUtils.class.getName()).append(";\n")
                .append(withComponentStereotype ? "import org.springframework.stereotype.Component;\n" : "")
                .append(annotatedElements.stream()
                        .filter(elem -> !packageName.equals(getPackageName(elem)))
                        .map(elem -> "import " + getPackageName(elem) + "." + elem.getSimpleName() + ";")
                        .collect(Collectors.joining("\n", "\n", "\n"))
                )
                .append("import ").append(DiffFunctions.class.getName()).append(";\n\n");
    }

    private void addClassDefAndBasicMethods(String className, StringBuilder source, boolean withComponentStereotype) {
        source.append(withComponentStereotype ? "@Component\n" : "")
                .append("public class ").append(className).append(" implements DiffFunctions").append("{\n")
                .append("  Map<String, BiFunction<Object, Object, DiffResult>> diffFunctions = new HashMap<>();\n")
                .append("  @Override\n  public <T> DiffResult diffFor(Class<T> c, T object, T other) {\n")
                .append("    return diffFunctions.get(c.getSimpleName()).apply(object, other);\n")
                .append("  }\n\n")
                .append("  @Override\n  public <T> boolean containsClass(Class<T> c) {\n")
                .append("    return diffFunctions.containsKey(c.getSimpleName());\n")
                .append("  }\n\n");
    }

    private static void addConstructor(String className, StringBuilder source, List<String> functionNames) {
        source.append("  public ").append(className).append("() {")
                .append(functionNames.stream().map(object -> "diffFunctions.put(\"" + object + "\", " +
                        "(object, other) -> " + DiffUtils.toFieldName(object) + "((" + object + ")object, (" + object + ")other));").collect(
                        Collectors.joining("\n    ", "\n    ", "\n")))
                .append("  }\n\n");
    }

    private List<String> addMethodForAnnotatedClass(StringBuilder source, String packageName,
            List<String> functionNames, Element element) {
        String functionName = element.getSimpleName().toString();
        functionNames.add(functionName);

        StringBuilder diffSourceBuilder = new StringBuilder();
        diffSourceBuilder.append("  private DiffResult ").append(DiffUtils.toFieldName(functionName))
                .append("(").append(functionName).append(" object").append(", ").append(functionName).append(" other) {\n");

        List<String> fieldsToAppend = new ArrayList<>();
        if (element.getAnnotation(GenerateDiffFunction.class).internalDiff()) {
            addInternalCallToMethod(diffSourceBuilder);
        } else {
            fieldsToAppend = generateDiffMethodContentForMethods(diffSourceBuilder, packageName, element);
        }

        diffSourceBuilder.append("  }\n");

        source.append(diffSourceBuilder);

        return fieldsToAppend;
    }

    private Set<VariableElement> relevantFieldsInElement(Element element) {
        return notDistinctFieldsInElement(element).stream()
                .filter(field -> field.getAnnotation(DiffIgnore.class) == null)
                .filter(distinctByKey(e -> e.getSimpleName().toString()))
                .collect(Collectors.toSet());
    }

    private Set<VariableElement> notDistinctFieldsInElement(Element element) {
        if (element == null ||
                Stream.of("Object", "Enum").anyMatch(element.getSimpleName().toString()::equals) ||
                Stream.of(ElementKind.CLASS, ElementKind.ENUM, ElementKind.INTERFACE)
                        .noneMatch(element.getKind()::equals)) {
            return ImmutableSet.of();
        }

        ImmutableSet.Builder<VariableElement> builder = ImmutableSet.builder();

        List<VariableElement> currentTypeFields =
                filterAndSortFields(ElementFilter.fieldsIn(element.getEnclosedElements()))
                        .collect(Collectors.toList());

        Set<String> currentTypeFieldsNames = currentTypeFields.stream()
                .map(VariableElement::getSimpleName)
                .map(Name::toString)
                .collect(Collectors.toSet());

        typeUtils.directSupertypes(element.asType()).stream()
                .map(typeUtils::asElement)
                .map(this::notDistinctFieldsInElement)
                .flatMap(Collection::stream)
                .filter(field -> !currentTypeFieldsNames.contains(field.getSimpleName().toString()))
                .forEach(builder::add);

        builder.addAll(currentTypeFields);

        return builder.build();
    }

    private static <T> Predicate<T> distinctByKey(Function<? super T, ?> keyExtractor) {
        Map<Object,Boolean> seen = new ConcurrentHashMap<>();
        return t -> seen.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }

    /**
     * @return Fields to add to wrapping class
     */
    private List<String> generateDiffMethodContentForMethods(StringBuilder source, String packageName,
            Element element) {
        source.append("    DiffBuilder db = new DiffBuilder(object, other, ToStringStyle.DEFAULT_STYLE, false);\n");

        List<String> fields = filterAndSortFields(relevantFieldsInElement(element))
                .map(field -> fieldToLocalDiffMethod(packageName, field))
                .peek(sourceAndFields -> source.append(sourceAndFields.getLeft()))
                .map(Pair::getRight)
                .filter(StringUtils::isNotEmpty)
                .collect(Collectors.toList());

        source.append("    return db.build();\n");

        return fields;
    }

    private Stream<VariableElement> filterAndSortFields(Collection<VariableElement> fields) {
        boolean diffIncludePresent = fields.stream().anyMatch(m -> m.getAnnotation(DiffInclude.class) != null);
        return fields.stream()
                .filter(field -> !diffIncludePresent || field.getAnnotation(DiffInclude.class) != null)
                .filter(field -> field.getModifiers().stream().noneMatch(Modifier.STATIC::equals))
                .sorted(Comparator.comparing(e -> e.getSimpleName().toString()));
    }

    private void addInternalCallToMethod(StringBuilder source) {
        source.append("    return object.diff(other);");
    }

    private Pair<String, String> fieldToLocalDiffMethod(String packageName, VariableElement field) {
        TypeMirror returnType = field.asType();
        String methodName = field.getSimpleName().toString();

        if (isPrimitiveOrSimilar(returnType) ||
                field.getAnnotation(DiffAtomic.class) != null ||
                field.getAnnotation(DiffReference.class) != null ||
                isCollectionOrMapOfPrimitive(returnType)) {
            return simpleAppend(field, methodName, DiffUtils.toMethodName(methodName, isBoolean(returnType)));
        } else {
            return diffMethodForComplexClass(packageName, field);
        }
    }

    private boolean isCollectionOrMapOfPrimitive(TypeMirror returnType) {
        if (isMap(returnType)) {
            return isPrimitiveOrSimilar(((DeclaredType) returnType).getTypeArguments().get(1));
        }

        return isCollection(returnType) &&
                isPrimitiveOrSimilar(((DeclaredType) returnType).getTypeArguments().get(0));
    }

    private Pair<String, String> simpleAppend(VariableElement field, String fieldName,
            String methodName) {
        return simpleAppend(field, fieldName, "object." + methodName + "()", "other." + methodName + "()");
    }

    private Pair<String, String> simpleAppend(VariableElement field, String fieldName, String firstArgumentExtractor, String secondArgumentExtractor) {
        return Pair.of("    db.append(\"" + fieldName + "\", " + appendKeyMethodExecution(firstArgumentExtractor, field) + ", " + appendKeyMethodExecution(secondArgumentExtractor, field) + ");\n", null);
    }

    /**
     * @return Method string and fields to add
     */
    private Pair<String, String> diffMethodForComplexClass(String packageName, VariableElement field) {
        String fieldName = field.getSimpleName().toString();
        TypeMirror returnType = extractSpecificType(field.asType());
        String methodName = DiffUtils.toMethodName(fieldName, isBoolean(returnType));

        Optional<ExecutableElement> method = ElementFilter.methodsIn(field.getEnclosingElement().getEnclosedElements())
                .stream()
                .filter(m -> m.getSimpleName().toString().equals(methodName))
                .findFirst();

        String cast = "";
        if (method.isPresent()) {
            TypeMirror methodReturnType = method.get().getReturnType();

            if (!methodReturnType.equals(returnType)) {
                cast = "(" + returnType.toString() + ")";
            }
        } else {
            processingEnv.getMessager().printMessage(Diagnostic.Kind.NOTE, "Couldn't find method for field " + fieldName +
                    " in class " + field.getEnclosingElement().getSimpleName().toString() + ". Not casting to method return type.");
        }

        if (ElementKind.ENUM.equals(typeUtils.asElement(returnType).getKind())) {
            String enumMethod = typeUtils.asElement(returnType).getAnnotation(GenerateDiffFunction.class).typeMethod();
            return simpleAppend(field, fieldName,
                    "object." + methodName + "() == null ? \"\" : object."+methodName + "()." + enumMethod + "()",
                    "other." + methodName + "() == null ? \"\" : other."+methodName + "()." + enumMethod + "()");
        }

        StringBuilder diffMethod = new StringBuilder();

        diffMethod.append("\n    ").append("DiffUtils.appendDiffResult(db, \"").append(fieldName).append("\", ");

        String fieldToAdd;
        boolean isCollection = false;
        if (isMap(returnType) || (isCollection = isCollection(returnType))) {
            fieldToAdd = diffMethodForMapOrCollection(packageName, returnType, diffMethod, methodName, isCollection);
        } else {
            fieldToAdd = diffMethodForObjectOrPrimitive(packageName, returnType, diffMethod);

            diffMethod.append("\n        ")
                    .append(cast).append(" ").append("object.")
                    .append(methodName).append("()").append(", \n       ")
                    .append(cast).append(" ").append("other.")
                    .append(methodName).append("()").append("));\n");
        }

        return Pair.of(diffMethod.toString(), fieldToAdd);
    }

    private String appendKeyMethodExecution(String toWrap, VariableElement field) {
        if (field.getAnnotation(DiffReference.class) != null) {
            String prefix = "Objects.isNull(" + toWrap + ") ? null : " + toWrap + ".";
            TypeMirror fieldType = field.asType();

            String methodName = DiffUtils.toMethodName(findKeyExtractorMethod(fieldType), false);
            if (isCollection(fieldType)) {
                return prefix + "stream()\n      .map(v -> v." + methodName + "())\n      .collect(Collectors.toList())";
            } else {
                return prefix + methodName + "()";
            }
        }

        return toWrap;
    }

    private String diffMethodForObjectOrPrimitive(String packageName, TypeMirror returnType, StringBuilder diffMethod) {
        Element returnTypeElement = typeUtils.asElement(returnType);
        diffMethod.append("\n        DiffUtils.diffInternalClass(");
        String newField = null;
        if (!isPrimitiveOrSimilar(returnType) && !packageName.equals(getPackageName(returnTypeElement))) {
            String fieldName = DiffUtils.toFieldName(getPackageName(returnTypeElement));
            newField = fieldForPackage(fieldName, returnTypeElement);
            diffMethod.append(fieldName).append(",").append("\n        ")
                    .append(getPackageName(returnTypeElement)).append(".")
                    .append(returnTypeElement.getSimpleName().toString()).append(".class, ");
        } else {
            diffMethod.append("this,").append("\n        ")
                    .append(returnTypeElement.getSimpleName().toString()).append(".class, ");
        }

        return newField;
    }

    /**
     *
     * @return Field to add to the wrapping class
     */
    private String diffMethodForMapOrCollection(String packageName, TypeMirror returnType, StringBuilder diffMethod,
            String methodName, boolean isCollection) {
        TypeMirror erasureType = findErasure(returnType);
        TypeMirror extractedType = extractSpecificType(erasureType);

        String erasureName = "Object";
        String diffFunctions = "this";
        String newField = null;
        String cast = "";

        if (erasureType != null) {
            Element erasureTypeElement = typeUtils.asElement(removeWildcard(extractedType));

            if (!isPrimitiveOrSimilar(erasureType) && !packageName.equals(getPackageName(erasureTypeElement))) {
                diffFunctions = DiffUtils.toFieldName(getPackageName(erasureTypeElement));
                newField = fieldForPackage(diffFunctions, erasureTypeElement);
            }

            if (!erasureType.equals(extractedType)) {
                cast = isCollection ? "(Collection)" : "(Map)";
            }

            erasureName = ((TypeElement) erasureTypeElement).getQualifiedName().toString();
        }

        diffMethod.append("\n        ").append("DiffUtils.diff")
                .append(isCollection ? "Collection" : "Map").append("(")
                .append(diffFunctions).append(",\n        ").append(erasureName);

        diffMethod.append(".class, ");

        if (isCollection) {
            diffMethod.append(erasureType == null || isPrimitiveOrSimilar(erasureType) ? "Object::toString, " :
                    "v -> v." + DiffUtils.toMethodName(findKeyExtractorMethod(extractedType), false) + "(), ");
        }

        diffMethod.append("\n        ")
                .append(cast).append("object.").append(methodName).append("(),")
                .append(cast).append("other.").append(methodName).append("()").append("));\n");

        return newField;
    }

    private String fieldForPackage(String normalizedPackageName, Element element) {
        return "private " + DiffFunctions.class.getSimpleName() + " " + normalizedPackageName +
                " = new " + getPackageName(element) + "." + DiffFunctions.class.getSimpleName() + "Impl();";
    }

    private TypeMirror findErasure(TypeMirror type) {
        if (!(type instanceof DeclaredType)) {
            return null;
        }
        DeclaredType declaredType = ((DeclaredType) type);
        boolean isCollection = isCollection(declaredType);
        TypeMirror erasureType = null;
        if (declaredType.getTypeArguments().size() >= (isCollection ? 1 : 2)) {
            erasureType = declaredType.getTypeArguments().get(isCollection ? 0 : 1);
        }
        return erasureType;
    }

    private TypeMirror removeWildcard(TypeMirror type) {
        return type.getKind() == TypeKind.WILDCARD ? ((WildcardType) type).getExtendsBound() : type;
    }

    private TypeMirror extractSpecificType(TypeMirror type) {
        if (type == null) {
            return null;
        }

        TypeMirror specificType = removeWildcard(type);

        return typeUtils.asElement(specificType).getAnnotationMirrors().stream()
                .filter(a -> a.getAnnotationType().toString().equals(GenerateDiffFunction.class.getName()))
                .findFirst()
                .map(annotation -> annotation.getElementValues().entrySet().stream()
                        .filter(entry -> "defaultImpl".equals(entry.getKey().getSimpleName().toString()))
                        .map(Map.Entry::getValue)
                        .map(AnnotationValue::getValue)
                        .filter(val -> !val.equals(Object.class.getName()))
                        .findFirst()
                        .map(val -> elementUtils.getTypeElement(val.toString()).asType())
                        .orElse(specificType))
                .orElse(specificType);
    }

    private String findKeyExtractorMethod(TypeMirror type) {
        TypeMirror extractedType = extractSpecificType(isCollection(type) ? findErasure(type) : type);

        return relevantFieldsInElement(typeUtils.asElement(extractedType)).stream()
                .filter(field -> field.getAnnotation(DiffKey.class) != null)
                .map(method -> method.getSimpleName().toString())
                .findFirst()
                .orElseThrow(() -> new FailWithError("Classes that are referenced as List or by @" + DiffReference.class.getSimpleName() +
                        " must define a method with @" + DiffKey.class.getSimpleName() + ". [" + type + "]", typeUtils.asElement(type)));
    }

    // Utility methods:

    private String elementName(Element elem) {
        return elem.getSimpleName().toString();
    }

    private Stream<TypeMirror> streamOfNotAtomicNotReferenceFieldsTypes(Element element) {
        return relevantFieldsInElement(element).stream()
                .filter(varType -> varType.getAnnotation(DiffAtomic.class) == null)
                .filter(varType -> varType.getAnnotation(DiffReference.class) == null)
                .map(VariableElement::asType);
    }

    private PackageElement getPackage(Element elem) {
        return (PackageElement) elem.getEnclosingElement();
    }

    private String getPackageName(Element elem) {
        return getPackage(elem).getQualifiedName().toString();
    }

    private boolean isCollection(TypeMirror returnType) {
        DeclaredType wildcardList = typeUtils.getDeclaredType(
                elementUtils.getTypeElement(Collection.class.getName()),
                typeUtils.getWildcardType(null, null));
        return typeUtils.isAssignable(returnType, wildcardList);
    }

    private boolean isBoolean(TypeMirror returnType) {
        return returnType.getKind().equals(TypeKind.BOOLEAN);
    }

    private boolean isPrimitiveOrSimilar(TypeMirror returnType) {
        TypeMirror typeToCheck = returnType.getKind() != TypeKind.ARRAY
                ? returnType : ((ArrayType) returnType).getComponentType();

        return returnType.getKind().isPrimitive() ||
                Stream.of(String.class, Long.class, Integer.class, Boolean.class)
                        .anyMatch(type -> isType(typeToCheck, type));
    }

    private boolean isMap(TypeMirror returnType) {
        DeclaredType wildcardMap = typeUtils.getDeclaredType(
                elementUtils.getTypeElement(Map.class.getName()),
                typeUtils.getWildcardType(null, null),
                typeUtils.getWildcardType(null, null));
        return typeUtils.isAssignable(returnType, wildcardMap);
    }

    private boolean isType(TypeMirror returnType, Class<?> c) {
        return typeUtils.isSameType(returnType, elementUtils.getTypeElement(c.getName()).asType());
    }

    private void writeJavaSource(StringBuilder builder, CharSequence className) {
        try {
            JavaFileObject javaFileObject = processingEnv.getFiler().createSourceFile(className);
            Writer writer = javaFileObject.openWriter();
            writer.write(builder.toString());
            writer.close();
        } catch (IOException e) {
            processingEnv.getMessager().printMessage(
                    Diagnostic.Kind.ERROR, "Couldn't write java source file to " + className + ". " + e.getMessage());
            e.printStackTrace();
        }
    }

    private class FailWithError extends RuntimeException {
        private Element element;

        FailWithError(String message, Element element) {
            super(message);
            this.element = element;
        }
    }
}

