/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.beans.metadata.api.model;

import java.io.ObjectStreamException;
import java.io.Serializable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.xml.bind.annotation.XmlType;

import org.jboss.util.JBossObject;
import org.jboss.util.JBossStringBuilder;
import org.jboss.xb.annotations.JBossXmlAdaptedType;

/**
 * How a qualifier should be parsed
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
@XmlType(propOrder = {})
@JBossXmlAdaptedType(valueAdapter=QualifierContentValueAdapter.class)
public class QualifierContent extends JBossObject implements Serializable
{
   private static final long serialVersionUID = 1L;

   /** The qualifier is a simple string */
   public static QualifierContent STRING = new QualifierContent("STRING");
   
   /** The qualifier is an annotation */
   public static QualifierContent ANNOTATION = new QualifierContent("ANNOTATION");

   //Not likely to be more than one thread updating at any time
   private static ConcurrentMap<String, QualifierContent> values = new ConcurrentHashMap<String, QualifierContent>(3, .75f, 1);

   static
   {
      values.put(STRING.getContentString(), STRING);
      values.put(ANNOTATION.getContentString(), ANNOTATION);
   }

   private String contentString;
   
   private QualifierContent(String contentString)
   {
      this.contentString = checkString(contentString);
   }
   
   /**
    * Get the content string
    * 
    * @return the content string
    */
   public String getContentString()
   {
      return contentString;
   }
   
   /**
    * Checks if the other object is also a QualifierContent and has the same 
    * contentString
    * @param object The object we want to compare with.
    */
   public boolean equals(Object object)
   {
      if (object == null || object instanceof QualifierContent == false)
         return false;
      QualifierContent other = (QualifierContent) object;
      return contentString.equals(other.contentString);
   }
   
   public void toString(JBossStringBuilder buffer)
   {
      buffer.append(contentString);
   }

   public void toShortString(JBossStringBuilder buffer)
   {
      buffer.append(contentString);
   }

   protected int getHashCode()
   {
      return contentString.hashCode();
   }

   protected Object readResolve() throws ObjectStreamException
   {
      return values.get(contentString);   
   }
   
   private static String checkString(String contentString)
   {
      if (contentString == null)
         throw new IllegalArgumentException("Null content string");
      if (contentString.trim().length() == 0)
         throw new IllegalArgumentException("Empty content string");
      return contentString.trim().toUpperCase();
   }
   
   /**
    * Get the correct content type for the passed in string. If the content type
    * has already been created return the existing instance. Otherwise create it
    * and store it for future reference.
    * 
    * @param contentString content type string
    * @return the content type instance
    */
   public static QualifierContent getContent(String contentString)
   {
      String str = checkString(contentString);
      QualifierContent content = values.get(str);
      if (content == null)
      {
         content = new QualifierContent(str);
         QualifierContent old = values.putIfAbsent(str, content);
         if (old != null)
            content = old;
      }
      return content;
   }
}
