/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.kernel.plugins.dependency;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.jboss.beans.metadata.api.model.QualifierPoint;
import org.jboss.beans.metadata.spi.MetaDataVisitorNode;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.kernel.spi.dependency.KernelControllerContext;
import org.jboss.kernel.spi.qualifier.QualifierMatchers;
import org.jboss.util.JBossObject;
import org.jboss.util.JBossStringBuilder;

/**
 * If an injection point uses contextual injection and also has qualifiers attached,
 * this will be used when trying to find a matching context in the controller
 *
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class ClassAndQualifiersKey extends JBossObject implements QualifierKey
{
   /**
    * The type of class we are looking for
    */
   protected final Class<?> type;

   /**
    * The state of the dependency
    */
   protected final ControllerState dependentState;

   /**
    * The context containing the wanted qualifiers
    */
   protected final KernelControllerContext context;

   /**
    * The qualifiers defined on the injection point
    */
   protected final Set<Object> injectionPointQualifiers;

   /**
    * If injection point qualifiers are specified, set this to true to ignore any qualifiers from bean level
    */
   protected final boolean ignoreBeanQualifiers;

   /**
    * The injection point type that this lookup will be done for
    */
   protected final QualifierPoint injectionPointType;

   /**
    * The parent nodes, with the lowest node first
    */
   protected final List<MetaDataVisitorNode> parentNodes;

   /**
    * Constructor
    *
    * @param dependentState           the state the dependency must be in
    * @param context                  the controller context owning this dependency
    * @param ignoreBeanQualifiers     if injection point qualifiers are specified, set this to true to ignore any qualifiers from bean level
    * @param injectionPointType       injection point type
    * @param parentNodes              the nodes containing the injection. Lowest level come first in the list
    * @param injectionPointQualifiers the qualifiers applying only to this injection point
    * @param type                     the type of class we are looking for
    */
   public ClassAndQualifiersKey(ControllerState dependentState, KernelControllerContext context, boolean ignoreBeanQualifiers, QualifierPoint injectionPointType, List<MetaDataVisitorNode> parentNodes, Set<Object> injectionPointQualifiers, Class<?> type)
   {
      this.dependentState = dependentState == null ? ControllerState.INSTALLED : dependentState;
      this.context = context;
      this.injectionPointQualifiers = injectionPointQualifiers != null && injectionPointQualifiers.size() > 0 ? injectionPointQualifiers : new HashSet<Object>();
      this.type = type;
      this.ignoreBeanQualifiers = ignoreBeanQualifiers;
      this.parentNodes = parentNodes;
      this.injectionPointType = injectionPointType;
   }

   /**
    * Get the parentNodes
    *
    * @return the parentNodes
    */
   public List<MetaDataVisitorNode> getParentNodes()
   {
      return parentNodes;
   }

   /**
    * Add qualifiers coming from injection point annotations
    *
    * @param annotations the qualifier annotations
    */
   public void addQualifiersFromAnnotations(Set<Annotation> annotations)
   {
      injectionPointQualifiers.addAll(annotations);
   }

   /**
    * Look for a context in the controller that most closely matches the type
    * and qualifiers contained in this key
    *
    * @param controller the controller
    */
   public ControllerContext search(KernelController controller)
   {
      Set<ControllerContext> contexts = controller.getContexts(type, dependentState);

      if (contexts == null || contexts.size() == 0)
         return null;

      Set<Object> requiredQualifiers;
      if (injectionPointQualifiers != null)
      {
         if (ignoreBeanQualifiers)
            requiredQualifiers = injectionPointQualifiers;
         else
         {
            requiredQualifiers = QualifiersMdrUtil.mergeRequiredQualifiersFromMdr(context, injectionPointType);
            if (requiredQualifiers == null)
               requiredQualifiers = new HashSet<Object>();
            requiredQualifiers.addAll(injectionPointQualifiers);
         }
      }
      else
      {
         requiredQualifiers = QualifiersMdrUtil.mergeRequiredQualifiersFromMdr(context, injectionPointType);
      }

      Set<Object> optionalQualifiers = ignoreBeanQualifiers ? null : QualifiersMdrUtil.mergeOptionalQualifiersFromMdr(context, injectionPointType);
      if ((requiredQualifiers == null || requiredQualifiers.size() == 0) && (optionalQualifiers == null || optionalQualifiers.size() == 0))
         return getFirstContext(contexts);

      List<ControllerContext> found = getContextWithAllRequiredAndMostOptionalQualifiers(requiredQualifiers, optionalQualifiers, contexts);
      return getFirstContext(found);
   }

   private ControllerContext getFirstContext(Collection<? extends ControllerContext> contexts)
   {
      if (contexts.size() == 0)
         return null;

      ControllerContext found = null;
      if (contexts.size() > 0)
      {
         for (ControllerContext context : contexts)
         {
            if (found != null)
            {
               log.warn("Multiple beans match qualifiers and class [enable trace log for details]: " + type);
               if (log.isTraceEnabled())
               {
                  log.trace(this + ". Matching contexts: " + contexts);
               }
               return null;
            }
            found = context;
         }
      }

      return found;
   }

   private List<ControllerContext> getContextWithAllRequiredAndMostOptionalQualifiers(Set<Object> requiredQualifiers, Set<Object> optionalQualifiers, Collection<ControllerContext> contexts)
   {
      int max = 0;
      List<ControllerContext> found = new ArrayList<ControllerContext>();

      for (ControllerContext context : contexts)
      {
         Set<Object> suppliedQualifiers = QualifiersMdrUtil.mergeSuppliedQualifiersFromMdr(context);
         if (suppliedQualifiers == null) // TODO -- too strict?
            suppliedQualifiers = Collections.emptySet();

         int matches = 0;

         boolean allRequired = true;
         if (requiredQualifiers != null && requiredQualifiers.size() > 0)
         {
            for (Object qualifier : requiredQualifiers)
            {
               if (QualifierMatchers.getInstance().matches(context, suppliedQualifiers, qualifier))
               {
                  matches++;
                  continue;
               }
               allRequired = false;
               break;
            }
         }
         if (allRequired && optionalQualifiers != null && optionalQualifiers.size() > 0)
         {
            for (Object qualifier : optionalQualifiers)
            {
               if (QualifierMatchers.getInstance().matches(context, suppliedQualifiers, qualifier))
               {
                  matches++;
                  continue;
               }
            }
         }
         if (allRequired)
         {
            if (matches > max)
            {
               found.clear();
               found.add(context);
               max = matches;
            }
            else if (matches == max)
            {
               found.add(context);
            }
         }
      }
      return found;
   }

   @Override
   protected void toString(JBossStringBuilder buffer)
   {
      buffer.append("class=");
      buffer.append(type);
      buffer.append(" qualifiers=");
      buffer.append(QualifiersMdrUtil.mergeRequiredQualifiersFromMdr(context, injectionPointType));
   }

   @Override
   public void toShortString(JBossStringBuilder buffer)
   {
      toString(buffer);
   }
}
