/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
  *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bootstrap.api.descriptor;

import java.io.InputStream;
import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.logging.Logger;

/**
 * {@link BootstrapDescriptor} implementation backed by a named resource
 * to be obtained via a {@link ClassLoader}
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 */
public class ClassLoaderResourceBootstrapDescriptor extends BootstrapDescriptorBase implements BootstrapDescriptor
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Logger
    */
   private static final Logger log = Logger.getLogger(ClassLoaderResourceBootstrapDescriptor.class.getName());

   /**
    * {@link PrivilegedAction} to obtain the TCCL
    *
    * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
    */
   private static enum GetTcclAction implements PrivilegedAction<ClassLoader> {
      INSTANCE;

      @Override
      public ClassLoader run()
      {
         return Thread.currentThread().getContextClassLoader();
      };
   }

   //-------------------------------------------------------------------------------------||
   // Instance Members -------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Underlying handler
    */
   private final BootstrapDescriptor urlDescriptor;

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Creates a new instance backed by a resource of the specified 
    * name located within (or accessible from) the Thread Context {@link ClassLoader}
    * @param name
    * @throws IllegalArgumentException If the name is not specified, or
    * if a resource of the specified name could not be found from TCCL.
    */
   public ClassLoaderResourceBootstrapDescriptor(final String name) throws IllegalArgumentException
   {
      // Get TCCL
      this(name, AccessController.doPrivileged(GetTcclAction.INSTANCE));
   }

   /**
    * Creates a new instance backed by a resource of the specified 
    * name located within (or accessible from) the specified {@link ClassLoader} 
    * @param name
    * @param cl
    * @throws IllegalArgumentException If either the cl or name is not specified, or
    * if a resource of the specified name could not be found from the specified cl.
    */
   public ClassLoaderResourceBootstrapDescriptor(final String name, final ClassLoader cl)
         throws IllegalArgumentException
   {
      // Precondition checks
      if (cl == null)
      {
         throw new IllegalArgumentException("cl must be specified");
      }
      if (name == null || name.length() == 0)
      {
         throw new IllegalArgumentException("name must be specified");
      }

      // Get the URL for this CL Resource
      final URL url = cl.getResource(name);

      // Ensure found
      if (url == null)
      {
         throw new IllegalArgumentException("Could not find requested resource \"" + name + "\" in " + cl);
      }

      // Make a descriptor impl which can handle this type
      final BootstrapDescriptor delegate = new UrlBootstrapDescriptor(url);

      // Set
      this.urlDescriptor = delegate;
   }

   //-------------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * @see org.jboss.bootstrap.api.descriptor.BootstrapDescriptor#getName()
    */
   @Override
   public String getName()
   {
      return urlDescriptor.getName();
   }

   /**
    * @see org.jboss.bootstrap.api.descriptor.BootstrapDescriptor#getContent()
    */
   public InputStream getContent()
   {
      return urlDescriptor.getContent();
   }
}
