/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
  *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bootstrap.api.descriptor;

import java.io.File;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.logging.Logger;

/**
 * {@link BootstrapDescriptor} implementation backed by a {@link File}
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 */
public class FileBootstrapDescriptor extends BootstrapDescriptorBase implements BootstrapDescriptor
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Logger
    */
   private static final Logger log = Logger.getLogger(FileBootstrapDescriptor.class.getName());

   //-------------------------------------------------------------------------------------||
   // Instance Members -------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Underlying handler
    */
   private final BootstrapDescriptor urlDescriptor;

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Creates a new instance with the specified {@link File}
    * @param file
    * @throws IllegalArgumentException If the file is not specified, does not
    * exist, or is a directory
    */
   public FileBootstrapDescriptor(final File file) throws IllegalArgumentException
   {
      // Precondition checks
      if (file == null)
      {
         throw new IllegalArgumentException("file must be specified");
      }
      if (!file.exists())
      {
         throw new IllegalArgumentException("file must exist: " + file.getAbsolutePath());
      }
      if (file.isDirectory())
      {
         throw new IllegalArgumentException("file must not be a directory: " + file.getAbsolutePath());
      }

      // Get the URL for this File
      final URL url;
      try
      {
         url = file.toURI().toURL();
      }
      catch (final MalformedURLException e)
      {
         throw new RuntimeException("Error in obtaining URL for File: " + file.getAbsolutePath(), e);
      }

      // Make a descriptor impl which can handle this type
      final BootstrapDescriptor delegate = new UrlBootstrapDescriptor(url);

      // Set
      this.urlDescriptor = delegate;
   }

   //-------------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * @see org.jboss.bootstrap.api.descriptor.BootstrapDescriptor#getName()
    */
   @Override
   public String getName()
   {
      return urlDescriptor.getName();
   }

   /**
    * @see org.jboss.bootstrap.api.descriptor.BootstrapDescriptor#getContent()
    */
   public InputStream getContent()
   {
      return urlDescriptor.getContent();
   }
}
