/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.server.network;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InterfaceAddress;
import java.net.NetworkInterface;
import java.net.UnknownHostException;
import java.util.Enumeration;
import java.util.function.Predicate;
import org.infinispan.server.Server;

public class NetworkAddress {
    static final int[] NETMASK_BY_PREFIX = new int[]{0, 128, 192, 224, 240, 248, 252, 254, 255};
    private final String name;
    private final InetAddress address;
    private final short prefixLength;

    NetworkAddress(String name, InterfaceAddress address) {
        this.name = name;
        this.address = address.getAddress();
        this.prefixLength = address.getNetworkPrefixLength();
    }

    NetworkAddress(String name, InetAddress address) {
        this.name = name;
        this.address = address;
        this.prefixLength = 0;
    }

    public String getName() {
        return this.name;
    }

    public InetAddress getAddress() {
        return this.address;
    }

    public String cidr() {
        StringBuilder sb = new StringBuilder();
        byte[] bytes = this.address.getAddress();
        short prefix = this.prefixLength;
        for (int i = 0; i < bytes.length; ++i) {
            if (i > 0) {
                sb.append('.');
            }
            int bits = prefix > 8 ? 8 : (int)prefix;
            sb.append(bytes[i] & NETMASK_BY_PREFIX[bits]);
            prefix = (short)(prefix - bits);
        }
        sb.append('/').append(this.prefixLength);
        return sb.toString();
    }

    public String toString() {
        return "NetworkAddress{name='" + this.name + '\'' + ", address=" + this.address + ", prefixLength=" + this.prefixLength + '}';
    }

    public short getPrefixLength() {
        return this.prefixLength;
    }

    public static NetworkAddress fromString(String name, String value) throws IOException {
        switch (value) {
            case "GLOBAL": {
                return NetworkAddress.globalAddress(name);
            }
            case "LOOPBACK": {
                return NetworkAddress.loopback(name);
            }
            case "NON_LOOPBACK": {
                return NetworkAddress.nonLoopback(name);
            }
            case "SITE_LOCAL": {
                return NetworkAddress.siteLocal(name);
            }
            case "LINK_LOCAL": {
                return NetworkAddress.linkLocalAddress(name);
            }
        }
        if (value.startsWith("match-interface:")) {
            return NetworkAddress.matchInterface(name, value.substring(value.indexOf(58) + 1));
        }
        if (value.startsWith("match-address:")) {
            return NetworkAddress.matchAddress(name, value.substring(value.indexOf(58) + 1));
        }
        if (value.startsWith("match-host:")) {
            return NetworkAddress.matchHost(name, value.substring(value.indexOf(58) + 1));
        }
        return NetworkAddress.inetAddress(name, value);
    }

    public static NetworkAddress globalAddress(String name) throws IOException {
        return new NetworkAddress(name, NetworkAddress.findAddress(a -> !a.isLoopbackAddress() && !a.isSiteLocalAddress() && !a.isLinkLocalAddress()));
    }

    public static NetworkAddress loopback(String name) throws IOException {
        return new NetworkAddress(name, NetworkAddress.findAddress(InetAddress::isLoopbackAddress));
    }

    public static NetworkAddress nonLoopback(String name) throws IOException {
        return new NetworkAddress(name, NetworkAddress.findAddress(a -> !a.isLoopbackAddress()));
    }

    public static NetworkAddress siteLocal(String name) throws IOException {
        return new NetworkAddress(name, NetworkAddress.findAddress(InetAddress::isSiteLocalAddress));
    }

    public static NetworkAddress matchInterface(String name, String regex) throws IOException {
        return new NetworkAddress(name, NetworkAddress.findInterface(i -> i.getName().matches(regex)).getInterfaceAddresses().get(0));
    }

    public static NetworkAddress matchAddress(String name, String regex) throws IOException {
        return new NetworkAddress(name, NetworkAddress.findAddress(a -> a.getHostAddress().matches(regex)));
    }

    public static NetworkAddress matchHost(String name, String regex) throws IOException {
        return new NetworkAddress(name, NetworkAddress.findAddress(a -> a.getHostName().matches(regex)));
    }

    public static NetworkAddress match(String name, Predicate<NetworkInterface> ifMatcher, Predicate<InetAddress> addressMatcher) throws IOException {
        NetworkInterface networkInterface = NetworkAddress.findInterface(ifMatcher);
        InterfaceAddress address = NetworkAddress.findAddress(networkInterface, addressMatcher);
        if (address != null) {
            return new NetworkAddress(name, address);
        }
        throw new IOException("No matching addresses found");
    }

    public static NetworkAddress inetAddress(String name, String value) throws UnknownHostException {
        return new NetworkAddress(name, InetAddress.getByName(value));
    }

    public static NetworkAddress anyAddress(String name) throws UnknownHostException {
        return new NetworkAddress(name, InetAddress.getByAddress(new byte[]{0, 0, 0, 0}));
    }

    public static NetworkAddress linkLocalAddress(String name) throws IOException {
        return new NetworkAddress(name, NetworkAddress.findAddress(InetAddress::isLinkLocalAddress));
    }

    private static InterfaceAddress findAddress(Predicate<InetAddress> matcher) throws IOException {
        Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
        while (interfaces.hasMoreElements()) {
            InterfaceAddress ifAddress;
            NetworkInterface networkInterface = interfaces.nextElement();
            if (!networkInterface.isUp()) continue;
            if (Server.log.isDebugEnabled()) {
                Server.log.debugf("Network interface %s", networkInterface);
            }
            if ((ifAddress = NetworkAddress.findAddress(networkInterface, matcher)) == null) continue;
            return ifAddress;
        }
        throw new IOException("No matching addresses found");
    }

    private static InterfaceAddress findAddress(NetworkInterface networkInterface, Predicate<InetAddress> matcher) throws IOException {
        for (InterfaceAddress ifAddress : networkInterface.getInterfaceAddresses()) {
            InetAddress address = ifAddress.getAddress();
            if (Server.log.isDebugEnabled()) {
                Server.log.debugf("Network address %s (loopback=%b linklocal=%b sitelocal=%b)", new Object[]{address, address.isLoopbackAddress(), address.isLinkLocalAddress(), address.isSiteLocalAddress()});
            }
            if (!matcher.test(address)) continue;
            return ifAddress;
        }
        return null;
    }

    private static NetworkInterface findInterface(Predicate<NetworkInterface> matcher) throws IOException {
        Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
        while (interfaces.hasMoreElements()) {
            NetworkInterface networkInterface = interfaces.nextElement();
            if (!networkInterface.isUp()) continue;
            if (Server.log.isDebugEnabled()) {
                Server.log.debugf("Network interface %s", networkInterface);
            }
            if (!matcher.test(networkInterface)) continue;
            return networkInterface;
        }
        throw new IOException("No matching addresses found");
    }
}

