package cdm.observable.common.functions;

import cdm.base.math.FinancialUnitEnum;
import cdm.base.math.NonNegativeQuantitySchedule;
import cdm.base.math.QuantitySchedule;
import cdm.base.math.functions.FilterQuantityByCurrencyExists;
import cdm.base.math.functions.FilterQuantityByFinancialUnit;
import cdm.observable.asset.PriceSchedule;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.expression.MapperMaths;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import java.math.BigDecimal;
import java.util.List;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(CashPriceQuantityNoOfUnitsTriangulation.CashPriceQuantityNoOfUnitsTriangulationDefault.class)
public abstract class CashPriceQuantityNoOfUnitsTriangulation implements RosettaFunction {
	
	// RosettaFunction dependencies
	//
	@Inject protected FilterQuantityByCurrencyExists filterQuantityByCurrencyExists;
	@Inject protected FilterQuantityByFinancialUnit filterQuantityByFinancialUnit;

	/**
	* @param quantity 
	* @param price 
	* @return success 
	*/
	public Boolean evaluate(List<? extends NonNegativeQuantitySchedule> quantity, List<? extends PriceSchedule> price) {
		Boolean success = doEvaluate(quantity, price);
		
		return success;
	}

	protected abstract Boolean doEvaluate(List<? extends NonNegativeQuantitySchedule> quantity, List<? extends PriceSchedule> price);

	protected abstract Mapper<BigDecimal> notional(List<? extends NonNegativeQuantitySchedule> quantity, List<? extends PriceSchedule> price);

	protected abstract Mapper<BigDecimal> noOfUnits(List<? extends NonNegativeQuantitySchedule> quantity, List<? extends PriceSchedule> price);

	protected abstract Mapper<BigDecimal> cashPrice(List<? extends NonNegativeQuantitySchedule> quantity, List<? extends PriceSchedule> price);

	public static class CashPriceQuantityNoOfUnitsTriangulationDefault extends CashPriceQuantityNoOfUnitsTriangulation {
		@Override
		protected Boolean doEvaluate(List<? extends NonNegativeQuantitySchedule> quantity, List<? extends PriceSchedule> price) {
			Boolean success = null;
			return assignOutput(success, quantity, price);
		}
		
		protected Boolean assignOutput(Boolean success, List<? extends NonNegativeQuantitySchedule> quantity, List<? extends PriceSchedule> price) {
			success = MapperUtils.toComparisonResult(MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(cashPrice(quantity, price).get())).and(exists(MapperS.of(noOfUnits(quantity, price).get()))).and(exists(MapperS.of(notional(quantity, price).get()))).getOrDefault(false)) {
					return areEqual(MapperMaths.<BigDecimal, BigDecimal, BigDecimal>multiply(MapperS.of(cashPrice(quantity, price).get()), MapperS.of(noOfUnits(quantity, price).get())), MapperS.of(notional(quantity, price).get()), CardinalityOperator.All);
				}
				else {
					return null;
				}
			})).get();
			
			return success;
		}
		
		@Override
		protected Mapper<BigDecimal> notional(List<? extends NonNegativeQuantitySchedule> quantity, List<? extends PriceSchedule> price) {
			return MapperC.<QuantitySchedule>of(filterQuantityByCurrencyExists.evaluate(MapperC.<NonNegativeQuantitySchedule>of(quantity).getMulti()))
				.mapItem(item -> (MapperS<BigDecimal>)item.<BigDecimal>map("getValue", measureBase -> measureBase.getValue()))
				.apply(item -> distinct(item))
				.apply(item -> MapperS.of(item.get()));
		}
		
		@Override
		protected Mapper<BigDecimal> noOfUnits(List<? extends NonNegativeQuantitySchedule> quantity, List<? extends PriceSchedule> price) {
			return MapperC.<QuantitySchedule>of(filterQuantityByFinancialUnit.evaluate(MapperC.<NonNegativeQuantitySchedule>of(quantity).getMulti(), MapperS.of(FinancialUnitEnum.SHARE).get()))
				.mapItem(item -> (MapperS<BigDecimal>)item.<BigDecimal>map("getValue", measureBase -> measureBase.getValue()))
				.apply(item -> MapperS.of(item.get()));
		}
		
		@Override
		protected Mapper<BigDecimal> cashPrice(List<? extends NonNegativeQuantitySchedule> quantity, List<? extends PriceSchedule> price) {
			return MapperS.of(MapperC.<PriceSchedule>of(price).<BigDecimal>map("getValue", measureBase -> measureBase.getValue()).get());
		}
	}
}
