/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.watcher.watch;

import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.action.admin.indices.refresh.RefreshRequest;
import org.elasticsearch.action.admin.indices.refresh.RefreshResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.json.JsonXContent;
import org.elasticsearch.index.IndexNotFoundException;
import org.elasticsearch.index.engine.DocumentMissingException;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.xpack.common.stats.Counters;
import org.elasticsearch.xpack.security.InternalClient;
import org.elasticsearch.xpack.watcher.actions.ActionWrapper;
import org.elasticsearch.xpack.watcher.support.Exceptions;
import org.elasticsearch.xpack.watcher.support.init.proxy.WatcherClientProxy;
import org.elasticsearch.xpack.watcher.trigger.schedule.Schedule;
import org.elasticsearch.xpack.watcher.trigger.schedule.ScheduleTrigger;
import org.elasticsearch.xpack.watcher.watch.Watch;
import org.elasticsearch.xpack.watcher.watch.WatchStoreUtils;

public class WatchStore
extends AbstractComponent {
    public static final String INDEX = ".watches";
    public static final String DOC_TYPE = "watch";
    private final WatcherClientProxy client;
    private final Watch.Parser watchParser;
    private final ConcurrentMap<String, Watch> watches;
    private final AtomicBoolean started = new AtomicBoolean(false);
    private final int scrollSize;
    private final TimeValue scrollTimeout;

    @Inject
    public WatchStore(Settings settings, InternalClient client, Watch.Parser watchParser) {
        this(settings, new WatcherClientProxy(settings, client), watchParser);
    }

    public WatchStore(Settings settings, WatcherClientProxy client, Watch.Parser watchParser) {
        super(settings);
        this.client = client;
        this.watchParser = watchParser;
        this.watches = ConcurrentCollections.newConcurrentMap();
        this.scrollTimeout = settings.getAsTime("xpack.watcher.watch.scroll.timeout", TimeValue.timeValueSeconds((long)30L));
        this.scrollSize = settings.getAsInt("xpack.watcher.watch.scroll.size", Integer.valueOf(100));
    }

    public void start(ClusterState state) throws Exception {
        if (this.started.get()) {
            this.logger.debug("watch store already started");
            return;
        }
        try {
            IndexMetaData indexMetaData = WatchStoreUtils.getConcreteIndex(INDEX, state.metaData());
            int count = this.loadWatches(indexMetaData.getNumberOfShards());
            this.logger.debug("loaded [{}] watches from the watches index [{}]", (Object)count, (Object)indexMetaData.getIndex().getName());
        }
        catch (IndexNotFoundException indexMetaData) {
        }
        catch (Exception e) {
            this.logger.debug(() -> new ParameterizedMessage("failed to load watches for watch index [{}]", (Object)INDEX), (Throwable)e);
            this.watches.clear();
            throw e;
        }
        this.started.set(true);
    }

    public boolean validate(ClusterState state) {
        try {
            IndexMetaData indexMetaData = WatchStoreUtils.getConcreteIndex(INDEX, state.metaData());
            if (indexMetaData.getState() == IndexMetaData.State.CLOSE) {
                this.logger.debug("watch index [{}] is marked as closed, watcher cannot be started", (Object)indexMetaData.getIndex().getName());
                return false;
            }
            return state.routingTable().index(indexMetaData.getIndex()).allPrimaryShardsActive();
        }
        catch (IndexNotFoundException e) {
            return true;
        }
        catch (IllegalStateException e) {
            this.logger.trace(() -> new ParameterizedMessage("error getting index meta data [{}]: ", (Object)INDEX), (Throwable)e);
            return false;
        }
    }

    public boolean started() {
        return this.started.get();
    }

    public void stop() {
        if (this.started.compareAndSet(true, false)) {
            this.watches.clear();
            this.logger.info("stopped watch store");
        }
    }

    public Watch get(String id) {
        this.ensureStarted();
        return (Watch)this.watches.get(id);
    }

    public WatchPut put(Watch watch) throws IOException {
        this.ensureStarted();
        IndexRequest indexRequest = this.createIndexRequest(watch.id(), watch.getAsBytes(), -3L);
        IndexResponse response = this.client.index(indexRequest, (TimeValue)null);
        watch.status().version(response.getVersion());
        watch.version(response.getVersion());
        Watch previous = this.watches.put(watch.id(), watch);
        return new WatchPut(previous, watch, response);
    }

    public void updateStatus(Watch watch) throws IOException {
        this.ensureStarted();
        if (!watch.status().dirty()) {
            return;
        }
        XContentBuilder source = JsonXContent.contentBuilder().startObject().field(Watch.Field.STATUS.getPreferredName(), (ToXContent)watch.status(), ToXContent.EMPTY_PARAMS).endObject();
        UpdateRequest updateRequest = new UpdateRequest(INDEX, DOC_TYPE, watch.id());
        updateRequest.doc(source);
        updateRequest.version(watch.version());
        try {
            UpdateResponse response = this.client.update(updateRequest);
            watch.status().version(response.getVersion());
            watch.version(response.getVersion());
            watch.status().resetDirty();
        }
        catch (DocumentMissingException e) {
            this.logger.warn("Watch [{}] was deleted during watch execution, not updating watch status", (Object)watch.id());
        }
    }

    public WatchDelete delete(String id) {
        this.ensureStarted();
        Watch watch = (Watch)this.watches.remove(id);
        DeleteRequest request = new DeleteRequest(INDEX, DOC_TYPE, id);
        DeleteResponse response = this.client.delete(request);
        if (watch != null) {
            watch.version(response.getVersion());
        }
        return new WatchDelete(response);
    }

    public Collection<Watch> watches() {
        return this.watches.values();
    }

    public Collection<Watch> activeWatches() {
        HashSet<Watch> watches = new HashSet<Watch>();
        for (Watch watch : this.watches()) {
            if (!watch.status().state().isActive()) continue;
            watches.add(watch);
        }
        return watches;
    }

    public Map<String, Object> usageStats() {
        Counters counters = new Counters("count.total", "count.active");
        for (Watch watch : this.watches.values()) {
            Object type;
            boolean isActive = watch.status().state().isActive();
            this.addToCounters("count", isActive, counters);
            if (watch.trigger() != null) {
                this.addToCounters("watch.trigger._all", isActive, counters);
                if ("schedule".equals(watch.trigger().type())) {
                    Schedule schedule = ((ScheduleTrigger)watch.trigger()).getSchedule();
                    this.addToCounters("watch.trigger.schedule._all", isActive, counters);
                    this.addToCounters("watch.trigger.schedule." + schedule.type(), isActive, counters);
                }
            }
            if (watch.input() != null) {
                type = watch.input().type();
                this.addToCounters("watch.input._all", isActive, counters);
                this.addToCounters("watch.input." + (String)type, isActive, counters);
            }
            if (watch.condition() != null) {
                type = watch.condition().type();
                this.addToCounters("watch.condition._all", isActive, counters);
                this.addToCounters("watch.condition." + (String)type, isActive, counters);
            }
            for (ActionWrapper actionWrapper : watch.actions()) {
                String type2 = actionWrapper.action().type();
                this.addToCounters("watch.action." + type2, isActive, counters);
                if (actionWrapper.transform() == null) continue;
                String transformType = actionWrapper.transform().type();
                this.addToCounters("watch.transform." + transformType, isActive, counters);
            }
            if (watch.transform() != null) {
                type = watch.transform().type();
                this.addToCounters("watch.transform." + (String)type, isActive, counters);
            }
            if (watch.metadata() == null || watch.metadata().size() <= 0) continue;
            this.addToCounters("watch.metadata", isActive, counters);
        }
        return counters.toMap();
    }

    private void addToCounters(String name, boolean isActive, Counters counters) {
        counters.inc(name + ".total");
        if (isActive) {
            counters.inc(name + ".active");
        }
    }

    IndexRequest createIndexRequest(String id, BytesReference source, long version) {
        IndexRequest indexRequest = new IndexRequest(INDEX, DOC_TYPE, id);
        indexRequest.source(BytesReference.toBytes((BytesReference)source));
        indexRequest.version(version);
        return indexRequest;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int loadWatches(int numPrimaryShards) {
        assert (this.watches.isEmpty()) : "no watches should reside, but there are [" + this.watches.size() + "] watches.";
        RefreshResponse refreshResponse = this.client.refresh(new RefreshRequest(new String[]{INDEX}));
        if (refreshResponse.getSuccessfulShards() < numPrimaryShards) {
            throw Exceptions.illegalState("not all required shards have been refreshed", new Object[0]);
        }
        int count = 0;
        SearchRequest searchRequest = new SearchRequest(new String[]{INDEX}).types(new String[]{DOC_TYPE}).preference("_primary").scroll(this.scrollTimeout).source(new SearchSourceBuilder().size(this.scrollSize).sort((SortBuilder)SortBuilders.fieldSort((String)"_doc")).version(Boolean.valueOf(true)));
        SearchResponse response = this.client.search(searchRequest, null);
        try {
            if (response.getTotalShards() != response.getSuccessfulShards()) {
                throw new ElasticsearchException("Partial response while loading watches", new Object[0]);
            }
            while (response.getHits().hits().length != 0) {
                for (SearchHit hit : response.getHits()) {
                    String id = hit.getId();
                    try {
                        Watch watch = this.watchParser.parse(id, true, hit.getSourceRef(), true);
                        watch.status().version(hit.version());
                        watch.version(hit.version());
                        this.watches.put(id, watch);
                        ++count;
                    }
                    catch (Exception e) {
                        this.logger.error(() -> new ParameterizedMessage("couldn't load watch [{}], ignoring it...", (Object)id), (Throwable)e);
                    }
                }
                response = this.client.searchScroll(response.getScrollId(), this.scrollTimeout);
            }
        }
        finally {
            this.client.clearScroll(response.getScrollId());
        }
        return count;
    }

    private void ensureStarted() {
        if (!this.started.get()) {
            throw new IllegalStateException("watch store not started");
        }
    }

    public void clearWatchesInMemory() {
        this.watches.clear();
    }

    public class WatchDelete {
        private final DeleteResponse response;

        public WatchDelete(DeleteResponse response) {
            this.response = response;
        }

        public DeleteResponse deleteResponse() {
            return this.response;
        }
    }

    public class WatchPut {
        private final Watch previous;
        private final Watch current;
        private final IndexResponse response;

        public WatchPut(Watch previous, Watch current, IndexResponse response) {
            this.current = current;
            this.previous = previous;
            this.response = response;
        }

        public Watch current() {
            return this.current;
        }

        public Watch previous() {
            return this.previous;
        }

        public IndexResponse indexResponse() {
            return this.response;
        }
    }
}

