/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.metrics;

import com.codahale.metrics.Clock;
import com.codahale.metrics.Counter;
import com.codahale.metrics.Gauge;
import com.codahale.metrics.Histogram;
import com.codahale.metrics.Meter;
import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricFilter;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.ScheduledReporter;
import com.codahale.metrics.Timer;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.module.afterburner.AfterburnerModule;
import java.io.IOException;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.elasticsearch.metrics.JsonMetrics;
import org.elasticsearch.metrics.MetricsElasticsearchModule;
import org.elasticsearch.metrics.percolation.Notifier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ElasticsearchReporter
extends ScheduledReporter {
    private static final Logger LOGGER = LoggerFactory.getLogger(ElasticsearchReporter.class);
    private final String[] hosts;
    private final Clock clock;
    private final String prefix;
    private final String index;
    private final int bulkSize;
    private final int timeout;
    private final ObjectMapper objectMapper = new ObjectMapper();
    private final ObjectWriter writer;
    private MetricFilter percolationFilter;
    private Notifier notifier;
    private String currentIndexName;
    private SimpleDateFormat indexDateFormat = null;
    private boolean checkedForIndexTemplate = false;

    public static Builder forRegistry(MetricRegistry registry) {
        return new Builder(registry);
    }

    public ElasticsearchReporter(MetricRegistry registry, String[] hosts, int timeout, String index, String indexDateFormat, int bulkSize, Clock clock, String prefix, TimeUnit rateUnit, TimeUnit durationUnit, MetricFilter filter, MetricFilter percolationFilter, Notifier percolationNotifier, String timestampFieldname, Map<String, Object> additionalFields) throws MalformedURLException {
        super(registry, "elasticsearch-reporter", filter, rateUnit, durationUnit);
        this.hosts = hosts;
        this.index = index;
        this.bulkSize = bulkSize;
        this.clock = clock;
        this.prefix = prefix;
        this.timeout = timeout;
        if (indexDateFormat != null && indexDateFormat.length() > 0) {
            this.indexDateFormat = new SimpleDateFormat(indexDateFormat);
        }
        if (percolationNotifier != null && percolationFilter != null) {
            this.percolationFilter = percolationFilter;
            this.notifier = percolationNotifier;
        }
        if (timestampFieldname == null || timestampFieldname.trim().length() == 0) {
            LOGGER.error("Timestampfieldname {}\u00a0is not valid, using default @timestamp", (Object)timestampFieldname);
            timestampFieldname = "@timestamp";
        }
        this.objectMapper.configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false);
        this.objectMapper.configure(SerializationFeature.CLOSE_CLOSEABLE, false);
        this.objectMapper.configure(JsonGenerator.Feature.AUTO_CLOSE_JSON_CONTENT, false);
        this.objectMapper.configure(JsonGenerator.Feature.AUTO_CLOSE_TARGET, false);
        this.objectMapper.registerModule((Module)new AfterburnerModule());
        this.objectMapper.registerModule((Module)new MetricsElasticsearchModule(rateUnit, durationUnit, timestampFieldname, additionalFields));
        this.writer = this.objectMapper.writer();
        this.checkForIndexTemplate();
    }

    public void report(SortedMap<String, Gauge> gauges, SortedMap<String, Counter> counters, SortedMap<String, Histogram> histograms, SortedMap<String, Meter> meters, SortedMap<String, Timer> timers) {
        if (gauges.isEmpty() && counters.isEmpty() && histograms.isEmpty() && meters.isEmpty() && timers.isEmpty()) {
            LOGGER.info("All metrics empty, nothing to report");
            return;
        }
        if (!this.checkedForIndexTemplate) {
            this.checkForIndexTemplate();
        }
        long timestamp = this.clock.getTime() / 1000L;
        this.currentIndexName = this.index;
        if (this.indexDateFormat != null) {
            this.currentIndexName = this.currentIndexName + "-" + this.indexDateFormat.format(new Date(timestamp * 1000L));
        }
        try {
            JsonMetrics.JsonMetric jsonMetric;
            HttpURLConnection connection = this.openConnection("/_bulk", "POST");
            if (connection == null) {
                LOGGER.error("Could not connect to any configured elasticsearch instances: {}", Arrays.asList(this.hosts));
                return;
            }
            ArrayList<JsonMetrics.JsonMetric> percolationMetrics = new ArrayList<JsonMetrics.JsonMetric>();
            AtomicInteger entriesWritten = new AtomicInteger(0);
            for (Map.Entry<String, Gauge> entry : gauges.entrySet()) {
                if (entry.getValue().getValue() == null) continue;
                jsonMetric = new JsonMetrics.JsonGauge(MetricRegistry.name((String)this.prefix, (String[])new String[]{entry.getKey()}), timestamp, entry.getValue());
                connection = this.writeJsonMetricAndRecreateConnectionIfNeeded(jsonMetric, connection, entriesWritten);
                this.addJsonMetricToPercolationIfMatching(jsonMetric, percolationMetrics);
            }
            for (Map.Entry<String, Gauge> entry : counters.entrySet()) {
                jsonMetric = new JsonMetrics.JsonCounter(MetricRegistry.name((String)this.prefix, (String[])new String[]{entry.getKey()}), timestamp, (Counter)entry.getValue());
                connection = this.writeJsonMetricAndRecreateConnectionIfNeeded(jsonMetric, connection, entriesWritten);
                this.addJsonMetricToPercolationIfMatching(jsonMetric, percolationMetrics);
            }
            for (Map.Entry<String, Gauge> entry : histograms.entrySet()) {
                jsonMetric = new JsonMetrics.JsonHistogram(MetricRegistry.name((String)this.prefix, (String[])new String[]{entry.getKey()}), timestamp, (Histogram)entry.getValue());
                connection = this.writeJsonMetricAndRecreateConnectionIfNeeded(jsonMetric, connection, entriesWritten);
                this.addJsonMetricToPercolationIfMatching(jsonMetric, percolationMetrics);
            }
            for (Map.Entry<String, Gauge> entry : meters.entrySet()) {
                jsonMetric = new JsonMetrics.JsonMeter(MetricRegistry.name((String)this.prefix, (String[])new String[]{entry.getKey()}), timestamp, (Meter)entry.getValue());
                connection = this.writeJsonMetricAndRecreateConnectionIfNeeded(jsonMetric, connection, entriesWritten);
                this.addJsonMetricToPercolationIfMatching(jsonMetric, percolationMetrics);
            }
            for (Map.Entry<String, Gauge> entry : timers.entrySet()) {
                jsonMetric = new JsonMetrics.JsonTimer(MetricRegistry.name((String)this.prefix, (String[])new String[]{entry.getKey()}), timestamp, (Timer)entry.getValue());
                connection = this.writeJsonMetricAndRecreateConnectionIfNeeded(jsonMetric, connection, entriesWritten);
                this.addJsonMetricToPercolationIfMatching(jsonMetric, percolationMetrics);
            }
            this.closeConnection(connection);
            if (percolationMetrics.size() > 0 && this.notifier != null) {
                for (JsonMetrics.JsonMetric jsonMetric2 : percolationMetrics) {
                    List<String> matches = this.getPercolationMatches(jsonMetric2);
                    for (String match : matches) {
                        this.notifier.notify(jsonMetric2, match);
                    }
                }
            }
        }
        catch (IOException e) {
            LOGGER.error("Couldnt report to elasticsearch server", (Throwable)e);
        }
    }

    private List<String> getPercolationMatches(JsonMetrics.JsonMetric jsonMetric) throws IOException {
        HttpURLConnection connection = this.openConnection("/" + this.currentIndexName + "/" + jsonMetric.type() + "/_percolate", "POST");
        if (connection == null) {
            LOGGER.error("Could not connect to any configured elasticsearch instances for percolation: {}", Arrays.asList(this.hosts));
            return Collections.emptyList();
        }
        HashMap<String, JsonMetrics.JsonMetric> data = new HashMap<String, JsonMetrics.JsonMetric>(1);
        data.put("doc", jsonMetric);
        this.objectMapper.writeValue(connection.getOutputStream(), data);
        this.closeConnection(connection);
        if (connection.getResponseCode() != 200) {
            throw new RuntimeException("Error percolating " + jsonMetric);
        }
        Map input = (Map)this.objectMapper.readValue(connection.getInputStream(), (TypeReference)new TypeReference<Map<String, Object>>(){});
        ArrayList<String> matches = new ArrayList<String>();
        if (input.containsKey("matches") && input.get("matches") instanceof List) {
            List foundMatches = (List)input.get("matches");
            for (Map entry : foundMatches) {
                if (!entry.containsKey("_id")) continue;
                matches.add((String)entry.get("_id"));
            }
        }
        return matches;
    }

    private void addJsonMetricToPercolationIfMatching(JsonMetrics.JsonMetric<? extends Metric> jsonMetric, List<JsonMetrics.JsonMetric> percolationMetrics) {
        if (this.percolationFilter != null && this.percolationFilter.matches(jsonMetric.name(), jsonMetric.value())) {
            percolationMetrics.add(jsonMetric);
        }
    }

    private HttpURLConnection writeJsonMetricAndRecreateConnectionIfNeeded(JsonMetrics.JsonMetric jsonMetric, HttpURLConnection connection, AtomicInteger entriesWritten) throws IOException {
        this.writeJsonMetric(jsonMetric, this.writer, connection.getOutputStream());
        return this.createNewConnectionIfBulkSizeReached(connection, entriesWritten.incrementAndGet());
    }

    private void closeConnection(HttpURLConnection connection) throws IOException {
        connection.getOutputStream().close();
        connection.disconnect();
        if (connection.getResponseCode() != 200) {
            LOGGER.error("Reporting returned code {} {}: {}", (Object)connection.getResponseCode(), (Object)connection.getResponseMessage());
        }
    }

    private HttpURLConnection createNewConnectionIfBulkSizeReached(HttpURLConnection connection, int entriesWritten) throws IOException {
        if (entriesWritten % this.bulkSize == 0) {
            this.closeConnection(connection);
            return this.openConnection("/_bulk", "POST");
        }
        return connection;
    }

    private void writeJsonMetric(JsonMetrics.JsonMetric jsonMetric, ObjectWriter writer, OutputStream out) throws IOException {
        writer.writeValue(out, (Object)new MetricsElasticsearchModule.BulkIndexOperationHeader(this.currentIndexName, jsonMetric.type()));
        out.write("\n".getBytes());
        writer.writeValue(out, (Object)jsonMetric);
        out.write("\n".getBytes());
        out.flush();
    }

    private HttpURLConnection openConnection(String uri, String method) {
        for (String host : this.hosts) {
            try {
                URL templateUrl = new URL("http://" + host + uri);
                HttpURLConnection connection = (HttpURLConnection)templateUrl.openConnection();
                connection.setRequestMethod(method);
                connection.setConnectTimeout(this.timeout);
                connection.setUseCaches(false);
                if (method.equalsIgnoreCase("POST") || method.equalsIgnoreCase("PUT")) {
                    connection.setDoOutput(true);
                }
                connection.connect();
                return connection;
            }
            catch (IOException e) {
                LOGGER.error("Error connecting to {}: {}", (Object)host, (Object)e);
            }
        }
        return null;
    }

    private void checkForIndexTemplate() {
        try {
            boolean isTemplateMissing;
            HttpURLConnection connection = this.openConnection("/_template/metrics_template", "HEAD");
            if (connection == null) {
                LOGGER.error("Could not connect to any configured elasticsearch instances: {}", Arrays.asList(this.hosts));
                return;
            }
            connection.disconnect();
            boolean bl = isTemplateMissing = connection.getResponseCode() == 404;
            if (isTemplateMissing) {
                LOGGER.debug("No metrics template found in elasticsearch. Adding...");
                HttpURLConnection putTemplateConnection = this.openConnection("/_template/metrics_template", "PUT");
                if (putTemplateConnection == null) {
                    LOGGER.error("Error adding metrics template to elasticsearch");
                    return;
                }
                JsonGenerator json = new JsonFactory().createGenerator(putTemplateConnection.getOutputStream());
                json.writeStartObject();
                json.writeStringField("template", this.index + "*");
                json.writeObjectFieldStart("mappings");
                json.writeObjectFieldStart("_default_");
                json.writeObjectFieldStart("_all");
                json.writeBooleanField("enabled", false);
                json.writeEndObject();
                json.writeObjectFieldStart("properties");
                json.writeObjectFieldStart("name");
                json.writeObjectField("type", (Object)"string");
                json.writeObjectField("index", (Object)"not_analyzed");
                json.writeEndObject();
                json.writeEndObject();
                json.writeEndObject();
                json.writeEndObject();
                json.writeEndObject();
                json.flush();
                putTemplateConnection.disconnect();
                if (putTemplateConnection.getResponseCode() != 200) {
                    LOGGER.error("Error adding metrics template to elasticsearch: {}/{}" + putTemplateConnection.getResponseCode(), (Object)putTemplateConnection.getResponseMessage());
                }
            }
            this.checkedForIndexTemplate = true;
        }
        catch (IOException e) {
            LOGGER.error("Error when checking/adding metrics template to elasticsearch", (Throwable)e);
        }
    }

    public static class Builder {
        private final MetricRegistry registry;
        private Clock clock;
        private String prefix;
        private TimeUnit rateUnit;
        private TimeUnit durationUnit;
        private MetricFilter filter;
        private String[] hosts = new String[]{"localhost:9200"};
        private String index = "metrics";
        private String indexDateFormat = "yyyy-MM";
        private int bulkSize = 2500;
        private Notifier percolationNotifier;
        private MetricFilter percolationFilter;
        private int timeout = 1000;
        private String timestampFieldname = "@timestamp";
        private Map<String, Object> additionalFields;

        private Builder(MetricRegistry registry) {
            this.registry = registry;
            this.clock = Clock.defaultClock();
            this.prefix = null;
            this.rateUnit = TimeUnit.SECONDS;
            this.durationUnit = TimeUnit.MILLISECONDS;
            this.filter = MetricFilter.ALL;
        }

        public Builder withClock(Clock clock) {
            this.clock = clock;
            return this;
        }

        public Builder prefixedWith(String prefix) {
            this.prefix = prefix;
            return this;
        }

        public Builder convertRatesTo(TimeUnit rateUnit) {
            this.rateUnit = rateUnit;
            return this;
        }

        public Builder convertDurationsTo(TimeUnit durationUnit) {
            this.durationUnit = durationUnit;
            return this;
        }

        public Builder filter(MetricFilter filter) {
            this.filter = filter;
            return this;
        }

        public Builder hosts(String ... hosts) {
            this.hosts = hosts;
            return this;
        }

        public Builder timeout(int timeout) {
            this.timeout = timeout;
            return this;
        }

        public Builder index(String index) {
            this.index = index;
            return this;
        }

        public Builder indexDateFormat(String indexDateFormat) {
            this.indexDateFormat = indexDateFormat;
            return this;
        }

        public Builder bulkSize(int bulkSize) {
            this.bulkSize = bulkSize;
            return this;
        }

        public Builder percolationFilter(MetricFilter percolationFilter) {
            this.percolationFilter = percolationFilter;
            return this;
        }

        public Builder percolationNotifier(Notifier notifier) {
            this.percolationNotifier = notifier;
            return this;
        }

        public Builder timestampFieldname(String fieldName) {
            this.timestampFieldname = fieldName;
            return this;
        }

        public Builder additionalFields(Map<String, Object> additionalFields) {
            this.additionalFields = additionalFields;
            return this;
        }

        public ElasticsearchReporter build() throws IOException {
            return new ElasticsearchReporter(this.registry, this.hosts, this.timeout, this.index, this.indexDateFormat, this.bulkSize, this.clock, this.prefix, this.rateUnit, this.durationUnit, this.filter, this.percolationFilter, this.percolationNotifier, this.timestampFieldname, this.additionalFields);
        }
    }
}

