package dyvil.collection.impl;

import dyvil.annotation.internal.NonNull;
import dyvil.annotation.internal.Nullable;
import dyvil.collection.Entry;
import dyvil.collection.ImmutableMap;
import dyvil.collection.Map;
import dyvil.collection.MutableMap;
import dyvil.util.Option;
import dyvil.util.Some;

import java.io.IOException;
import java.util.*;
import java.util.function.Consumer;

public abstract class AbstractTreeMap<K, V> implements Map<K, V>
{
	protected static final class TreeEntry<K, V> implements dyvil.collection.Entry<K, V>
	{
		private static final long serialVersionUID = -8592912850607607269L;

		public transient K key;
		public transient V value;
		@Nullable
		protected transient TreeEntry<K, V> left  = null;
		@Nullable
		protected transient TreeEntry<K, V> right = null;
		@Nullable
		protected transient TreeEntry<K, V> parent;
		protected transient boolean color = BLACK;

		protected TreeEntry(K key, V value, TreeEntry<K, V> parent)
		{
			this.key = key;
			this.value = value;
			this.parent = parent;
		}

		@Override
		public K getKey()
		{
			return this.key;
		}

		@Override
		public V getValue()
		{
			return this.value;
		}

		@Override
		public boolean equals(Object o)
		{
			return Entry.entryEquals(this, o);
		}

		@Override
		public int hashCode()
		{
			return Entry.entryHashCode(this);
		}

		@NonNull
		@Override
		public String toString()
		{
			return this.key + " -> " + this.value;
		}

		private void writeObject(java.io.@NonNull ObjectOutputStream out) throws IOException
		{
			out.defaultWriteObject();

			out.writeObject(this.key);
			out.writeObject(this.value);
			out.writeObject(this.left);
			out.writeObject(this.right);
			out.writeObject(this.parent);
			out.writeBoolean(this.color);
		}

		private void readObject(java.io.@NonNull ObjectInputStream in) throws IOException, ClassNotFoundException
		{
			in.defaultReadObject();

			this.key = (K) in.readObject();
			this.value = (V) in.readObject();
			this.left = (TreeEntry<K, V>) in.readObject();
			this.right = (TreeEntry<K, V>) in.readObject();
			this.parent = (TreeEntry<K, V>) in.readObject();
			this.color = in.readBoolean();
		}
	}

	protected abstract class TreeEntryIterator<T> implements Iterator<T>
	{
		@Nullable TreeEntry<K, V> next;
		@Nullable TreeEntry<K, V> lastReturned;

		protected TreeEntryIterator(TreeEntry<K, V> first)
		{
			this.lastReturned = null;
			this.next = first;
		}

		@Override
		public final boolean hasNext()
		{
			return this.next != null;
		}

		@Nullable
		protected final TreeEntry<K, V> nextEntry()
		{
			TreeEntry<K, V> e = this.next;
			if (e == null)
			{
				throw new NoSuchElementException();
			}
			this.next = successor(e);
			this.lastReturned = e;
			return e;
		}

		@Nullable
		protected final TreeEntry<K, V> prevEntry()
		{
			TreeEntry<K, V> e = this.next;
			if (e == null)
			{
				throw new NoSuchElementException();
			}
			this.next = predecessor(e);
			this.lastReturned = e;
			return e;
		}

		@Override
		public void remove()
		{
			if (this.lastReturned == null)
			{
				throw new IllegalStateException();
			}

			if (this.lastReturned.left != null && this.lastReturned.right != null)
			{
				this.next = this.lastReturned;
			}
			AbstractTreeMap.this.deleteEntry(this.lastReturned);
			this.lastReturned = null;
		}
	}

	private static final long serialVersionUID = 4299609156116845922L;

	private static final boolean RED   = false;
	private static final boolean BLACK = true;

	@Nullable
	protected transient final Comparator<? super K> comparator;
	@Nullable
	protected transient       TreeEntry<K, V>       root;
	protected transient       int                   size;

	// Constructors

	public AbstractTreeMap()
	{
		this.comparator = null;
	}

	public AbstractTreeMap(Comparator<? super K> comparator)
	{
		this.comparator = comparator;
	}

	public AbstractTreeMap(Entry<? extends K, ? extends V> @NonNull [] entries)
	{
		this(entries, null);
	}

	public AbstractTreeMap(Entry<? extends K, ? extends V> @NonNull [] entries, Comparator<? super K> comparator)
	{
		this.comparator = comparator;
		for (Entry<? extends K, ? extends V> entry : entries)
		{
			this.putInternal(entry.getKey(), entry.getValue());
		}
	}

	public AbstractTreeMap(@NonNull Iterable<? extends @NonNull Entry<? extends K, ? extends V>> map)
	{
		this(map, null);
	}

	public AbstractTreeMap(@NonNull Iterable<? extends @NonNull Entry<? extends K, ? extends V>> map,
		                      Comparator<? super K> comparator)
	{
		this.comparator = comparator;
		this.putAllInternal(map);
	}

	public AbstractTreeMap(@NonNull AbstractTreeMap<? extends K, ? extends V> treeMap)
	{
		this(treeMap, null);
	}

	public AbstractTreeMap(@NonNull AbstractTreeMap<? extends K, ? extends V> treeMap, Comparator<? super K> comparator)
	{
		this.comparator = comparator;

		if (treeMap.comparator == comparator)
		{
			this.buildFromSorted(treeMap.size(), treeMap.iterator());
			return;
		}

		this.putAllInternal(treeMap);
	}

	// Implementation Methods

	@Nullable
	public Comparator<? super K> comparator()
	{
		return this.comparator;
	}

	@SuppressWarnings("unchecked")
	protected static int compare(@Nullable Comparator comparator, @NonNull Object k1, @NonNull Object k2)
	{
		return comparator == null ? ((Comparable) k1).compareTo(k2) : comparator.compare(k1, k2);
	}

	@Override
	public int size()
	{
		return this.size;
	}

	@Override
	public boolean isSorted()
	{
		return this.comparator == null || Map.super.isSorted();
	}

	@Override
	public boolean isSorted(@NonNull Comparator<? super K> comparator)
	{
		return comparator == this.comparator || Map.super.isSorted(comparator);
	}

	@NonNull
	@Override
	public Iterator<Entry<K, V>> iterator()
	{
		return new TreeEntryIterator<Entry<K, V>>(this.getFirstEntry())
		{
			@Nullable
			@Override
			public Entry<K, V> next()
			{
				return this.nextEntry();
			}
		};
	}

	@NonNull
	@Override
	public Spliterator<Entry<K, V>> spliterator()
	{
		return new EntrySpliterator<>(this, null, null, 0, -1);
	}

	@NonNull
	@Override
	public Iterator<K> keyIterator()
	{
		return new TreeEntryIterator<K>(this.getFirstEntry())
		{
			@Override
			public K next()
			{
				return this.nextEntry().key;
			}
		};
	}

	@NonNull
	@Override
	public final Spliterator<K> keySpliterator()
	{
		return new KeySpliterator<>(this, null, null, 0, -1);
	}

	@NonNull
	@Override
	public Iterator<V> valueIterator()
	{
		return new TreeEntryIterator<V>(this.getFirstEntry())
		{
			@Override
			public V next()
			{
				return this.nextEntry().value;
			}
		};
	}

	@NonNull
	@Override
	public Spliterator<V> valueSpliterator()
	{
		return new ValueSpliterator<>(this, null, null, 0, -1);
	}

	static class TreeMapSpliterator<K, V>
	{
		final     AbstractTreeMap<K, V> tree;
		@Nullable TreeEntry<K, V>       current;
		TreeEntry<K, V> fence;
		int             side;
		int             est;

		TreeMapSpliterator(AbstractTreeMap<K, V> tree, TreeEntry<K, V> origin, TreeEntry<K, V> fence, int side, int est)
		{
			this.tree = tree;
			this.current = origin;
			this.fence = fence;
			this.side = side;
			this.est = est;
		}

		final int getEstimate()
		{
			int s;
			AbstractTreeMap<K, V> t;
			if ((s = this.est) < 0)
			{
				if ((t = this.tree) != null)
				{
					this.current = s == -1 ? t.getFirstEntry() : t.getLastEntry();
					s = this.est = t.size;
				}
				else
				{
					s = this.est = 0;
				}
			}
			return s;
		}

		public final long estimateSize()
		{
			return this.getEstimate();
		}
	}

	static final class KeySpliterator<K, V> extends TreeMapSpliterator<K, V> implements Spliterator<K>
	{
		KeySpliterator(AbstractTreeMap<K, V> tree, TreeEntry<K, V> origin, TreeEntry<K, V> fence, int side, int est)
		{
			super(tree, origin, fence, side, est);
		}

		@Override
		public KeySpliterator<K, V> trySplit()
		{
			if (this.est < 0)
			{
				this.getEstimate();
			}
			int d = this.side;
			TreeEntry<K, V> e = this.current, f = this.fence, s = e == null || e == f ?
				                                                      null :
				                                                      d == 0 ?
					                                                      this.tree.root :
					                                                      d > 0 ?
						                                                      e.right :
						                                                      d < 0 && f != null ? f.left : null;
			if (s != null && s != e && s != f && compare(this.tree.comparator, e.key, s.key) < 0)
			{
				this.side = 1;
				return new KeySpliterator<>(this.tree, e, this.current = s, -1, this.est >>>= 1);
			}
			return null;
		}

		@Override
		public void forEachRemaining(@Nullable Consumer<? super K> action)
		{
			if (action == null)
			{
				throw new NullPointerException();
			}
			if (this.est < 0)
			{
				this.getEstimate();
			}
			TreeEntry<K, V> f = this.fence, e, p, pl;
			if ((e = this.current) != null && e != f)
			{
				this.current = f;
				do
				{
					action.accept(e.key);
					if ((p = e.right) != null)
					{
						while ((pl = p.left) != null)
						{
							p = pl;
						}
					}
					else
					{
						while ((p = e.parent) != null && e == p.right)
						{
							e = p;
						}
					}
				}
				while ((e = p) != null && e != f);
			}
		}

		@Override
		public boolean tryAdvance(@Nullable Consumer<? super K> action)
		{
			TreeEntry<K, V> e;
			if (action == null)
			{
				throw new NullPointerException();
			}
			if (this.est < 0)
			{
				this.getEstimate();
			}
			if ((e = this.current) == null || e == this.fence)
			{
				return false;
			}
			this.current = successor(e);
			action.accept(e.key);
			return true;
		}

		@Override
		public int characteristics()
		{
			return (this.side == 0 ? Spliterator.SIZED : 0) | Spliterator.DISTINCT | Spliterator.SORTED
				       | Spliterator.ORDERED;
		}

		@Nullable
		@Override
		public final Comparator<? super K> getComparator()
		{
			return this.tree.comparator;
		}
	}

	static final class ValueSpliterator<K, V> extends TreeMapSpliterator<K, V> implements Spliterator<V>
	{
		ValueSpliterator(AbstractTreeMap<K, V> tree, TreeEntry<K, V> origin, TreeEntry<K, V> fence, int side, int est)
		{
			super(tree, origin, fence, side, est);
		}

		@Override
		public ValueSpliterator<K, V> trySplit()
		{
			if (this.est < 0)
			{
				this.getEstimate();
			}
			int d = this.side;
			TreeEntry<K, V> e = this.current, f = this.fence, s = e == null || e == f ?
				                                                      null :
				                                                      d == 0 ?
					                                                      this.tree.root :
					                                                      d > 0 ?
						                                                      e.right :
						                                                      d < 0 && f != null ? f.left : null;
			if (s != null && s != e && s != f && compare(this.tree.comparator, e.key, s.key) < 0)
			{
				this.side = 1;
				return new ValueSpliterator<>(this.tree, e, this.current = s, -1, this.est >>>= 1);
			}
			return null;
		}

		@Override
		public void forEachRemaining(@Nullable Consumer<? super V> action)
		{
			if (action == null)
			{
				throw new NullPointerException();
			}
			if (this.est < 0)
			{
				this.getEstimate();
			}
			TreeEntry<K, V> f = this.fence, e, p, pl;
			if ((e = this.current) != null && e != f)
			{
				this.current = f;
				do
				{
					action.accept(e.value);
					if ((p = e.right) != null)
					{
						while ((pl = p.left) != null)
						{
							p = pl;
						}
					}
					else
					{
						while ((p = e.parent) != null && e == p.right)
						{
							e = p;
						}
					}
				}
				while ((e = p) != null && e != f);
			}
		}

		@Override
		public boolean tryAdvance(@Nullable Consumer<? super V> action)
		{
			TreeEntry<K, V> e;
			if (action == null)
			{
				throw new NullPointerException();
			}
			if (this.est < 0)
			{
				this.getEstimate();
			}
			if ((e = this.current) == null || e == this.fence)
			{
				return false;
			}
			this.current = successor(e);
			action.accept(e.value);
			return true;
		}

		@Override
		public int characteristics()
		{
			return (this.side == 0 ? Spliterator.SIZED : 0) | Spliterator.ORDERED;
		}
	}

	static final class EntrySpliterator<K, V> extends TreeMapSpliterator<K, V> implements Spliterator<Entry<K, V>>
	{
		EntrySpliterator(AbstractTreeMap<K, V> tree, TreeEntry<K, V> origin, TreeEntry<K, V> fence, int side, int est)
		{
			super(tree, origin, fence, side, est);
		}

		@Override
		public EntrySpliterator<K, V> trySplit()
		{
			if (this.est < 0)
			{
				this.getEstimate();
			}
			int d = this.side;
			TreeEntry<K, V> e = this.current, f = this.fence, s = e == null || e == f ?
				                                                      null :
				                                                      d == 0 ?
					                                                      this.tree.root :
					                                                      d > 0 ?
						                                                      e.right :
						                                                      d < 0 && f != null ? f.left : null;
			if (s != null && s != e && s != f && compare(this.tree.comparator, e.key, s.key) < 0)
			{
				this.side = 1;
				return new EntrySpliterator<>(this.tree, e, this.current = s, -1, this.est >>>= 1);
			}
			return null;
		}

		@Override
		public void forEachRemaining(@Nullable Consumer<? super Entry<K, V>> action)
		{
			if (action == null)
			{
				throw new NullPointerException();
			}
			if (this.est < 0)
			{
				this.getEstimate();
			}
			TreeEntry<K, V> f = this.fence, e, p, pl;
			if ((e = this.current) != null && e != f)
			{
				this.current = f;
				do
				{
					action.accept(e);
					if ((p = e.right) != null)
					{
						while ((pl = p.left) != null)
						{
							p = pl;
						}
					}
					else
					{
						while ((p = e.parent) != null && e == p.right)
						{
							e = p;
						}
					}
				}
				while ((e = p) != null && e != f);
			}
		}

		@Override
		public boolean tryAdvance(@Nullable Consumer<? super Entry<K, V>> action)
		{
			TreeEntry<K, V> e;
			if (action == null)
			{
				throw new NullPointerException();
			}
			if (this.est < 0)
			{
				this.getEstimate();
			}
			if ((e = this.current) == null || e == this.fence)
			{
				return false;
			}
			this.current = successor(e);
			action.accept(e);
			return true;
		}

		@Override
		public int characteristics()
		{
			return (this.side == 0 ? Spliterator.SIZED : 0) | Spliterator.DISTINCT | Spliterator.SORTED
				       | Spliterator.ORDERED;
		}

		@NonNull
		@Override
		@SuppressWarnings("unchecked")
		public Comparator<Entry<K, V>> getComparator()
		{
			if (this.tree.comparator != null)
			{
				return Entry.comparingByKey(this.tree.comparator);
			}
			return (Comparator) Entry.<Comparable, V>comparingByKey();
		}
	}

	@Override
	public boolean containsKey(Object key)
	{
		return this.getEntryInternal(key) != null;
	}

	@Override
	public boolean containsValue(Object value)
	{
		for (TreeEntry<K, V> e = this.getFirstEntry(); e != null; e = successor(e))
		{
			if (Objects.equals(value, e.value))
			{
				return true;
			}
		}
		return false;
	}

	@Override
	public boolean contains(Object key, Object value)
	{
		final TreeEntry<K, V> entry = this.getEntryInternal(key);
		return entry != null && Objects.equals(value, entry.value);
	}

	@Override
	public @Nullable V get(Object key)
	{
		final TreeEntry<K, V> entry = this.getEntryInternal(key);
		return entry == null ? null : entry.value;
	}

	@Nullable
	@Override
	public Entry<K, V> getEntry(Object key)
	{
		return this.getEntryInternal(key);
	}

	@NonNull
	@Override
	public Option<V> getOption(Object key)
	{
		TreeEntry<K, V> p = this.getEntryInternal(key);
		return p == null ? Option.apply() : new Some<>(p.value);
	}

	protected final V putInternal(@Nullable K key, V value)
	{
		TreeEntry<K, V> t = this.root;
		if (t == null)
		{
			this.root = new TreeEntry<>(key, value, null);
			this.size = 1;
			return null;
		}
		int cmp;
		TreeEntry<K, V> parent;

		Comparator<? super K> cpr = this.comparator;
		if (cpr != null)
		{
			do
			{
				parent = t;
				cmp = cpr.compare(key, t.key);
				if (cmp < 0)
				{
					t = t.left;
				}
				else if (cmp > 0)
				{
					t = t.right;
				}
				else
				{
					V oldValue = t.value;
					t.value = value;
					return oldValue;
				}
			}
			while (t != null);
		}
		else
		{
			if (key == null)
			{
				throw new NullPointerException();
			}

			Comparable<? super K> k = (Comparable<? super K>) key;
			do
			{
				parent = t;
				cmp = k.compareTo(t.key);
				if (cmp < 0)
				{
					t = t.left;
				}
				else if (cmp > 0)
				{
					t = t.right;
				}
				else
				{
					V oldValue = t.value;
					t.value = value;
					return oldValue;
				}
			}
			while (t != null);
		}

		TreeEntry<K, V> e = new TreeEntry<>(key, value, parent);
		if (cmp < 0)
		{
			parent.left = e;
		}
		else
		{
			parent.right = e;
		}
		this.fixAfterInsertion(e);
		this.size++;
		return null;
	}

	private void putAllInternal(@NonNull Iterable<? extends @NonNull Entry<? extends K, ? extends V>> iterable)
	{
		for (Entry<? extends K, ? extends V> entry : iterable)
		{
			this.putInternal(entry.getKey(), entry.getValue());
		}
	}

	protected final TreeEntry<K, V> getEntryInternal(@Nullable Object key)
	{
		if (this.comparator != null)
		{
			K k = (K) key;
			TreeEntry<K, V> p = this.root;
			while (p != null)
			{
				int cmp = this.comparator.compare(k, p.key);
				if (cmp < 0)
				{
					p = p.left;
				}
				else if (cmp > 0)
				{
					p = p.right;
				}
				else
				{
					return p;
				}
			}
			return null;
		}

		if (key == null)
		{
			throw new NullPointerException();
		}

		Comparable<? super K> k = (Comparable<? super K>) key;
		TreeEntry<K, V> p = this.root;
		while (p != null)
		{
			int cmp = k.compareTo(p.key);
			if (cmp < 0)
			{
				p = p.left;
			}
			else if (cmp > 0)
			{
				p = p.right;
			}
			else
			{
				return p;
			}
		}
		return null;
	}

	@Nullable
	protected final TreeEntry<K, V> getFirstEntry()
	{
		TreeEntry<K, V> p = this.root;
		if (p != null)
		{
			while (p.left != null)
			{
				p = p.left;
			}
		}
		return p;
	}

	@Nullable
	final TreeEntry<K, V> getLastEntry()
	{
		TreeEntry<K, V> p = this.root;
		if (p != null)
		{
			while (p.right != null)
			{
				p = p.right;
			}
		}
		return p;
	}

	@Nullable
	protected static <K, V> TreeEntry<K, V> successor(@Nullable TreeEntry<K, V> t)
	{
		if (t == null)
		{
			return null;
		}
		else if (t.right != null)
		{
			TreeEntry<K, V> p = t.right;
			while (p.left != null)
			{
				p = p.left;
			}
			return p;
		}
		else
		{
			TreeEntry<K, V> p = t.parent;
			TreeEntry<K, V> ch = t;
			while (p != null && ch == p.right)
			{
				ch = p;
				p = p.parent;
			}
			return p;
		}
	}

	@Nullable
	protected static <K, V> TreeEntry<K, V> predecessor(@Nullable TreeEntry<K, V> t)
	{
		if (t == null)
		{
			return null;
		}
		else if (t.left != null)
		{
			TreeEntry<K, V> p = t.left;
			while (p.right != null)
			{
				p = p.right;
			}
			return p;
		}
		else
		{
			TreeEntry<K, V> p = t.parent;
			TreeEntry<K, V> ch = t;
			while (p != null && ch == p.left)
			{
				ch = p;
				p = p.parent;
			}
			return p;
		}
	}

	private static boolean colorOf(@Nullable TreeEntry<?, ?> p)
	{
		return p == null ? BLACK : p.color;
	}

	@Nullable
	private static <K, V> TreeEntry<K, V> parentOf(@Nullable TreeEntry<K, V> p)
	{
		return p == null ? null : p.parent;
	}

	private static void setColor(@Nullable TreeEntry<?, ?> p, boolean c)
	{
		if (p != null)
		{
			p.color = c;
		}
	}

	@Nullable
	private static <K, V> TreeEntry<K, V> leftOf(@Nullable TreeEntry<K, V> p)
	{
		return p == null ? null : p.left;
	}

	@Nullable
	private static <K, V> TreeEntry<K, V> rightOf(@Nullable TreeEntry<K, V> p)
	{
		return p == null ? null : p.right;
	}

	@SuppressWarnings("PointlessBooleanExpression")
	protected void fixAfterInsertion(@NonNull TreeEntry<K, V> x)
	{
		x.color = RED;

		while (x != null && x != this.root && x.parent.color == RED)
		{
			if (parentOf(x) == leftOf(parentOf(parentOf(x))))
			{
				TreeEntry<K, V> y = rightOf(parentOf(parentOf(x)));
				if (colorOf(y) == RED)
				{
					setColor(parentOf(x), BLACK);
					setColor(y, BLACK);
					setColor(parentOf(parentOf(x)), RED);
					x = parentOf(parentOf(x));
				}
				else
				{
					if (x == rightOf(parentOf(x)))
					{
						x = parentOf(x);
						this.rotateLeft(x);
					}
					setColor(parentOf(x), BLACK);
					setColor(parentOf(parentOf(x)), RED);
					this.rotateRight(parentOf(parentOf(x)));
				}
			}
			else
			{
				TreeEntry<K, V> y = leftOf(parentOf(parentOf(x)));
				if (colorOf(y) == RED)
				{
					setColor(parentOf(x), BLACK);
					setColor(y, BLACK);
					setColor(parentOf(parentOf(x)), RED);
					x = parentOf(parentOf(x));
				}
				else
				{
					if (x == leftOf(parentOf(x)))
					{
						x = parentOf(x);
						this.rotateRight(x);
					}
					setColor(parentOf(x), BLACK);
					setColor(parentOf(parentOf(x)), RED);
					this.rotateLeft(parentOf(parentOf(x)));
				}
			}
		}
		this.root.color = BLACK;
	}

	private void rotateLeft(@Nullable TreeEntry<K, V> p)
	{
		if (p != null)
		{
			TreeEntry<K, V> r = p.right;
			p.right = r.left;
			if (r.left != null)
			{
				r.left.parent = p;
			}
			r.parent = p.parent;
			if (p.parent == null)
			{
				this.root = r;
			}
			else if (p.parent.left == p)
			{
				p.parent.left = r;
			}
			else
			{
				p.parent.right = r;
			}
			r.left = p;
			p.parent = r;
		}
	}

	private void rotateRight(@Nullable TreeEntry<K, V> p)
	{
		if (p != null)
		{
			TreeEntry<K, V> l = p.left;
			p.left = l.right;
			if (l.right != null)
			{
				l.right.parent = p;
			}
			l.parent = p.parent;
			if (p.parent == null)
			{
				this.root = l;
			}
			else if (p.parent.right == p)
			{
				p.parent.right = l;
			}
			else
			{
				p.parent.left = l;
			}
			l.right = p;
			p.parent = l;
		}
	}

	@SuppressWarnings("PointlessBooleanExpression")
	private void fixAfterDeletion(TreeEntry<K, V> x)
	{
		while (x != this.root && colorOf(x) == BLACK)
		{
			if (x == leftOf(parentOf(x)))
			{
				TreeEntry<K, V> sib = rightOf(parentOf(x));

				if (colorOf(sib) == RED)
				{
					setColor(sib, BLACK);
					setColor(parentOf(x), RED);
					this.rotateLeft(parentOf(x));
					sib = rightOf(parentOf(x));
				}

				if (colorOf(leftOf(sib)) == BLACK && colorOf(rightOf(sib)) == BLACK)
				{
					setColor(sib, RED);
					x = parentOf(x);
				}
				else
				{
					if (colorOf(rightOf(sib)) == BLACK)
					{
						setColor(leftOf(sib), BLACK);
						setColor(sib, RED);
						this.rotateRight(sib);
						sib = rightOf(parentOf(x));
					}
					setColor(sib, colorOf(parentOf(x)));
					setColor(parentOf(x), BLACK);
					setColor(rightOf(sib), BLACK);
					this.rotateLeft(parentOf(x));
					x = this.root;
				}
			}
			else
			{
				TreeEntry<K, V> sib = leftOf(parentOf(x));

				if (colorOf(sib) == RED)
				{
					setColor(sib, BLACK);
					setColor(parentOf(x), RED);
					this.rotateRight(parentOf(x));
					sib = leftOf(parentOf(x));
				}

				if (colorOf(rightOf(sib)) == BLACK && colorOf(leftOf(sib)) == BLACK)
				{
					setColor(sib, RED);
					x = parentOf(x);
				}
				else
				{
					if (colorOf(leftOf(sib)) == BLACK)
					{
						setColor(rightOf(sib), BLACK);
						setColor(sib, RED);
						this.rotateLeft(sib);
						sib = leftOf(parentOf(x));
					}
					setColor(sib, colorOf(parentOf(x)));
					setColor(parentOf(x), BLACK);
					setColor(leftOf(sib), BLACK);
					this.rotateRight(parentOf(x));
					x = this.root;
				}
			}
		}

		setColor(x, BLACK);
	}

	@SuppressWarnings("PointlessBooleanExpression")
	protected void deleteEntry(@NonNull TreeEntry<K, V> p)
	{
		this.size--;

		if (p.left != null && p.right != null)
		{
			TreeEntry<K, V> s = successor(p);
			p.key = s.key;
			p.value = s.value;
			p = s;
		}

		TreeEntry<K, V> replacement = p.left != null ? p.left : p.right;

		if (replacement != null)
		{

			replacement.parent = p.parent;
			if (p.parent == null)
			{
				this.root = replacement;
			}
			else if (p == p.parent.left)
			{
				p.parent.left = replacement;
			}
			else
			{
				p.parent.right = replacement;
			}

			p.left = p.right = p.parent = null;

			if (p.color == BLACK)
			{
				this.fixAfterDeletion(replacement);
			}
		}
		else if (p.parent == null)
		{
			this.root = null;
		}
		else
		{
			if (p.color == BLACK)
			{
				this.fixAfterDeletion(p);
			}

			if (p.parent != null)
			{
				if (p == p.parent.left)
				{
					p.parent.left = null;
				}
				else if (p == p.parent.right)
				{
					p.parent.right = null;
				}
				p.parent = null;
			}
		}
	}

	protected void buildFromSorted(int size, @NonNull Iterator<? extends @NonNull Entry<? extends K, ? extends V>> iterator)
	{
		this.size = size;
		this.root = buildFromSorted(0, 0, size - 1, computeRedLevel(size), iterator);
	}

	private static <K, V> TreeEntry<K, V> buildFromSorted(int level, int lo, int hi, int redLevel,
		                                                     @NonNull Iterator<? extends @NonNull Entry<? extends K, ? extends V>> iterator)
	{
		if (hi < lo)
		{
			return null;
		}

		int mid = lo + hi >>> 1;

		TreeEntry<K, V> left = null;
		if (lo < mid)
		{
			left = buildFromSorted(level + 1, lo, mid - 1, redLevel, iterator);
		}

		// extract key and/or value from iterator
		Entry<? extends K, ? extends V> entry = iterator.next();
		TreeEntry<K, V> middle = new TreeEntry<>(entry.getKey(), entry.getValue(), null);

		// color nodes in non-full bottommost level red
		if (level == redLevel)
		{
			middle.color = RED;
		}

		if (left != null)
		{
			middle.left = left;
			left.parent = middle;
		}

		if (mid < hi)
		{
			TreeEntry<K, V> right = buildFromSorted(level + 1, mid + 1, hi, redLevel, iterator);
			assert right != null;

			middle.right = right;
			right.parent = middle;
		}

		return middle;
	}

	protected void buildFromSorted(int size, java.io.@NonNull ObjectInputStream str)
	{
		this.size = size;
		try
		{
			this.root = buildFromSorted(0, 0, size - 1, computeRedLevel(size), str);
		}
		catch (@NonNull IOException | ClassNotFoundException ex)
		{
			// ignored
		}
	}

	private static <K, V> TreeEntry<K, V> buildFromSorted(int level, int lo, int hi, int redLevel,
		                                                     java.io.@NonNull ObjectInputStream str)
		throws java.io.IOException, ClassNotFoundException
	{
		if (hi < lo)
		{
			return null;
		}

		int mid = lo + hi >>> 1;

		TreeEntry<K, V> left = null;
		if (lo < mid)
		{
			left = buildFromSorted(level + 1, lo, mid - 1, redLevel, str);
		}

		// extract key and/or value from stream
		K key = (K) str.readObject();
		V value = (V) str.readObject();

		TreeEntry<K, V> middle = new TreeEntry<>(key, value, null);

		// color nodes in non-full bottommost level red
		if (level == redLevel)
		{
			middle.color = RED;
		}

		if (left != null)
		{
			middle.left = left;
			left.parent = middle;
		}

		if (mid < hi)
		{
			TreeEntry<K, V> right = buildFromSorted(level + 1, mid + 1, hi, redLevel, str);
			assert right != null;

			middle.right = right;
			right.parent = middle;
		}

		return middle;
	}

	private static int computeRedLevel(int sz)
	{
		int level = 0;
		for (int m = sz - 1; m >= 0; m = m / 2 - 1)
		{
			level++;
		}
		return level;
	}

	@NonNull
	@Override
	public <RK, RV> MutableMap<RK, RV> emptyCopy()
	{
		return new dyvil.collection.mutable.TreeMap<>();
	}

	@NonNull
	@Override
	public MutableMap<K, V> mutableCopy()
	{
		return new dyvil.collection.mutable.TreeMap<>(this, this.comparator);
	}

	@NonNull
	@Override
	public ImmutableMap<K, V> immutableCopy()
	{
		return new dyvil.collection.immutable.TreeMap<>(this, this.comparator);
	}

	@Override
	public <RK, RV> ImmutableMap.Builder<RK, RV> immutableBuilder()
	{
		return dyvil.collection.immutable.TreeMap.builder();
	}

	@Override
	public java.util.@NonNull Map<K, V> toJava()
	{
		java.util.TreeMap<K, V> map = new java.util.TreeMap<>(this.comparator);
		for (TreeEntry<K, V> first = this.getFirstEntry(); first != null; first = successor(first))
		{
			map.put(first.key, first.value);
		}
		return map;
	}

	@Override
	public String toString()
	{
		return Map.mapToString(this);
	}

	@Override
	public boolean equals(Object obj)
	{
		return Map.mapEquals(this, obj);
	}

	@Override
	public int hashCode()
	{
		return Map.mapHashCode(this);
	}

	private void writeObject(java.io.@NonNull ObjectOutputStream out) throws IOException
	{
		out.defaultWriteObject();

		out.writeInt(this.size);

		for (TreeEntry<K, V> entry = this.getFirstEntry(); entry != null; entry = successor(entry))
		{
			out.writeObject(entry.key);
			out.writeObject(entry.value);
		}
	}

	private void readObject(java.io.@NonNull ObjectInputStream in) throws IOException, ClassNotFoundException
	{
		in.defaultReadObject();

		this.buildFromSorted(in.readInt(), in);
	}
}
