/*
 * Copyright (C) 2004-2014 Volker Bergmann (volker.bergmann@bergmann-it.de).
 * All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.databene.commons.converter;

import org.databene.commons.ArrayUtil;
import org.databene.commons.Converter;

/**
 * Parent class for {@link Converter} implementations that 
 * holds references to several other converter objects.<br/><br/>
 * Created: 26.02.2010 13:50:43
 * @since 0.5.0
 * @author Volker Bergmann
 */
public abstract class MultiConverterWrapper<S, T> implements Cloneable {

	protected Converter<S, T>[] components;

	protected MultiConverterWrapper(Converter<S, T>[] components) {
	    this.components = components;
    }

	// properties ------------------------------------------------------------------------------------------------------

	public Converter<S, T>[] getComponents() {
        return components;
    }
	
	public void setComponents(Converter<S, T>[] converters) {
        this.components = converters;
    }
	
	public void addComponent(Converter<S, T> converter) {
        this.components = ArrayUtil.append(converter, this.components);
    }
	
	// Converter interface implementation ------------------------------------------------------------------------------

	public boolean isThreadSafe() {
		for (Converter<?,?> converter : components)
			if (!converter.isThreadSafe())
				return false;
		return true;
	}
	
	public boolean isParallelizable() {
		for (Converter<?,?> converter : components)
			if (!converter.isParallelizable())
				return false;
		return true;
	}
	
	@SuppressWarnings({ "unchecked", "rawtypes" })
    @Override
    public Object clone() {
        try {
    		MultiConverterWrapper copy = (MultiConverterWrapper) super.clone();
			copy.components = ConverterManager.cloneIfSupported(this.components);
			return copy;
		} catch (CloneNotSupportedException e) {
	        throw new RuntimeException(e);
        }
	}
	
}
