/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.gitlab;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.json.JsonMapper;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.nifi.gitlab.GitLabAuthenticationType;
import org.apache.nifi.gitlab.PersonalAccessToken;
import org.apache.nifi.registry.flow.FlowRegistryException;
import org.apache.nifi.registry.flow.git.client.GitCommit;
import org.apache.nifi.registry.flow.git.client.GitCreateContentRequest;
import org.apache.nifi.registry.flow.git.client.GitRepositoryClient;
import org.gitlab4j.api.CommitsApi;
import org.gitlab4j.api.Constants;
import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.ProjectApi;
import org.gitlab4j.api.RepositoryApi;
import org.gitlab4j.api.models.AccessLevel;
import org.gitlab4j.api.models.Branch;
import org.gitlab4j.api.models.Commit;
import org.gitlab4j.api.models.Permissions;
import org.gitlab4j.api.models.Project;
import org.gitlab4j.api.models.ProjectAccess;
import org.gitlab4j.api.models.RepositoryFile;
import org.gitlab4j.api.models.TreeItem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GitLabRepositoryClient
implements GitRepositoryClient {
    private static final Logger LOGGER = LoggerFactory.getLogger(GitLabRepositoryClient.class);
    private static final String PERSONAL_ACCESS_TOKENS_SELF_PATH = "/personal_access_tokens/self";
    private static final String PRIVATE_TOKEN_HEADER = "PRIVATE-TOKEN";
    private static final String READ_API_SCOPE = "read_api";
    private static final String WRITE_API_SCOPE = "api";
    private static final String DIRECTORY_MODE = "040000";
    private static final int DEFAULT_ITEMS_PER_PAGE = 100;
    private static final TokenInfo UNKNOWN_TOKEN = new TokenInfo("unknown", false, false);
    private final ObjectMapper objectMapper = ((JsonMapper.Builder)JsonMapper.builder().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)).build();
    private final String clientId;
    private final String repoNamespace;
    private final String repoName;
    private final String repoPath;
    private final String projectPath;
    private final int connectTimeout;
    private final int readTimeout;
    private final GitLabApi gitLab;
    private final boolean canRead;
    private final boolean canWrite;

    private GitLabRepositoryClient(Builder builder) throws FlowRegistryException {
        String apiUrl = Objects.requireNonNull(builder.apiUrl, "API URL is required");
        GitLabApi.ApiVersion apiVersion = Objects.requireNonNull(builder.apiVersion, "API version is required");
        GitLabAuthenticationType authenticationType = Objects.requireNonNull(builder.authenticationType, "Authentication type is required");
        String authToken = Objects.requireNonNull(builder.authToken, "Authentication token is required");
        switch (authenticationType) {
            default: {
                throw new MatchException(null, null);
            }
            case ACCESS_TOKEN: 
        }
        Constants.TokenType tokenType = Constants.TokenType.ACCESS;
        this.clientId = Objects.requireNonNull(builder.clientId, "Client Id is required");
        this.repoNamespace = Objects.requireNonNull(builder.repoNamespace, "Repository Group is required");
        this.repoName = Objects.requireNonNull(builder.repoName, "Repository Name is required");
        this.repoPath = builder.repoPath;
        this.projectPath = this.repoNamespace + "/" + this.repoName;
        this.connectTimeout = builder.connectTimeout;
        this.readTimeout = builder.readTimeout;
        this.gitLab = new GitLabApi(apiVersion, apiUrl, tokenType, authToken);
        this.gitLab.setRequestTimeout(Integer.valueOf(builder.connectTimeout), Integer.valueOf(builder.readTimeout));
        this.gitLab.setDefaultPerPage(100);
        TokenInfo tokenInfo = this.getTokenInfo();
        Optional<Project> projectOptional = this.getProject();
        if (projectOptional.isPresent()) {
            Project project = projectOptional.get();
            this.canRead = true;
            this.canWrite = tokenInfo.canWriteApi() && this.hasMinimumAccessLevel(project, AccessLevel.DEVELOPER);
        } else {
            this.canRead = false;
            this.canWrite = false;
        }
        LOGGER.info("Created {} for clientId = [{}], repository [{}], canRead [{}], canWrite [{}]", new Object[]{this.getClass().getSimpleName(), this.clientId, this.projectPath, this.canRead, this.canWrite});
    }

    public String getRepoNamespace() {
        return this.repoNamespace;
    }

    public String getRepoName() {
        return this.repoName;
    }

    public boolean hasReadPermission() {
        return this.canRead;
    }

    public boolean hasWritePermission() {
        return this.canWrite;
    }

    public Set<String> getBranches() throws FlowRegistryException {
        LOGGER.debug("Getting branches for repository [{}]", (Object)this.projectPath);
        RepositoryApi repositoryApi = this.gitLab.getRepositoryApi();
        return this.execute(() -> repositoryApi.getBranchesStream((Object)this.projectPath).map(Branch::getName).collect(Collectors.toSet()));
    }

    public Set<String> getTopLevelDirectoryNames(String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath("");
        LOGGER.debug("Getting top-level directories for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.projectPath});
        RepositoryApi repositoryApi = this.gitLab.getRepositoryApi();
        return this.execute(() -> repositoryApi.getTreeStream((Object)this.projectPath, resolvedPath, branch).filter(this::isDirectory).map(TreeItem::getName).collect(Collectors.toSet()));
    }

    public Set<String> getFileNames(String directory, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(directory);
        LOGGER.debug("Getting filenames for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.projectPath});
        RepositoryApi repositoryApi = this.gitLab.getRepositoryApi();
        return this.execute(() -> repositoryApi.getTreeStream((Object)this.projectPath, resolvedPath, branch).filter(this::isFile).map(TreeItem::getName).collect(Collectors.toSet()));
    }

    public List<GitCommit> getCommits(String path, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        LOGGER.debug("Getting commits for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.projectPath});
        CommitsApi commitsApi = this.gitLab.getCommitsApi();
        return this.execute(() -> commitsApi.getCommits((Object)this.projectPath, branch, resolvedPath).stream().map(this::toGitCommit).toList());
    }

    public InputStream getContentFromBranch(String path, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        LOGGER.debug("Getting content for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.projectPath});
        return this.execute(() -> this.gitLab.getRepositoryFileApi().getRawFile((Object)this.projectPath, branch, resolvedPath));
    }

    public InputStream getContentFromCommit(String path, String commitSha) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        LOGGER.debug("Getting content for path [{}] from commit [{}] in repository [{}]", new Object[]{resolvedPath, commitSha, this.projectPath});
        return this.execute(() -> this.gitLab.getRepositoryFileApi().getRawFile((Object)this.projectPath, commitSha, resolvedPath));
    }

    public Optional<String> getContentSha(String path, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        LOGGER.debug("Getting content SHA for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.projectPath});
        return this.execute(() -> this.gitLab.getRepositoryFileApi().getOptionalFileInfo((Object)this.projectPath, resolvedPath, branch).map(RepositoryFile::getCommitId));
    }

    public String createContent(GitCreateContentRequest request) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(request.getPath());
        String branch = request.getBranch();
        LOGGER.debug("Creating content at path [{}] on branch [{}] in repository [{}] ", new Object[]{resolvedPath, branch, this.projectPath});
        return this.execute(() -> {
            Optional existingFileInfo = this.gitLab.getRepositoryFileApi().getOptionalFileInfo((Object)this.projectPath, resolvedPath, branch);
            if (existingFileInfo.isPresent()) {
                LOGGER.debug("Updating existing file [{}]", (Object)resolvedPath);
                RepositoryFile existingFile = (RepositoryFile)existingFileInfo.get();
                existingFile.encodeAndSetContent(request.getContent());
                this.gitLab.getRepositoryFileApi().updateFile((Object)this.projectPath, existingFile, branch, request.getMessage());
            } else {
                LOGGER.debug("Creating new file [{}]", (Object)resolvedPath);
                RepositoryFile newFile = new RepositoryFile();
                newFile.setFilePath(resolvedPath);
                newFile.encodeAndSetContent(request.getContent());
                this.gitLab.getRepositoryFileApi().createFile((Object)this.projectPath, newFile, branch, request.getMessage());
            }
            return this.gitLab.getRepositoryFileApi().getFileInfo((Object)this.projectPath, resolvedPath, branch).getCommitId();
        });
    }

    public InputStream deleteContent(String filePath, String commitMessage, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(filePath);
        LOGGER.debug("Deleting content at path [{}] on branch [{}] in repository [{}] ", new Object[]{resolvedPath, branch, this.projectPath});
        return this.execute(() -> {
            InputStream content = this.gitLab.getRepositoryFileApi().getRawFile((Object)this.projectPath, branch, resolvedPath);
            this.gitLab.getRepositoryFileApi().deleteFile((Object)this.projectPath, resolvedPath, branch, commitMessage);
            return content;
        });
    }

    public void close() {
        this.gitLab.close();
    }

    private Optional<Project> getProject() throws FlowRegistryException {
        try {
            ProjectApi projectApi = this.gitLab.getProjectApi();
            Project project = projectApi.getProject(this.repoNamespace, this.repoName);
            LOGGER.debug("Successfully retrieved project [{}] for client [{}]", (Object)this.projectPath, (Object)this.clientId);
            return Optional.of(project);
        }
        catch (GitLabApiException e) {
            if (e.getHttpStatus() == 401) {
                LOGGER.warn("Client [{}] does not have permissions to access repository [{}]", (Object)this.clientId, (Object)this.projectPath);
                return Optional.empty();
            }
            if (e.getHttpStatus() == 404) {
                throw new FlowRegistryException(String.format("Repository [%s] not found", this.projectPath), (Throwable)e);
            }
            throw new FlowRegistryException(e.getMessage(), (Throwable)e);
        }
    }

    private boolean hasMinimumAccessLevel(Project project, AccessLevel accessLevel) {
        ProjectAccess groupAccess;
        Permissions permissions = project.getPermissions();
        LOGGER.debug("Checking if client [{}] has access level [{}] in project [{}]", new Object[]{this.clientId, accessLevel.name(), this.projectPath});
        ProjectAccess projectAccess = permissions.getProjectAccess();
        if (projectAccess != null) {
            AccessLevel projectAccessLevel = projectAccess.getAccessLevel();
            LOGGER.debug("Client [{}] has project access level [{}] for project [{}]", new Object[]{this.clientId, projectAccessLevel.name(), this.projectPath});
            if (projectAccessLevel.toValue() >= accessLevel.toValue()) {
                return true;
            }
        }
        if ((groupAccess = permissions.getGroupAccess()) != null) {
            AccessLevel groupAccessLevel = groupAccess.getAccessLevel();
            LOGGER.debug("Client [{}] has group access level [{}] for project [{}]", new Object[]{this.clientId, groupAccessLevel.name(), this.projectPath});
            if (groupAccessLevel.toValue() >= accessLevel.toValue()) {
                return true;
            }
        }
        LOGGER.debug("Client [{}] does not have minimum access level [{}] for project [{}]", new Object[]{this.clientId, accessLevel.name(), this.projectPath});
        return false;
    }

    private TokenInfo getTokenInfo() {
        TokenInfo tokenInfo = this.getPersonalAccessToken().map(this::createTokenInfo).orElse(UNKNOWN_TOKEN);
        LOGGER.debug("Created token info {} for client [{}]", (Object)tokenInfo, (Object)this.clientId);
        return tokenInfo;
    }

    private TokenInfo createTokenInfo(PersonalAccessToken personalAccessToken) {
        HashSet<String> tokenScopes = new HashSet<String>(personalAccessToken.scopes());
        boolean canReadApi = tokenScopes.contains(READ_API_SCOPE) || tokenScopes.contains(WRITE_API_SCOPE);
        boolean canWriteApi = tokenScopes.contains(WRITE_API_SCOPE);
        return new TokenInfo(personalAccessToken.name(), canReadApi, canWriteApi);
    }

    private Optional<PersonalAccessToken> getPersonalAccessToken() {
        try {
            PersonalAccessToken personalAccessToken = this.retrievePersonalAccessToken();
            return Optional.of(personalAccessToken);
        }
        catch (FlowRegistryException e) {
            LOGGER.warn("Failed to get personal access token for client [{}]", (Object)this.clientId, (Object)e);
            return Optional.empty();
        }
    }

    private PersonalAccessToken retrievePersonalAccessToken() throws FlowRegistryException {
        String responseContent;
        int responseCode;
        HttpURLConnection connection = null;
        try {
            connection = this.createConnection(PERSONAL_ACCESS_TOKENS_SELF_PATH);
            responseCode = connection.getResponseCode();
            responseContent = IOUtils.toString((InputStream)connection.getInputStream(), (Charset)StandardCharsets.UTF_8);
        }
        catch (Exception e) {
            throw new FlowRegistryException("Unable to retrieve personal access token details", (Throwable)e);
        }
        finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
        LOGGER.debug("Personal access token response code = {}", (Object)responseCode);
        if (responseCode != 200) {
            throw new FlowRegistryException("Unable to retrieve personal access token details: " + responseContent);
        }
        try {
            return (PersonalAccessToken)this.objectMapper.readValue(responseContent, PersonalAccessToken.class);
        }
        catch (IOException e) {
            throw new FlowRegistryException("Unable to parse personal access token details", (Throwable)e);
        }
    }

    private HttpURLConnection createConnection(String subPath) throws IOException {
        String gitLabServerUrl = this.gitLab.getGitLabServerUrl().endsWith("/") ? this.gitLab.getGitLabServerUrl().replaceAll("/$", "") : this.gitLab.getGitLabServerUrl();
        URL gitLabApiUrl = URI.create(gitLabServerUrl + this.gitLab.getApiVersion().getApiNamespace() + subPath).toURL();
        LOGGER.debug("Connecting to GitLab URL [{}] for client [{}]", (Object)gitLabApiUrl, (Object)this.clientId);
        HttpURLConnection connection = (HttpURLConnection)gitLabApiUrl.openConnection();
        connection.setRequestProperty(PRIVATE_TOKEN_HEADER, this.gitLab.getAuthToken());
        connection.setConnectTimeout(this.connectTimeout);
        connection.setReadTimeout(this.readTimeout);
        return connection;
    }

    private String getResolvedPath(String path) {
        return this.repoPath == null ? path : this.repoPath + "/" + path;
    }

    private boolean isDirectory(TreeItem item) {
        return item.getMode().equals(DIRECTORY_MODE);
    }

    private boolean isFile(TreeItem item) {
        return !this.isDirectory(item);
    }

    private GitCommit toGitCommit(Commit commit) {
        return new GitCommit(commit.getId(), commit.getAuthorName(), commit.getMessage(), Instant.ofEpochMilli(commit.getCommittedDate().getTime()));
    }

    private <T> T execute(GitLabRequest<T> action) throws FlowRegistryException {
        try {
            return action.execute();
        }
        catch (GitLabApiException e) {
            switch (e.getHttpStatus()) {
                case 401: {
                    throw new FlowRegistryException("Client does not have permission to perform the given action", (Throwable)e);
                }
                case 404: {
                    throw new FlowRegistryException("Path or Branch not found", (Throwable)e);
                }
            }
            throw new FlowRegistryException(e.getMessage(), (Throwable)e);
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String clientId;
        private String apiUrl;
        private GitLabApi.ApiVersion apiVersion;
        private GitLabAuthenticationType authenticationType;
        private String authToken;
        private String repoNamespace;
        private String repoName;
        private String repoPath;
        private int connectTimeout;
        private int readTimeout;

        public Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public Builder apiUrl(String apiUrl) {
            this.apiUrl = apiUrl;
            return this;
        }

        public Builder apiVersion(GitLabApi.ApiVersion apiVersion) {
            this.apiVersion = apiVersion;
            return this;
        }

        public Builder authenticationType(GitLabAuthenticationType authenticationType) {
            this.authenticationType = authenticationType;
            return this;
        }

        public Builder authToken(String authToken) {
            this.authToken = authToken;
            return this;
        }

        public Builder repoNamespace(String repoNamespace) {
            this.repoNamespace = repoNamespace;
            return this;
        }

        public Builder repoName(String repoName) {
            this.repoName = repoName;
            return this;
        }

        public Builder repoPath(String repoPath) {
            this.repoPath = repoPath;
            return this;
        }

        public Builder connectTimeout(int connectTimeout) {
            this.connectTimeout = connectTimeout;
            return this;
        }

        public Builder readTimeout(int readTimeout) {
            this.readTimeout = readTimeout;
            return this;
        }

        public GitLabRepositoryClient build() throws FlowRegistryException {
            return new GitLabRepositoryClient(this);
        }
    }

    private record TokenInfo(String name, boolean canReadApi, boolean canWriteApi) {
        @Override
        public String toString() {
            return new ToStringBuilder((Object)this).append("name", (Object)this.name).append("canReadApi", this.canReadApi).append("canWriteApi", this.canWriteApi).toString();
        }
    }

    private static interface GitLabRequest<T> {
        public T execute() throws GitLabApiException, FlowRegistryException;
    }
}

