/*
 * Decompiled with CFR 0.152.
 */
package org.apache.maven.plugin.compiler;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Stream;
import org.apache.maven.plugin.compiler.AbstractCompilerMojo;
import org.apache.maven.plugin.compiler.CompilationFailureException;
import org.apache.maven.plugin.compiler.SourceFile;

final class IncrementalBuild {
    private static final LinkOption[] LINK_OPTIONS = new LinkOption[0];
    private static final long MAGIC_NUMBER = -8163803035240576921L;
    private static final byte NEW_SOURCE_DIRECTORY = 1;
    private static final byte NEW_TARGET_DIRECTORY = 2;
    private static final byte EXPLICIT_OUTPUT_FILE = 4;
    private final Path cacheFile;
    private boolean cacheLoaded;
    private final List<SourceFile> sourceFiles;
    private final long buildTime;
    private long previousBuildTime;
    private int previousOptionsHash;
    private final boolean showCompilationChanges;

    IncrementalBuild(AbstractCompilerMojo mojo, List<SourceFile> sourceFiles) throws IOException {
        this.sourceFiles = sourceFiles;
        Path file = mojo.mojoStatusPath;
        this.cacheFile = Files.createDirectories(file.getParent(), new FileAttribute[0]).resolve(file.getFileName());
        this.showCompilationChanges = mojo.showCompilationChanges;
        this.previousBuildTime = this.buildTime = System.currentTimeMillis();
    }

    public void writeCache(int optionsHash, boolean sources) throws IOException {
        try (DataOutputStream out = new DataOutputStream(new BufferedOutputStream(Files.newOutputStream(this.cacheFile, StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING)));){
            out.writeLong(-8163803035240576921L);
            out.writeLong(this.buildTime);
            out.writeInt(optionsHash);
            out.writeInt(sources ? this.sourceFiles.size() : 0);
            if (sources) {
                Path srcDir = null;
                Path tgtDir = null;
                Path previousParent = null;
                for (SourceFile source : this.sourceFiles) {
                    Path sourceFile = source.file;
                    Path outputFile = source.getOutputFile(false);
                    Path path = srcDir;
                    srcDir = source.directory.root;
                    boolean sameSrcDir = Objects.equals(path, srcDir);
                    Path path2 = tgtDir;
                    tgtDir = source.directory.outputDirectory;
                    boolean sameTgtDir = Objects.equals(path2, tgtDir);
                    boolean sameOutput = outputFile == null || outputFile.equals(SourceInfo.toOutputFile(srcDir, tgtDir, sourceFile));
                    out.writeByte((sameSrcDir ? 0 : 1) | (sameTgtDir ? 0 : 2) | (sameOutput ? 0 : 4));
                    if (!sameSrcDir) {
                        previousParent = srcDir;
                        out.writeUTF(previousParent.toString());
                    }
                    if (!sameTgtDir) {
                        out.writeUTF(tgtDir.toString());
                    }
                    if (!sameOutput) {
                        out.writeUTF(outputFile.toString());
                    }
                    out.writeUTF(previousParent.relativize(sourceFile).toString());
                    out.writeLong(source.lastModified);
                    previousParent = sourceFile.getParent();
                }
            }
        }
    }

    private Map<Path, SourceInfo> loadCache() throws IOException {
        HashMap<Path, SourceInfo> previousBuild;
        try (DataInputStream in = new DataInputStream(new BufferedInputStream(Files.newInputStream(this.cacheFile, StandardOpenOption.READ)));){
            if (in.readLong() != -8163803035240576921L) {
                throw new IOException("Invalid cache file.");
            }
            this.previousBuildTime = in.readLong();
            this.previousOptionsHash = in.readInt();
            int remaining = in.readInt();
            previousBuild = new HashMap<Path, SourceInfo>(remaining + remaining / 3);
            Path srcDir = null;
            Path tgtDir = null;
            Path srcFile = null;
            while (--remaining >= 0) {
                byte flags = in.readByte();
                if ((flags & 0xFFFFFFF8) != 0) {
                    throw new IOException("Invalid cache file.");
                }
                boolean newSrcDir = (flags & 1) != 0;
                boolean newTgtDir = (flags & 2) != 0;
                boolean newOutput = (flags & 4) != 0;
                Path output = null;
                if (newSrcDir) {
                    srcDir = Path.of(in.readUTF(), new String[0]);
                }
                if (newTgtDir) {
                    tgtDir = Path.of(in.readUTF(), new String[0]);
                }
                if (newOutput) {
                    output = Path.of(in.readUTF(), new String[0]);
                }
                String path = in.readUTF();
                Path path2 = srcFile = newSrcDir ? srcDir.resolve(path) : srcFile.resolveSibling(path);
                if (previousBuild.put(srcFile = srcFile.normalize(), new SourceInfo(srcDir, tgtDir, output, in.readLong())) == null) continue;
                throw new IOException("Duplicated source file declared in the cache: " + String.valueOf(srcFile));
            }
        }
        this.cacheLoaded = true;
        return previousBuild;
    }

    String inputFileTreeChanges(long staleMillis, boolean rebuildOnAdd) throws IOException {
        Map<Path, SourceInfo> previousBuild;
        try {
            previousBuild = this.loadCache();
        }
        catch (NoSuchFileException e) {
            return "Compiling all files.";
        }
        catch (IOException e) {
            return IncrementalBuild.causeOfRebuild("information about the previous build cannot be read", true).append(System.lineSeparator()).append(e).toString();
        }
        boolean rebuild = false;
        boolean allChanged = true;
        ArrayList<Path> added = new ArrayList<Path>();
        for (SourceFile sourceFile : this.sourceFiles) {
            SourceInfo previous = previousBuild.remove(sourceFile.file);
            if (previous != null) {
                if (sourceFile.lastModified - previous.lastModified <= staleMillis) {
                    allChanged = false;
                    Path output = sourceFile.getOutputFile(true);
                    if (Files.exists(output, LINK_OPTIONS)) {
                        continue;
                    }
                }
            } else if (!sourceFile.ignoreModification) {
                if (this.showCompilationChanges) {
                    added.add(sourceFile.file);
                }
                rebuild |= rebuildOnAdd;
            }
            sourceFile.isNewOrModified = true;
        }
        if (previousBuild.isEmpty()) {
            if (allChanged) {
                return IncrementalBuild.causeOfRebuild("all source files changed", false).toString();
            }
            if (!rebuild) {
                return null;
            }
        }
        for (Map.Entry entry : previousBuild.entrySet()) {
            ((SourceInfo)entry.getValue()).deleteClassFiles((Path)entry.getKey());
        }
        StringBuilder causeOfRebuild = IncrementalBuild.causeOfRebuild("of added or removed source files", this.showCompilationChanges);
        if (this.showCompilationChanges) {
            for (Path fileAdded : added) {
                causeOfRebuild.append(System.lineSeparator()).append("  + ").append(fileAdded);
            }
            for (Path fileRemoved : previousBuild.keySet()) {
                causeOfRebuild.append(System.lineSeparator()).append("  - ").append(fileRemoved);
            }
        }
        return causeOfRebuild.toString();
    }

    String dependencyChanges(Iterable<List<Path>> dependencies, Collection<String> fileExtensions) throws IOException {
        if (!this.cacheLoaded) {
            this.loadCache();
        }
        FileTime changeTime = FileTime.fromMillis(this.previousBuildTime);
        ArrayList updated = new ArrayList();
        for (List<Path> roots : dependencies) {
            for (Path root : roots) {
                try {
                    Stream<Path> files = Files.walk(root, new FileVisitOption[0]);
                    try {
                        files.filter(f -> {
                            String name = f.getFileName().toString();
                            int s = name.lastIndexOf(46);
                            if (s < 0 || !fileExtensions.contains(name.substring(s + 1))) {
                                return false;
                            }
                            try {
                                return Files.isRegularFile(f, new LinkOption[0]) && Files.getLastModifiedTime(f, new LinkOption[0]).compareTo(changeTime) >= 0;
                            }
                            catch (IOException e) {
                                throw new UncheckedIOException(e);
                            }
                        }).forEach(updated::add);
                    }
                    finally {
                        if (files == null) continue;
                        files.close();
                    }
                }
                catch (UncheckedIOException e) {
                    throw e.getCause();
                }
            }
        }
        if (updated.isEmpty()) {
            return null;
        }
        StringBuilder causeOfRebuild = IncrementalBuild.causeOfRebuild("some dependencies changed", this.showCompilationChanges);
        if (this.showCompilationChanges) {
            for (Path file : updated) {
                causeOfRebuild.append(System.lineSeparator()).append("    ").append(file);
            }
        }
        return causeOfRebuild.toString();
    }

    String optionChanges(int optionsHash) throws IOException {
        if (!this.cacheLoaded) {
            this.loadCache();
        }
        if (optionsHash == this.previousOptionsHash) {
            return null;
        }
        return IncrementalBuild.causeOfRebuild("of changes in compiler options", false).toString();
    }

    private static StringBuilder causeOfRebuild(String cause, boolean colon) {
        return new StringBuilder(128).append("Recompiling all files because ").append(cause).append(colon ? (char)':' : '.');
    }

    String markNewOrModifiedSources(long staleMillis, boolean rebuildOnAdd) throws IOException {
        for (SourceFile source : this.sourceFiles) {
            if (source.isNewOrModified) continue;
            Path output = source.getOutputFile(true);
            if (Files.exists(output, LINK_OPTIONS)) {
                FileTime t = Files.getLastModifiedTime(output, LINK_OPTIONS);
                if (source.lastModified - t.toMillis() <= staleMillis) {
                    continue;
                }
            } else if (rebuildOnAdd) {
                StringBuilder causeOfRebuild = IncrementalBuild.causeOfRebuild("of added source files", this.showCompilationChanges);
                if (this.showCompilationChanges) {
                    causeOfRebuild.append(System.lineSeparator()).append("  + ").append(source.file);
                }
                return causeOfRebuild.toString();
            }
            source.isNewOrModified = true;
        }
        return null;
    }

    List<SourceFile> getModifiedSources() {
        return this.sourceFiles.stream().filter(s -> s.isNewOrModified).toList();
    }

    static boolean isEmptyOrIgnorable(List<SourceFile> sourceFiles) {
        return !sourceFiles.stream().anyMatch(s -> !s.ignoreModification);
    }

    private record SourceInfo(Path sourceDirectory, Path outputDirectory, Path outputFile, long lastModified) {
        private static final String OUTPUT_EXTENSION = ".class";

        static Path toOutputFile(Path sourceDirectory, Path outputDirectory, Path sourceFile) {
            return SourceFile.toOutputFile(sourceDirectory, outputDirectory, sourceFile, ".java", OUTPUT_EXTENSION);
        }

        void deleteClassFiles(Path sourceFile) throws IOException {
            String filename;
            Path output = this.outputFile;
            if (output == null) {
                output = SourceInfo.toOutputFile(this.sourceDirectory, this.outputDirectory, sourceFile);
            }
            if ((filename = output.getFileName().toString()).endsWith(OUTPUT_EXTENSION)) {
                List<Path> outputs;
                String prefix = filename.substring(0, filename.length() - OUTPUT_EXTENSION.length());
                try (Stream<Path> files = Files.walk(output.getParent(), 1, new FileVisitOption[0]);){
                    outputs = files.filter(f -> {
                        String name = f.getFileName().toString();
                        return name.startsWith(prefix) && name.endsWith(OUTPUT_EXTENSION) && (name.equals(filename) || name.charAt(prefix.length()) == '$');
                    }).toList();
                }
                for (Path p : outputs) {
                    Files.delete(p);
                }
            } else {
                Files.deleteIfExists(output);
            }
        }
    }

    static enum Aspect {
        OPTIONS(Set.of()),
        DEPENDENCIES(Set.of()),
        SOURCES(Set.of()),
        CLASSES(Set.of()),
        ADDITIONS(Set.of()),
        MODULES(Set.of(SOURCES, CLASSES, ADDITIONS)),
        NONE(Set.of(OPTIONS, DEPENDENCIES, SOURCES, CLASSES, ADDITIONS, MODULES));

        private final Set<Aspect> excludes;

        private Aspect(Set<Aspect> excludes) {
            this.excludes = excludes;
        }

        public String toString() {
            return this.name().toLowerCase(Locale.US);
        }

        static EnumSet<Aspect> parse(String values) {
            EnumSet<Aspect> aspects = EnumSet.noneOf(Aspect.class);
            for (String value : values.split(",")) {
                value = value.trim();
                try {
                    aspects.add(Aspect.valueOf(value.toUpperCase(Locale.US)));
                }
                catch (IllegalArgumentException e) {
                    StringBuilder sb = new StringBuilder(256).append("Illegal incremental build setting: \"").append(value);
                    String s = "\". Valid values are ";
                    for (Aspect aspect : Aspect.values()) {
                        sb.append(s).append((Object)aspect);
                        s = ", ";
                    }
                    throw new CompilationFailureException(sb.append('.').toString(), e);
                }
            }
            for (Aspect aspect : aspects) {
                for (Aspect exclude : aspect.excludes) {
                    if (!aspects.contains((Object)exclude)) continue;
                    throw new CompilationFailureException("Illegal incremental build setting: \"" + String.valueOf((Object)aspect) + "\" and \"" + String.valueOf((Object)exclude) + "\" are mutually exclusive.");
                }
            }
            if (aspects.isEmpty()) {
                throw new CompilationFailureException("Incremental build setting cannot be empty.");
            }
            return aspects;
        }
    }
}

