/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.backup;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.ChoreService;
import org.apache.hadoop.hbase.HBaseClassTestRule;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.ScheduledChore;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.backup.HFileArchiver;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.master.cleaner.DirScanPool;
import org.apache.hadoop.hbase.master.cleaner.HFileCleaner;
import org.apache.hadoop.hbase.regionserver.ConstantSizeRegionSplitPolicy;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.regionserver.HRegionServer;
import org.apache.hadoop.hbase.regionserver.HStoreFile;
import org.apache.hadoop.hbase.testclassification.MediumTests;
import org.apache.hadoop.hbase.testclassification.MiscTests;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.HFileArchiveTestingUtil;
import org.apache.hadoop.hbase.util.HFileArchiveUtil;
import org.apache.hadoop.hbase.util.StoppableImplementation;
import org.apache.hadoop.security.UserGroupInformation;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.TestName;
import org.mockito.ArgumentCaptor;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Category(value={MediumTests.class, MiscTests.class})
public class TestHFileArchiving {
    @ClassRule
    public static final HBaseClassTestRule CLASS_RULE = HBaseClassTestRule.forClass(TestHFileArchiving.class);
    private static final Logger LOG = LoggerFactory.getLogger(TestHFileArchiving.class);
    private static final HBaseTestingUtility UTIL = new HBaseTestingUtility();
    private static final byte[] TEST_FAM = Bytes.toBytes((String)"fam");
    private static DirScanPool POOL;
    @Rule
    public TestName name = new TestName();

    @BeforeClass
    public static void setupCluster() throws Exception {
        TestHFileArchiving.setupConf(UTIL.getConfiguration());
        UTIL.startMiniCluster();
        UTIL.getMiniHBaseCluster().getMaster().getHFileCleaner().cancel(true);
        POOL = new DirScanPool(UTIL.getConfiguration());
    }

    private static void setupConf(Configuration conf) {
        conf.setInt("hbase.regionsever.info.port", -1);
        conf.setInt("hbase.hregion.memstore.flush.size", 25000);
        conf.setInt("hbase.hregion.majorcompaction", 0);
        conf.set("hbase.regionserver.region.split.policy", ConstantSizeRegionSplitPolicy.class.getName());
    }

    @After
    public void tearDown() throws Exception {
        this.clearArchiveDirectory();
    }

    @AfterClass
    public static void cleanupTest() throws Exception {
        UTIL.shutdownMiniCluster();
        POOL.shutdownNow();
    }

    @Test
    public void testArchiveStoreFilesDifferentFileSystemsWallWithSchemaPlainRoot() throws Exception {
        String walDir = "mockFS://mockFSAuthority:9876/mockDir/wals/";
        String baseDir = FSUtils.getRootDir((Configuration)UTIL.getConfiguration()).toString() + "/";
        this.testArchiveStoreFilesDifferentFileSystems(walDir, baseDir, HFileArchiver::archiveStoreFiles);
    }

    @Test
    public void testArchiveStoreFilesDifferentFileSystemsWallNullPlainRoot() throws Exception {
        String baseDir = FSUtils.getRootDir((Configuration)UTIL.getConfiguration()).toString() + "/";
        this.testArchiveStoreFilesDifferentFileSystems(null, baseDir, HFileArchiver::archiveStoreFiles);
    }

    @Test
    public void testArchiveStoreFilesDifferentFileSystemsWallAndRootSame() throws Exception {
        String baseDir = FSUtils.getRootDir((Configuration)UTIL.getConfiguration()).toString() + "/";
        this.testArchiveStoreFilesDifferentFileSystems("/hbase/wals/", baseDir, HFileArchiver::archiveStoreFiles);
    }

    private void testArchiveStoreFilesDifferentFileSystems(String walDir, String expectedBase, ArchivingFunction<Configuration, FileSystem, RegionInfo, Path, byte[], Collection<HStoreFile>> archivingFunction) throws IOException {
        FileSystem mockedFileSystem = (FileSystem)Mockito.mock(FileSystem.class);
        Configuration conf = new Configuration(UTIL.getConfiguration());
        if (walDir != null) {
            conf.set("hbase.wal.dir", walDir);
        }
        Path filePath = new Path("/mockDir/wals/mockFile");
        Mockito.when((Object)mockedFileSystem.getScheme()).thenReturn((Object)"mockFS");
        Mockito.when((Object)mockedFileSystem.mkdirs((Path)ArgumentMatchers.any())).thenReturn((Object)true);
        Mockito.when((Object)mockedFileSystem.exists((Path)ArgumentMatchers.any())).thenReturn((Object)true);
        RegionInfo mockedRegion = (RegionInfo)Mockito.mock(RegionInfo.class);
        TableName tableName = TableName.valueOf((String)"mockTable");
        Mockito.when((Object)mockedRegion.getTable()).thenReturn((Object)tableName);
        Mockito.when((Object)mockedRegion.getEncodedName()).thenReturn((Object)"mocked-region-encoded-name");
        Path tableDir = new Path("mockFS://mockDir/tabledir");
        byte[] family = Bytes.toBytes((String)"testfamily");
        HStoreFile mockedFile = (HStoreFile)Mockito.mock(HStoreFile.class);
        ArrayList<HStoreFile> list = new ArrayList<HStoreFile>();
        list.add(mockedFile);
        Mockito.when((Object)mockedFile.getPath()).thenReturn((Object)filePath);
        Mockito.when((Object)mockedFileSystem.rename((Path)ArgumentMatchers.any(), (Path)ArgumentMatchers.any())).thenReturn((Object)true);
        archivingFunction.apply(conf, mockedFileSystem, mockedRegion, tableDir, family, list);
        ArgumentCaptor pathCaptor = ArgumentCaptor.forClass(Path.class);
        ((FileSystem)Mockito.verify((Object)mockedFileSystem, (VerificationMode)Mockito.times((int)2))).rename((Path)pathCaptor.capture(), (Path)ArgumentMatchers.any());
        String expectedDir = expectedBase + "archive/data/default/mockTable/mocked-region-encoded-name/testfamily/mockFile";
        Assert.assertTrue((boolean)((Path)pathCaptor.getAllValues().get(0)).toString().equals(expectedDir));
    }

    @Test
    public void testArchiveRecoveredEditsWalDirNull() throws Exception {
        this.testArchiveRecoveredEditsWalDirNullOrSame(null);
    }

    @Test
    public void testArchiveRecoveredEditsWalDirSameFsStoreFiles() throws Exception {
        this.testArchiveRecoveredEditsWalDirNullOrSame("/wal-dir");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testArchiveRecoveredEditsWalDirNullOrSame(String walDir) throws Exception {
        String originalRootDir = UTIL.getConfiguration().get("hbase.rootdir");
        try {
            String baseDir = "mockFS://mockFSAuthority:9876/hbase/";
            UTIL.getConfiguration().set("hbase.rootdir", baseDir);
            this.testArchiveStoreFilesDifferentFileSystems(walDir, baseDir, (conf, fs, region, dir, family, list) -> HFileArchiver.archiveRecoveredEdits((Configuration)conf, (FileSystem)fs, (RegionInfo)region, (byte[])family, (Collection)list));
        }
        finally {
            UTIL.getConfiguration().set("hbase.rootdir", originalRootDir);
        }
    }

    @Test(expected=IOException.class)
    public void testArchiveRecoveredEditsWrongFS() throws Exception {
        String baseDir = FSUtils.getRootDir((Configuration)UTIL.getConfiguration()).toString() + "/";
        this.testArchiveStoreFilesDifferentFileSystems("/wal-dir", baseDir, (conf, fs, region, dir, family, list) -> HFileArchiver.archiveRecoveredEdits((Configuration)conf, (FileSystem)fs, (RegionInfo)region, (byte[])family, (Collection)list));
    }

    @Test
    public void testArchiveRecoveredEditsWalDirDifferentFS() throws Exception {
        String walDir = "mockFS://mockFSAuthority:9876/mockDir/wals/";
        this.testArchiveStoreFilesDifferentFileSystems(walDir, walDir, (conf, fs, region, dir, family, list) -> HFileArchiver.archiveRecoveredEdits((Configuration)conf, (FileSystem)fs, (RegionInfo)region, (byte[])family, (Collection)list));
    }

    @Test
    public void testRemoveRegionDirOnArchive() throws Exception {
        TableName tableName = TableName.valueOf((String)this.name.getMethodName());
        UTIL.createTable(tableName, TEST_FAM);
        Admin admin = UTIL.getAdmin();
        List<HRegion> servingRegions = UTIL.getHBaseCluster().getRegions(tableName);
        Assert.assertEquals((long)1L, (long)servingRegions.size());
        HRegion region = servingRegions.get(0);
        UTIL.loadRegion(region, TEST_FAM);
        admin.disableTable(tableName);
        FileSystem fs = UTIL.getTestFileSystem();
        Path rootDir = region.getRegionFileSystem().getTableDir().getParent();
        Path regionDir = FSUtils.getRegionDirFromRootDir((Path)rootDir, (RegionInfo)region.getRegionInfo());
        HFileArchiver.archiveRegion((Configuration)UTIL.getConfiguration(), (FileSystem)fs, (RegionInfo)region.getRegionInfo());
        Path archiveDir = HFileArchiveTestingUtil.getRegionArchiveDir(UTIL.getConfiguration(), region);
        Assert.assertTrue((boolean)fs.exists(archiveDir));
        FileStatus[] stores = fs.listStatus(archiveDir, new PathFilter(){

            public boolean accept(Path p) {
                return !p.getName().contains("recovered.edits");
            }
        });
        Assert.assertTrue((stores.length == 1 ? 1 : 0) != 0);
        FileStatus[] storeFiles = fs.listStatus(stores[0].getPath());
        Assert.assertTrue((storeFiles.length > 0 ? 1 : 0) != 0);
        Assert.assertFalse((boolean)fs.exists(regionDir));
        UTIL.deleteTable(tableName);
    }

    @Test
    public void testDeleteRegionWithNoStoreFiles() throws Exception {
        FileStatus[] storeDirs;
        TableName tableName = TableName.valueOf((String)this.name.getMethodName());
        UTIL.createTable(tableName, TEST_FAM);
        List<HRegion> servingRegions = UTIL.getHBaseCluster().getRegions(tableName);
        Assert.assertEquals((long)1L, (long)servingRegions.size());
        HRegion region = servingRegions.get(0);
        FileSystem fs = region.getRegionFileSystem().getFileSystem();
        Path rootDir = FSUtils.getRootDir((Configuration)fs.getConf());
        Path regionDir = FSUtils.getRegionDirFromRootDir((Path)rootDir, (RegionInfo)region.getRegionInfo());
        FileStatus[] regionFiles = FSUtils.listStatus((FileSystem)fs, (Path)regionDir, null);
        Assert.assertNotNull((String)"No files in the region directory", (Object)regionFiles);
        if (LOG.isDebugEnabled()) {
            ArrayList<Path> files = new ArrayList<Path>();
            for (FileStatus file : regionFiles) {
                files.add(file.getPath());
            }
            LOG.debug("Current files:" + files);
        }
        FSUtils.DirFilter dirFilter = new FSUtils.DirFilter(fs);
        PathFilter nonHidden = new PathFilter((PathFilter)dirFilter){
            final /* synthetic */ PathFilter val$dirFilter;
            {
                this.val$dirFilter = pathFilter;
            }

            public boolean accept(Path file) {
                return this.val$dirFilter.accept(file) && !file.getName().toString().startsWith(".");
            }
        };
        for (FileStatus store : storeDirs = FSUtils.listStatus((FileSystem)fs, (Path)regionDir, (PathFilter)nonHidden)) {
            LOG.debug("Deleting store for test");
            fs.delete(store.getPath(), true);
        }
        HFileArchiver.archiveRegion((Configuration)UTIL.getConfiguration(), (FileSystem)fs, (RegionInfo)region.getRegionInfo());
        Assert.assertFalse((String)("Region directory (" + regionDir + "), still exists."), (boolean)fs.exists(regionDir));
        UTIL.deleteTable(tableName);
    }

    private List<HRegion> initTableForArchivingRegions(TableName tableName) throws IOException {
        byte[][] splitKeys = new byte[][]{Bytes.toBytes((String)"b"), Bytes.toBytes((String)"c"), Bytes.toBytes((String)"d")};
        UTIL.createTable(tableName, TEST_FAM, (byte[][])splitKeys);
        List<HRegion> regions = UTIL.getHBaseCluster().getRegions(tableName);
        Assert.assertEquals((long)4L, (long)regions.size());
        try (Table table = UTIL.getConnection().getTable(tableName);){
            UTIL.loadTable(table, TEST_FAM);
        }
        UTIL.getAdmin().disableTable(tableName);
        return regions;
    }

    @Test
    public void testArchiveRegions() throws Exception {
        TableName tableName = TableName.valueOf((String)this.name.getMethodName());
        List<HRegion> regions = this.initTableForArchivingRegions(tableName);
        FileSystem fs = UTIL.getTestFileSystem();
        Path rootDir = FSUtils.getRootDir((Configuration)UTIL.getConfiguration());
        Path tableDir = FSUtils.getTableDir((Path)rootDir, (TableName)regions.get(0).getRegionInfo().getTable());
        List regionDirList = regions.stream().map(region -> FSUtils.getRegionDirFromTableDir((Path)tableDir, (RegionInfo)region.getRegionInfo())).collect(Collectors.toList());
        HFileArchiver.archiveRegions((Configuration)UTIL.getConfiguration(), (FileSystem)fs, (Path)rootDir, (Path)tableDir, regionDirList);
        for (HRegion region2 : regions) {
            Path archiveDir = HFileArchiveTestingUtil.getRegionArchiveDir(UTIL.getConfiguration(), region2);
            Assert.assertTrue((boolean)fs.exists(archiveDir));
            FileStatus[] stores = fs.listStatus(archiveDir, p -> !p.getName().contains("recovered.edits"));
            Assert.assertTrue((stores.length == 1 ? 1 : 0) != 0);
            FileStatus[] storeFiles = fs.listStatus(stores[0].getPath());
            Assert.assertTrue((storeFiles.length > 0 ? 1 : 0) != 0);
        }
        for (Path regionDir : regionDirList) {
            Assert.assertFalse((boolean)fs.exists(regionDir));
        }
        UTIL.deleteTable(tableName);
    }

    @Test(expected=IOException.class)
    public void testArchiveRegionsWhenPermissionDenied() throws Exception {
        TableName tableName = TableName.valueOf((String)this.name.getMethodName());
        List<HRegion> regions = this.initTableForArchivingRegions(tableName);
        Path rootDir = FSUtils.getRootDir((Configuration)UTIL.getConfiguration());
        Path tableDir = FSUtils.getTableDir((Path)rootDir, (TableName)regions.get(0).getRegionInfo().getTable());
        List regionDirList = regions.stream().map(region -> FSUtils.getRegionDirFromTableDir((Path)tableDir, (RegionInfo)region.getRegionInfo())).collect(Collectors.toList());
        UserGroupInformation ugi = UserGroupInformation.createUserForTesting((String)"foo1234", (String[])new String[]{"group1"});
        try {
            ugi.doAs(() -> {
                FileSystem fs = UTIL.getTestFileSystem();
                HFileArchiver.archiveRegions((Configuration)UTIL.getConfiguration(), (FileSystem)fs, (Path)rootDir, (Path)tableDir, (List)regionDirList);
                return null;
            });
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)e.getCause().getMessage().contains("Permission denied"));
            throw e;
        }
        finally {
            UTIL.deleteTable(tableName);
        }
    }

    @Test
    public void testArchiveOnTableDelete() throws Exception {
        TableName tableName = TableName.valueOf((String)this.name.getMethodName());
        UTIL.createTable(tableName, TEST_FAM);
        List<HRegion> servingRegions = UTIL.getHBaseCluster().getRegions(tableName);
        Assert.assertEquals((long)1L, (long)servingRegions.size());
        HRegion region = servingRegions.get(0);
        HRegionServer hrs = UTIL.getRSForFirstRegionInTable(tableName);
        FileSystem fs = hrs.getFileSystem();
        LOG.debug("-------Loading table");
        UTIL.loadRegion(region, TEST_FAM);
        List regions = hrs.getRegions(tableName);
        Assert.assertEquals((String)"More that 1 region for test table.", (long)1L, (long)regions.size());
        region = (HRegion)regions.get(0);
        region.waitForFlushesAndCompactions();
        UTIL.getAdmin().disableTable(tableName);
        LOG.debug("Disabled table");
        this.clearArchiveDirectory();
        byte[][] columns = (byte[][])region.getTableDescriptor().getColumnFamilyNames().toArray((T[])new byte[0][]);
        List storeFiles = region.getStoreFileList(columns);
        UTIL.deleteTable(tableName);
        LOG.debug("Deleted table");
        this.assertArchiveFiles(fs, storeFiles, 30000L);
    }

    private void assertArchiveFiles(FileSystem fs, List<String> storeFiles, long timeout) throws IOException {
        int i;
        long end = System.currentTimeMillis() + timeout;
        Path archiveDir = HFileArchiveUtil.getArchivePath((Configuration)UTIL.getConfiguration());
        List<Object> archivedFiles = new ArrayList();
        while (System.currentTimeMillis() < end && (archivedFiles = this.getAllFileNames(fs, archiveDir)).size() < storeFiles.size()) {
        }
        Collections.sort(storeFiles);
        Collections.sort(archivedFiles);
        LOG.debug("Store files:");
        for (i = 0; i < storeFiles.size(); ++i) {
            LOG.debug(i + " - " + storeFiles.get(i));
        }
        LOG.debug("Archive files:");
        for (i = 0; i < archivedFiles.size(); ++i) {
            LOG.debug(i + " - " + (String)archivedFiles.get(i));
        }
        Assert.assertTrue((String)"Archived files are missing some of the store files!", (boolean)archivedFiles.containsAll(storeFiles));
    }

    @Test
    public void testArchiveOnTableFamilyDelete() throws Exception {
        TableName tableName = TableName.valueOf((String)this.name.getMethodName());
        UTIL.createTable(tableName, (byte[][])new byte[][]{TEST_FAM, Bytes.toBytes((String)"fam2")});
        List<HRegion> servingRegions = UTIL.getHBaseCluster().getRegions(tableName);
        Assert.assertEquals((long)1L, (long)servingRegions.size());
        HRegion region = servingRegions.get(0);
        HRegionServer hrs = UTIL.getRSForFirstRegionInTable(tableName);
        FileSystem fs = hrs.getFileSystem();
        LOG.debug("-------Loading table");
        UTIL.loadRegion(region, TEST_FAM);
        List regions = hrs.getRegions(tableName);
        Assert.assertEquals((String)"More that 1 region for test table.", (long)1L, (long)regions.size());
        region = (HRegion)regions.get(0);
        region.waitForFlushesAndCompactions();
        UTIL.getAdmin().disableTable(tableName);
        LOG.debug("Disabled table");
        this.clearArchiveDirectory();
        byte[][] columns = (byte[][])region.getTableDescriptor().getColumnFamilyNames().toArray((T[])new byte[0][]);
        List storeFiles = region.getStoreFileList(columns);
        UTIL.getAdmin().deleteColumnFamily(tableName, TEST_FAM);
        this.assertArchiveFiles(fs, storeFiles, 30000L);
        UTIL.deleteTable(tableName);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testCleaningRace() throws Exception {
        long TEST_TIME = 20000L;
        ChoreService choreService = new ChoreService("TEST_SERVER_NAME");
        Configuration conf = UTIL.getMiniHBaseCluster().getMaster().getConfiguration();
        Path rootDir = UTIL.getDataTestDirOnTestFS("testCleaningRace");
        FileSystem fs = UTIL.getTestFileSystem();
        Path archiveDir = new Path(rootDir, "archive");
        Path regionDir = new Path(FSUtils.getTableDir((Path)new Path("./"), (TableName)TableName.valueOf((String)this.name.getMethodName())), "abcdef");
        Path familyDir = new Path(regionDir, "cf");
        Path sourceRegionDir = new Path(rootDir, regionDir);
        fs.mkdirs(sourceRegionDir);
        StoppableImplementation stoppable = new StoppableImplementation();
        HFileCleaner cleaner = new HFileCleaner(1, (Stoppable)stoppable, conf, fs, archiveDir, POOL);
        try {
            choreService.scheduleChore((ScheduledChore)cleaner);
            long startTime = System.currentTimeMillis();
            long fid = 0L;
            while (System.currentTimeMillis() - startTime < 20000L) {
                Path file = new Path(familyDir, String.valueOf(fid));
                Path sourceFile = new Path(rootDir, file);
                Path archiveFile = new Path(archiveDir, file);
                fs.createNewFile(sourceFile);
                try {
                    HFileArchiver.archiveRegion((FileSystem)fs, (Path)rootDir, (Path)sourceRegionDir.getParent(), (Path)sourceRegionDir);
                    LOG.debug("hfile=" + fid + " should be in the archive");
                    Assert.assertTrue((boolean)fs.exists(archiveFile));
                    Assert.assertFalse((boolean)fs.exists(sourceFile));
                }
                catch (IOException e) {
                    LOG.debug("hfile=" + fid + " should be in the source location");
                    Assert.assertFalse((boolean)fs.exists(archiveFile));
                    Assert.assertTrue((boolean)fs.exists(sourceFile));
                    fs.delete(sourceFile, false);
                }
                ++fid;
            }
        }
        finally {
            stoppable.stop("test end");
            cleaner.cancel(true);
            choreService.shutdown();
            fs.delete(rootDir, true);
        }
    }

    @Test
    public void testArchiveRegionTableAndRegionDirsNull() throws IOException {
        Path rootDir = UTIL.getDataTestDirOnTestFS("testCleaningRace");
        FileSystem fileSystem = UTIL.getTestFileSystem();
        Assert.assertFalse((boolean)HFileArchiver.archiveRegion((FileSystem)fileSystem, (Path)rootDir, null, null));
    }

    @Test
    public void testArchiveRegionWithTableDirNull() throws IOException {
        Path regionDir = new Path(FSUtils.getTableDir((Path)new Path("./"), (TableName)TableName.valueOf((String)this.name.getMethodName())), "xyzabc");
        Path familyDir = new Path(regionDir, "rd");
        Path rootDir = UTIL.getDataTestDirOnTestFS("testCleaningRace");
        Path file = new Path(familyDir, "1");
        Path sourceFile = new Path(rootDir, file);
        FileSystem fileSystem = UTIL.getTestFileSystem();
        fileSystem.createNewFile(sourceFile);
        Path sourceRegionDir = new Path(rootDir, regionDir);
        fileSystem.mkdirs(sourceRegionDir);
        Assert.assertFalse((boolean)HFileArchiver.archiveRegion((FileSystem)fileSystem, (Path)rootDir, null, (Path)sourceRegionDir));
        Assert.assertFalse((boolean)fileSystem.exists(sourceRegionDir));
    }

    @Test
    public void testArchiveRegionWithRegionDirNull() throws IOException {
        Path regionDir = new Path(FSUtils.getTableDir((Path)new Path("./"), (TableName)TableName.valueOf((String)this.name.getMethodName())), "elgn4nf");
        Path familyDir = new Path(regionDir, "rdar");
        Path rootDir = UTIL.getDataTestDirOnTestFS("testCleaningRace");
        Path file = new Path(familyDir, "2");
        Path sourceFile = new Path(rootDir, file);
        FileSystem fileSystem = UTIL.getTestFileSystem();
        fileSystem.createNewFile(sourceFile);
        Path sourceRegionDir = new Path(rootDir, regionDir);
        fileSystem.mkdirs(sourceRegionDir);
        Assert.assertFalse((boolean)HFileArchiver.archiveRegion((FileSystem)fileSystem, (Path)rootDir, (Path)sourceRegionDir.getParent(), null));
        Assert.assertTrue((boolean)fileSystem.exists(sourceRegionDir));
        fileSystem.delete(sourceRegionDir, true);
    }

    private void clearArchiveDirectory() throws IOException {
        UTIL.getTestFileSystem().delete(new Path(UTIL.getDefaultRootDirPath(), "archive"), true);
    }

    private List<String> getAllFileNames(FileSystem fs, Path archiveDir) throws IOException {
        FileStatus[] files = FSUtils.listStatus((FileSystem)fs, (Path)archiveDir, (PathFilter)new PathFilter(){

            public boolean accept(Path p) {
                return !p.getName().contains("recovered.edits");
            }
        });
        return this.recurseOnFiles(fs, files, new ArrayList<String>());
    }

    private List<String> recurseOnFiles(FileSystem fs, FileStatus[] files, List<String> fileNames) throws IOException {
        if (files == null || files.length == 0) {
            return fileNames;
        }
        for (FileStatus file : files) {
            if (file.isDirectory()) {
                this.recurseOnFiles(fs, FSUtils.listStatus((FileSystem)fs, (Path)file.getPath(), null), fileNames);
                continue;
            }
            fileNames.add(file.getPath().getName());
        }
        return fileNames;
    }

    @FunctionalInterface
    private static interface ArchivingFunction<Configuration, FS, Region, Dir, Family, Files> {
        public void apply(Configuration var1, FS var2, Region var3, Dir var4, Family var5, Files var6) throws IOException;
    }
}

