/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.resource;

import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.annotation.Internal;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.configuration.PipelineOptions;
import org.apache.flink.configuration.ReadableConfig;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.shaded.guava31.com.google.common.io.Files;
import org.apache.flink.table.api.TableConfig;
import org.apache.flink.table.api.ValidationException;
import org.apache.flink.table.api.config.TableConfigOptions;
import org.apache.flink.table.resource.ResourceType;
import org.apache.flink.table.resource.ResourceUri;
import org.apache.flink.util.ExceptionUtils;
import org.apache.flink.util.FileUtils;
import org.apache.flink.util.FlinkUserCodeClassLoaders;
import org.apache.flink.util.JarUtils;
import org.apache.flink.util.MutableURLClassLoader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class ResourceManager
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(ResourceManager.class);
    private static final String JAR_SUFFIX = "jar";
    private static final String FILE_SCHEME = "file";
    private final Path localResourceDir;
    protected final Map<ResourceUri, URL> resourceInfos;
    protected final MutableURLClassLoader userClassLoader;

    public static ResourceManager createResourceManager(URL[] urls, ClassLoader parent, ReadableConfig config) {
        MutableURLClassLoader mutableURLClassLoader = FlinkUserCodeClassLoaders.create((URL[])urls, (ClassLoader)parent, (ReadableConfig)config);
        return new ResourceManager(config, mutableURLClassLoader);
    }

    public ResourceManager(ReadableConfig config, MutableURLClassLoader userClassLoader) {
        this.localResourceDir = new Path((String)config.get(TableConfigOptions.RESOURCES_DOWNLOAD_DIR), String.format("flink-table-%s", UUID.randomUUID()));
        this.resourceInfos = new HashMap<ResourceUri, URL>();
        this.userClassLoader = userClassLoader;
    }

    public void registerJarResources(List<ResourceUri> resourceUris) throws IOException {
        this.registerResources(this.prepareStagingResources(resourceUris, ResourceType.JAR, true, url -> {
            try {
                JarUtils.checkJarFile((URL)url);
            }
            catch (IOException e) {
                throw new ValidationException(String.format("Failed to register jar resource [%s]", url), (Throwable)e);
            }
        }), true);
    }

    public String registerFileResource(ResourceUri resourceUri) throws IOException {
        Map<ResourceUri, URL> stagingResources = this.prepareStagingResources(Collections.singletonList(resourceUri), ResourceType.FILE, false, url -> {});
        this.registerResources(stagingResources, false);
        return this.resourceInfos.get(new ArrayList<ResourceUri>(stagingResources.keySet()).get(0)).getPath();
    }

    public URLClassLoader getUserClassLoader() {
        return this.userClassLoader;
    }

    public Map<ResourceUri, URL> getResources() {
        return Collections.unmodifiableMap(this.resourceInfos);
    }

    public Set<URL> getLocalJarResources() {
        return this.resourceInfos.entrySet().stream().filter(entry -> ResourceType.JAR.equals((Object)((ResourceUri)entry.getKey()).getResourceType())).map(Map.Entry::getValue).collect(Collectors.toSet());
    }

    public void addJarConfiguration(TableConfig tableConfig) {
        List jars = this.getLocalJarResources().stream().map(URL::toString).collect(Collectors.toList());
        if (jars.isEmpty()) {
            return;
        }
        Set jarFiles = tableConfig.getOptional(PipelineOptions.JARS).map(LinkedHashSet::new).orElseGet(LinkedHashSet::new);
        jarFiles.addAll(jars);
        tableConfig.set(PipelineOptions.JARS, new ArrayList(jarFiles));
    }

    @Override
    public void close() throws IOException {
        this.resourceInfos.clear();
        IOException exception = null;
        try {
            this.userClassLoader.close();
        }
        catch (IOException e) {
            LOG.debug("Error while closing user classloader.", (Throwable)e);
            exception = e;
        }
        FileSystem fileSystem = FileSystem.getLocalFileSystem();
        try {
            if (fileSystem.exists(this.localResourceDir)) {
                fileSystem.delete(this.localResourceDir, true);
            }
        }
        catch (IOException ioe) {
            LOG.debug(String.format("Error while delete directory [%s].", this.localResourceDir), (Throwable)ioe);
            exception = (IOException)ExceptionUtils.firstOrSuppressed((Throwable)ioe, (Throwable)exception);
        }
        if (exception != null) {
            throw exception;
        }
    }

    public boolean exists(Path filePath) throws IOException {
        return filePath.getFileSystem().exists(filePath);
    }

    public void syncFileResource(ResourceUri resourceUri, Consumer<String> resourceGenerator) throws IOException {
        Path targetPath = new Path(resourceUri.getUri());
        boolean remote = this.isRemotePath(targetPath);
        String localPath = remote ? this.getResourceLocalPath(targetPath).getPath() : this.getURLFromPath(targetPath).getPath();
        resourceGenerator.accept(localPath);
        if (remote) {
            if (this.exists(targetPath)) {
                targetPath.getFileSystem().delete(targetPath, false);
            }
            FileUtils.copy((Path)new Path(localPath), (Path)targetPath, (boolean)false);
        }
    }

    protected void checkPath(Path path, ResourceType expectedType) throws IOException {
        String fileExtension;
        FileSystem fs = FileSystem.getUnguardedFileSystem((URI)path.toUri());
        if (!fs.exists(path)) {
            throw new FileNotFoundException(String.format("%s resource [%s] not found.", expectedType.name().toLowerCase(), path));
        }
        if (fs.getFileStatus(path).isDir()) {
            throw new ValidationException(String.format("The registering or unregistering %s resource [%s] is a directory that is not allowed.", expectedType.name().toLowerCase(), path));
        }
        if (expectedType == ResourceType.JAR && !(fileExtension = Files.getFileExtension((String)path.getName())).toLowerCase().endsWith(JAR_SUFFIX)) {
            throw new ValidationException(String.format("The registering or unregistering jar resource [%s] must ends with '.jar' suffix.", path));
        }
    }

    @VisibleForTesting
    URL downloadResource(Path remotePath, boolean executable) throws IOException {
        Path localPath = this.getResourceLocalPath(remotePath);
        try {
            FileUtils.copy((Path)remotePath, (Path)localPath, (boolean)executable);
            LOG.info("Download resource [{}] to local path [{}] successfully.", (Object)remotePath, (Object)localPath);
        }
        catch (IOException e) {
            throw new IOException(String.format("Failed to download resource [%s] to local path [%s].", remotePath, localPath), e);
        }
        return this.getURLFromPath(localPath);
    }

    @VisibleForTesting
    protected URL getURLFromPath(Path path) throws IOException {
        if (path.toUri().getScheme() == null) {
            path = path.makeQualified(FileSystem.getLocalFileSystem());
        }
        return path.toUri().toURL();
    }

    @VisibleForTesting
    Path getLocalResourceDir() {
        return this.localResourceDir;
    }

    @VisibleForTesting
    boolean isRemotePath(Path path) {
        String scheme = path.toUri().getScheme();
        if (scheme == null) {
            return !FILE_SCHEME.equalsIgnoreCase(FileSystem.getDefaultFsUri().getScheme());
        }
        return !FILE_SCHEME.equalsIgnoreCase(scheme);
    }

    private Path getResourceLocalPath(Path remotePath) {
        String fileName = remotePath.getName();
        String fileExtension = Files.getFileExtension((String)fileName);
        String fileNameWithUUID = StringUtils.isEmpty((CharSequence)fileExtension) ? String.format("%s-%s", fileName, UUID.randomUUID()) : String.format("%s-%s.%s", Files.getNameWithoutExtension((String)fileName), UUID.randomUUID(), fileExtension);
        return new Path(this.localResourceDir, fileNameWithUUID);
    }

    private void checkResources(List<ResourceUri> resourceUris, ResourceType expectedType) throws IOException {
        if (resourceUris.stream().anyMatch(resourceUri -> expectedType != resourceUri.getResourceType())) {
            throw new ValidationException(String.format("Expect the resource type to be %s, but encounter a resource %s.", expectedType.name().toLowerCase(), resourceUris.stream().filter(resourceUri -> expectedType != resourceUri.getResourceType()).findFirst().map(resourceUri -> String.format("[%s] with type %s", resourceUri.getUri(), resourceUri.getResourceType().name().toLowerCase())).get()));
        }
        for (ResourceUri resourceUri2 : resourceUris) {
            this.checkPath(new Path(resourceUri2.getUri()), expectedType);
        }
    }

    private Map<ResourceUri, URL> prepareStagingResources(List<ResourceUri> resourceUris, ResourceType expectedType, boolean executable, Consumer<URL> resourceChecker) throws IOException {
        this.checkResources(resourceUris, expectedType);
        HashMap<ResourceUri, URL> stagingResourceLocalURLs = new HashMap<ResourceUri, URL>();
        boolean supportOverwrite = !executable;
        for (ResourceUri resourceUri : resourceUris) {
            URL localUrl;
            if (this.resourceInfos.containsKey(resourceUri) && this.resourceInfos.get(resourceUri) != null && !supportOverwrite) {
                LOG.info("Resource [{}] has been registered, overwriting of registered resource is not supported in the current version, skipping.", (Object)resourceUri.getUri());
                continue;
            }
            Path path = new Path(resourceUri.getUri());
            if (this.isRemotePath(path)) {
                localUrl = this.downloadResource(path, executable);
            } else {
                localUrl = this.getURLFromPath(path);
                resourceUri = new ResourceUri(expectedType, localUrl.getPath());
            }
            resourceChecker.accept(localUrl);
            stagingResourceLocalURLs.put(resourceUri, localUrl);
        }
        return stagingResourceLocalURLs;
    }

    private void registerResources(Map<ResourceUri, URL> stagingResources, boolean addToClassLoader) {
        stagingResources.forEach((resourceUri, url) -> {
            if (addToClassLoader) {
                this.userClassLoader.addURL(url);
                LOG.info("Added {} resource [{}] to class path.", (Object)resourceUri.getResourceType().name(), url);
            }
            this.resourceInfos.put((ResourceUri)resourceUri, (URL)url);
            LOG.info("Register resource [{}] successfully.", (Object)resourceUri.getUri());
        });
    }
}

