/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.felix.framework.util;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * Abstract base class for CopyOnWrite {@link Map} implementations that delegate to an
 * internal map.
 */
abstract class AbstractCopyOnWriteMap implements Map, ConcurrentMap, Serializable
{
    private static final long serialVersionUID = 4508989182041753878L;

    private volatile Map delegate;
    private final transient EntrySet entrySet = new EntrySet();
    private final transient KeySet keySet = new KeySet();
    private final transient Values values = new Values();
    private final transient Object lock = new Object();

    static void notNull(String name, Object check)
    {
        if (check == null)
        {
            throw new IllegalArgumentException(name + " cannot be null");
        }
    }

    /**
     * Create a new {@link AbstractCopyOnWriteMap} with the supplied {@link Map} to
     * initialize the values.
     * 
     * @param map the initial map to initialize with
     */
    protected AbstractCopyOnWriteMap(Map map)
    {
        notNull("map", map);
        this.delegate = map;
    }

    /**
     * Create a new empty {@link AbstractCopyOnWriteMap} with the supplied {@link Map} to
     * initialize the values.
     * 
     * @param map the initial map to initialize with
     */
    protected AbstractCopyOnWriteMap()
    {
        this(Collections.EMPTY_MAP);
    }

    /**
     * Copy function, implemented by sub-classes.
     * 
     * @param <N> the map to copy and return.
     * @param map the initial values of the newly created map.
     * @return a new map. Will never be modified after construction.
     */
    abstract Map copy(Map map);

    //
    // mutable operations
    //

    public final void clear()
    {
        synchronized (lock)
        {
            final Map map = copy();
            map.clear();
            set(map);
        }
    }

    public final Object remove(final Object key)
    {
        synchronized (lock)
        {
            // short circuit if key doesn't exist
            if (!delegate.containsKey(key))
            {
                return null;
            }
            final Map map = copy();
            final Object result = map.remove(key);
            set(map);
            return result;
        }
    }

    public boolean remove(final Object key, final Object value)
    {
        synchronized (lock)
        {
            if (delegate.containsKey(key) && equals(value, delegate.get(key)))
            {
                final Map map = copy();
                map.remove(key);
                set(map);
                return true;
            }
            else
            {
                return false;
            }
        }
    }

    public boolean replace(final Object key, final Object oldValue, final Object newValue)
    {
        synchronized (lock)
        {
            if (!delegate.containsKey(key) || !equals(oldValue, delegate.get(key)))
            {
                return false;
            }
            final Map map = copy();
            map.put(key, newValue);
            set(map);
            return true;
        }
    };

    public Object replace(final Object key, final Object value)
    {
        synchronized (lock)
        {
            if (!delegate.containsKey(key))
            {
                return null;
            }
            final Map map = copy();
            try
            {
                return map.put(key, value);
            }
            finally
            {
                set(map);
            }
        }
    }

    public final Object put(final Object key, final Object value)
    {
        synchronized (lock)
        {
            final Map map = copy();
            notNull("copy", map);
            final Object result = map.put(key, value);
            set(map);
            return result;
        }
    }

    public Object putIfAbsent(final Object key, final Object value)
    {
        synchronized (lock)
        {
            if (!delegate.containsKey(key))
            {
                final Map map = copy();
                final Object result = map.put(key, value);
                set(map);
                return result;
            }
            return delegate.get(key);
        }
    }

    public final void putAll(final Map t)
    {
        synchronized (lock)
        {
            final Map map = copy();
            map.putAll(t);
            set(map);
        }
    }

    protected Map copy()
    {
        synchronized (lock)
        {
            return copy(delegate);
        }
    }

    //@GuardedBy("lock")
    protected void set(final Map map)
    {
        delegate = map;
    }

    //
    // Collection views
    //

    public final Set entrySet()
    {
        return entrySet;
    }

    public final Set keySet()
    {
        return keySet;
    }

    public final Collection values()
    {
        return values;
    }

    //
    // delegate operations
    //

    public final boolean containsKey(final Object key)
    {
        return delegate.containsKey(key);
    }

    public final boolean containsValue(final Object value)
    {
        return delegate.containsValue(value);
    }

    public final Object get(final Object key)
    {
        return delegate.get(key);
    }

    public final boolean isEmpty()
    {
        return delegate.isEmpty();
    }

    public final int size()
    {
        return delegate.size();
    }

    public final boolean equals(final Object o)
    {
        return delegate.equals(o);
    }

    public final int hashCode()
    {
        return delegate.hashCode();
    }

    protected final Map getDelegate()
    {
        return delegate;
    }

    public String toString()
    {
        return delegate.toString();
    }

    //
    // inner classes
    //

    private class KeySet extends CollectionView implements Set
    {

        Collection getDelegate()
        {
            return delegate.keySet();
        }

        //
        // mutable operations
        //

        public void clear()
        {
            synchronized (lock)
            {
                final Map map = copy();
                map.keySet().clear();
                set(map);
            }
        }

        public boolean remove(final Object o)
        {
            return AbstractCopyOnWriteMap.this.remove(o) != null;
        }

        public boolean removeAll(final Collection c)
        {
            synchronized (lock)
            {
                final Map map = copy();
                final boolean result = map.keySet().removeAll(c);
                set(map);
                return result;
            }
        }

        public boolean retainAll(final Collection c)
        {
            synchronized (lock)
            {
                final Map map = copy();
                final boolean result = map.keySet().retainAll(c);
                set(map);
                return result;
            }
        }
    }

    private final class Values extends CollectionView
    {
        Collection getDelegate()
        {
            return delegate.values();
        }

        public void clear()
        {
            synchronized (lock)
            {
                final Map map = copy();
                map.values().clear();
                set(map);
            }
        }

        public boolean remove(final Object o)
        {
            synchronized (lock)
            {
                if (!contains(o))
                {
                    return false;
                }
                final Map map = copy();
                final boolean result = map.values().remove(o);
                set(map);
                return result;
            }
        }

        public boolean removeAll(final Collection c)
        {
            synchronized (lock)
            {
                final Map map = copy();
                final boolean result = map.values().removeAll(c);
                set(map);
                return result;
            }
        }

        public boolean retainAll(final Collection c)
        {
            synchronized (lock)
            {
                final Map map = copy();
                final boolean result = map.values().retainAll(c);
                set(map);
                return result;
            }
        }
    }

    private class EntrySet extends CollectionView implements Set
    {
        Collection getDelegate()
        {
            return delegate.entrySet();
        }

        public void clear()
        {
            synchronized (lock)
            {
                final Map map = copy();
                map.entrySet().clear();
                set(map);
            }
        }

        public boolean remove(final Object o)
        {
            synchronized (lock)
            {
                if (!contains(o))
                {
                    return false;
                }
                final Map map = copy();
                final boolean result = map.entrySet().remove(o);
                set(map);
                return result;
            }
        }

        public boolean removeAll(final Collection c)
        {
            synchronized (lock)
            {
                final Map map = copy();
                final boolean result = map.entrySet().removeAll(c);
                set(map);
                return result;
            }
        }

        public boolean retainAll(final Collection c)
        {
            synchronized (lock)
            {
                final Map map = copy();
                final boolean result = map.entrySet().retainAll(c);
                set(map);
                return result;
            }
        }
    }

    private static class UnmodifiableIterator implements Iterator
    {
        private final Iterator delegate;

        public UnmodifiableIterator(final Iterator delegate)
        {
            this.delegate = delegate;
        }

        public boolean hasNext()
        {
            return delegate.hasNext();
        }

        public Object next()
        {
            return delegate.next();
        }

        public void remove()
        {
            throw new UnsupportedOperationException();
        }
    }

    protected static abstract class CollectionView implements Collection
    {
        abstract Collection getDelegate();

        //
        // delegate operations
        //

        public final boolean contains(final Object o)
        {
            return getDelegate().contains(o);
        }

        public final boolean containsAll(final Collection c)
        {
            return getDelegate().containsAll(c);
        }

        public final Iterator iterator()
        {
            return new UnmodifiableIterator(getDelegate().iterator());
        }

        public final boolean isEmpty()
        {
            return getDelegate().isEmpty();
        }

        public final int size()
        {
            return getDelegate().size();
        }

        public final Object[] toArray()
        {
            return getDelegate().toArray();
        }

        public final Object[] toArray(final Object[] a)
        {
            return getDelegate().toArray(a);
        }

        public int hashCode()
        {
            return getDelegate().hashCode();
        }

        public boolean equals(final Object obj)
        {
            return getDelegate().equals(obj);
        }

        public String toString()
        {
            return getDelegate().toString();
        }

        //
        // unsupported operations
        //

        public final boolean add(final Object o)
        {
            throw new UnsupportedOperationException();
        }

        public final boolean addAll(final Collection c)
        {
            throw new UnsupportedOperationException();
        }
    }

    private boolean equals(final Object o1, final Object o2)
    {
        if (o1 == null)
        {
            return o2 == null;
        }
        return o1.equals(o2);
    }
}
