/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.support.cache;

import java.util.Collection;
import java.util.Deque;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

public class SimpleLRUCache<K, V>
implements Map<K, V> {
    static final float DEFAULT_LOAD_FACTOR = 0.75f;
    static final int MINIMUM_QUEUE_SIZE = 128;
    private final AtomicBoolean eviction = new AtomicBoolean();
    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    private final int maximumCacheSize;
    private final AtomicReference<Deque<Map.Entry<K, ValueHolder<V>>>> lastChanges = new AtomicReference(new ConcurrentLinkedDeque());
    private final AtomicInteger totalChanges = new AtomicInteger();
    private final Consumer<V> evict;
    private final AtomicLong sequence = new AtomicLong();
    private final Map<K, ValueHolder<V>> delegate;

    public SimpleLRUCache(int initialCapacity, int maximumCacheSize, Consumer<V> evicted) {
        if (maximumCacheSize <= 0) {
            throw new IllegalArgumentException("The maximum cache size must be greater than 0");
        }
        this.delegate = new ConcurrentHashMap<K, ValueHolder<V>>(initialCapacity, 0.75f);
        this.maximumCacheSize = maximumCacheSize;
        this.evict = Objects.requireNonNull(evicted);
    }

    private ValueHolder<V> addChange(OperationContext<K, V> context, Function<? super K, ? extends V> mappingFunction) {
        Object key = context.key;
        V value = mappingFunction.apply(key);
        if (value == null) {
            return null;
        }
        ValueHolder<V> holder = this.newValue(value);
        this.lastChanges.get().add(Map.entry(key, holder));
        this.totalChanges.incrementAndGet();
        return holder;
    }

    @Override
    public int size() {
        return this.delegate.size();
    }

    @Override
    public boolean isEmpty() {
        return this.delegate.isEmpty();
    }

    @Override
    public boolean containsKey(Object key) {
        return this.delegate.containsKey(key);
    }

    @Override
    public boolean containsValue(Object value) {
        if (value == null) {
            throw new NullPointerException();
        }
        return this.delegate.values().stream().map(ValueHolder::get).anyMatch(v -> Objects.equals(v, value));
    }

    @Override
    public V get(Object key) {
        return this.extractValue(this.delegate.get(key));
    }

    @Override
    public boolean remove(Object key, Object value) {
        if (key == null || value == null) {
            throw new NullPointerException();
        }
        Object keyK = key;
        try (OperationContext context = new OperationContext(this, keyK);){
            this.delegate.compute(keyK, (k, v) -> {
                V extractedValue = this.extractValue((ValueHolder<V>)v);
                if (Objects.equals(value, extractedValue)) {
                    context.result = extractedValue;
                    return null;
                }
                return v;
            });
            boolean bl = context.result != null;
            return bl;
        }
    }

    @Override
    public V put(K key, V value) {
        if (key == null || value == null) {
            throw new NullPointerException();
        }
        try (OperationContext context = new OperationContext(this, key);){
            this.delegate.compute(key, (k, v) -> {
                context.result = this.extractValue((ValueHolder<V>)v);
                return this.addChange(context, x -> value);
            });
            Object v2 = context.result;
            return v2;
        }
    }

    @Override
    public V remove(Object key) {
        return this.extractValue(this.delegate.remove(key));
    }

    @Override
    public V putIfAbsent(K key, V value) {
        if (key == null || value == null) {
            throw new NullPointerException();
        }
        try (OperationContext context = new OperationContext(this, key);){
            this.delegate.compute(key, (k, v) -> {
                context.result = this.extractValue((ValueHolder<V>)v);
                if (v != null) {
                    return v;
                }
                return this.addChange(context, x -> value);
            });
            Object v2 = context.result;
            return v2;
        }
    }

    @Override
    public V computeIfAbsent(K key, Function<? super K, ? extends V> mappingFunction) {
        if (key == null || mappingFunction == null) {
            throw new NullPointerException();
        }
        try (OperationContext context = new OperationContext(this, key);){
            V v = this.extractValue(this.delegate.computeIfAbsent(key, k -> this.addChange(context, mappingFunction)));
            return v;
        }
    }

    @Override
    public V computeIfPresent(K key, BiFunction<? super K, ? super V, ? extends V> remappingFunction) {
        if (key == null || remappingFunction == null) {
            throw new NullPointerException();
        }
        try (OperationContext context = new OperationContext(this, key);){
            V v2 = this.extractValue(this.delegate.computeIfPresent(key, (k, v) -> this.addChange(context, x -> remappingFunction.apply((K)x, (V)this.extractValue((ValueHolder<V>)v)))));
            return v2;
        }
    }

    @Override
    public V compute(K key, BiFunction<? super K, ? super V, ? extends V> remappingFunction) {
        if (key == null || remappingFunction == null) {
            throw new NullPointerException();
        }
        try (OperationContext context = new OperationContext(this, key);){
            V v2 = this.extractValue(this.delegate.compute(key, (k, v) -> this.addChange(context, x -> remappingFunction.apply((K)x, (V)this.extractValue((ValueHolder<V>)v)))));
            return v2;
        }
    }

    @Override
    public V merge(K key, V value, BiFunction<? super V, ? super V, ? extends V> remappingFunction) {
        if (key == null || value == null || remappingFunction == null) {
            throw new NullPointerException();
        }
        try (OperationContext context = new OperationContext(this, key);){
            V v = this.extractValue(this.delegate.compute(key, (k, oldValue) -> {
                Object newValue = oldValue == null ? value : remappingFunction.apply((V)oldValue.get(), (V)value);
                return this.addChange(context, x -> newValue);
            }));
            return v;
        }
    }

    @Override
    public boolean replace(K key, V oldValue, V newValue) {
        if (key == null || oldValue == null || newValue == null) {
            throw new NullPointerException();
        }
        try (OperationContext context = new OperationContext(this, key);){
            this.delegate.computeIfPresent(key, (k, v) -> {
                if (Objects.equals(oldValue, this.extractValue((ValueHolder<V>)v))) {
                    ValueHolder<Object> result = this.addChange(context, x -> newValue);
                    context.result = this.extractValue(result);
                    return result;
                }
                return v;
            });
            boolean bl = context.result != null && Objects.equals(context.result, newValue);
            return bl;
        }
    }

    @Override
    public V replace(K key, V value) {
        if (key == null || value == null) {
            throw new NullPointerException();
        }
        try (OperationContext context = new OperationContext(this, key);){
            this.delegate.computeIfPresent(key, (k, v) -> {
                context.result = this.extractValue((ValueHolder<V>)v);
                return this.addChange(context, x -> value);
            });
            Object v2 = context.result;
            return v2;
        }
    }

    @Override
    public void putAll(Map<? extends K, ? extends V> m) {
        for (Map.Entry<K, V> e : m.entrySet()) {
            this.put(e.getKey(), e.getValue());
        }
    }

    @Override
    public void clear() {
        this.lock.writeLock().lock();
        try {
            this.lastChanges.getAndSet(new ConcurrentLinkedDeque());
            this.totalChanges.set(0);
            this.delegate.clear();
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public Set<K> keySet() {
        return this.delegate.keySet();
    }

    @Override
    public Collection<V> values() {
        return this.delegate.values().stream().map(ValueHolder::get).toList();
    }

    @Override
    public void replaceAll(BiFunction<? super K, ? super V, ? extends V> function) {
        if (function == null) {
            throw new NullPointerException();
        }
        for (Map.Entry<K, V> e : this.entrySet()) {
            K key = e.getKey();
            Object value = e.getValue();
            try (OperationContext context = new OperationContext(this, key);){
                this.delegate.computeIfPresent(key, (k, v) -> this.addChange(context, x -> function.apply((Object)x, (Object)value)));
            }
        }
    }

    @Override
    public Set<Map.Entry<K, V>> entrySet() {
        return this.delegate.entrySet().stream().map(entry -> new CacheEntry(this, entry.getKey(), ((ValueHolder)entry.getValue()).get())).collect(Collectors.toUnmodifiableSet());
    }

    int getQueueSize() {
        return this.totalChanges.get();
    }

    private boolean evictionNeeded() {
        return this.isCacheFull() || this.isQueueFull();
    }

    private boolean isCacheFull() {
        return this.size() > this.maximumCacheSize;
    }

    private boolean isQueueFull() {
        return this.getQueueSize() > Math.max(2 * this.maximumCacheSize, 128);
    }

    private Map.Entry<K, ValueHolder<V>> nextOldestChange() {
        Map.Entry<K, ValueHolder<V>> oldestChange = this.lastChanges.get().poll();
        this.totalChanges.decrementAndGet();
        return oldestChange;
    }

    private void compressChangesIfNeeded() {
        if (this.isQueueFull()) {
            Map.Entry entry;
            ConcurrentLinkedDeque<Map.Entry> newChanges = new ConcurrentLinkedDeque<Map.Entry>();
            Deque currentChanges = this.lastChanges.getAndSet(newChanges);
            HashSet keys = new HashSet();
            while ((entry = (Map.Entry)currentChanges.pollLast()) != null) {
                if (!keys.add(entry.getKey())) continue;
                newChanges.addFirst(entry);
            }
            this.totalChanges.set(keys.size());
        }
    }

    private void callEvictionIfNeeded() {
        if (this.evictionNeeded() && this.eviction.compareAndSet(false, true)) {
            try {
                this.callEviction();
            }
            finally {
                this.eviction.set(false);
            }
        }
    }

    private void callEviction() {
        this.lock.writeLock().lock();
        try {
            this.compressChangesIfNeeded();
            while (this.isCacheFull()) {
                Map.Entry<K, ValueHolder<V>> oldest = this.nextOldestChange();
                if (!this.delegate.remove(oldest.getKey(), oldest.getValue())) continue;
                this.evict.accept(oldest.getValue().get());
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    private ValueHolder<V> newValue(V value) {
        return new ValueHolder<V>(this.sequence.incrementAndGet(), value);
    }

    private V extractValue(ValueHolder<V> holder) {
        return holder == null ? null : (V)holder.get();
    }

    private static class OperationContext<K, V>
    implements AutoCloseable {
        V result;
        final K key;
        private final SimpleLRUCache<K, V> cache;

        OperationContext(SimpleLRUCache<K, V> cache, K key) {
            this.cache = cache;
            this.key = key;
            cache.lock.readLock().lock();
        }

        @Override
        public void close() {
            this.cache.lock.readLock().unlock();
            this.cache.callEvictionIfNeeded();
        }
    }

    private static class ValueHolder<V> {
        private final long revision;
        private final V value;

        ValueHolder(long revision, V value) {
            this.revision = revision;
            this.value = value;
        }

        V get() {
            return this.value;
        }

        public boolean equals(Object o) {
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ValueHolder that = (ValueHolder)o;
            return this.revision == that.revision;
        }

        public int hashCode() {
            return Objects.hashCode(this.revision);
        }
    }

    private static class CacheEntry<K, V>
    implements Map.Entry<K, V> {
        private final K key;
        private V val;
        private final SimpleLRUCache<K, V> cache;

        CacheEntry(SimpleLRUCache<K, V> cache, K key, V value) {
            this.cache = cache;
            this.key = key;
            this.val = value;
        }

        @Override
        public K getKey() {
            return this.key;
        }

        @Override
        public V getValue() {
            return this.val;
        }

        @Override
        public V setValue(V value) {
            if (value == null) {
                throw new NullPointerException();
            }
            V v = this.val;
            this.val = value;
            this.cache.put(this.key, value);
            return v;
        }
    }
}

