/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.tools.apt;

import java.io.IOException;
import java.io.Writer;
import java.util.Set;

import javax.annotation.processing.RoundEnvironment;
import javax.annotation.processing.SupportedAnnotationTypes;
import javax.lang.model.element.AnnotationMirror;
import javax.lang.model.element.TypeElement;
import javax.tools.FileObject;
import javax.tools.StandardLocation;

import org.apache.camel.spi.annotations.ServiceFactory;
import org.apache.camel.spi.annotations.SubServiceFactory;

import static org.apache.camel.tools.apt.helper.Strings.canonicalClassName;

@SupportedAnnotationTypes({"org.apache.camel.spi.annotations.*"})
public class SpiProcessor extends AbstractCamelAnnotationProcessor {

    @Override
    protected void doProcess(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) throws Exception {
        roundEnv.getRootElements().stream().filter(e -> e instanceof TypeElement).map(TypeElement.class::cast).forEach(this::processServiceFactory);
    }

    private void processServiceFactory(TypeElement element) {
        try {
            final String javaTypeName = canonicalClassName(element.getQualifiedName().toString());
            for (AnnotationMirror pam : element.getAnnotationMirrors()) {
                ServiceFactory sf = pam.getAnnotationType().asElement().getAnnotation(ServiceFactory.class);
                if (sf != null) {
                    String pvals = pam.getElementValues().values().iterator().next().getValue().toString();
                    for (String pval : pvals.split(",")) {
                        if (ServiceFactory.JDK_SERVICE.equals(sf.value())) {
                            FileObject resource = processingEnv.getFiler().createResource(StandardLocation.CLASS_OUTPUT, "",
                                    "META-INF/services/" + pval.replace('.', '/'), element);
                            try (Writer w = resource.openWriter()) {
                                w.append("# Generated by camel annotation processor\n");
                                w.append("class=").append(javaTypeName).append("\n");
                            }
                        } else {
                            FileObject resource = processingEnv.getFiler().createResource(StandardLocation.CLASS_OUTPUT, "",
                                    "META-INF/services/org/apache/camel/" + sf.value() + "/" + pval, element);
                            try (Writer w = resource.openWriter()) {
                                w.append("# Generated by camel annotation processor\n");
                                w.append("class=").append(javaTypeName).append("\n");
                                for (AnnotationMirror am : element.getAnnotationMirrors()) {
                                    SubServiceFactory factory = am.getAnnotationType().asElement().getAnnotation(SubServiceFactory.class);
                                    if (factory != null) {
                                        String key = factory.value();
                                        String val = am.getElementValues().values().iterator().next().getValue().toString();
                                        w.append(key).append(".class=").append(val).append("\n");
                                    }
                                }
                            }
                        }
                    }
                }
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

}
