/**
 * License Agreement.
 *
 * Ajax4jsf 1.1 - Natural Ajax for Java Server Faces (JSF)
 *
 * Copyright (C) 2007 Exadel, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 */

package org.ajax4jsf.taglib.html.facelets;

import java.io.IOException;

import javax.el.ELException;
import javax.el.ValueExpression;
import javax.faces.FacesException;
import javax.faces.component.UIComponent;

import org.ajax4jsf.framework.ajax.AjaxListener;
import org.ajax4jsf.framework.ajax.AjaxListenerHelper;
import org.ajax4jsf.framework.ajax.AjaxSource;
import org.ajax4jsf.framework.util.message.Messages;

import com.sun.facelets.FaceletContext;
import com.sun.facelets.FaceletException;
import com.sun.facelets.el.LegacyValueBinding;
import com.sun.facelets.tag.TagAttribute;
import com.sun.facelets.tag.TagAttributeException;
import com.sun.facelets.tag.TagConfig;
import com.sun.facelets.tag.TagException;
import com.sun.facelets.tag.TagHandler;

/**
 * Register an ActionListener instance on the UIComponent associated with the
 * closest parent UIComponent custom action. <p/> See <a target="_new"
 * href="http://java.sun.com/j2ee/javaserverfaces/1.1_01/docs/tlddocs/f/actionListener.html">tag
 * documentation</a>.
 * 
 * @see javax.faces.event.ActionListener
 * @see javax.faces.component.AjaxContainer
 * @author Jacob Hookom
 * @version $Id: AjaxListenerHandler.java,v 1.1.2.1 2007/02/01 15:31:23 alexsmirnov Exp $
 */
public final class AjaxListenerHandler extends TagHandler {

    private Class listenerType;

    private final TagAttribute type;

    private final TagAttribute binding;

    /**
     * @param config
     */
    public AjaxListenerHandler(TagConfig config) {
        super(config);
        this.binding = this.getAttribute("binding");
        this.type = this.getRequiredAttribute("type");
        if (type != null) {
            if (!type.isLiteral()) {
                throw new TagAttributeException(this.tag, this.type, Messages.getMessage(Messages.MUST_BE_LITERAL_ERROR));
            }
            try {
                this.listenerType = Class.forName(type.getValue());
            } catch (Exception e) {
                throw new TagAttributeException(this.tag, this.type, e);
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.sun.facelets.FaceletHandler#apply(com.sun.facelets.FaceletContext,
     *      javax.faces.component.UIComponent)
     */
    public void apply(FaceletContext ctx, UIComponent parent)
            throws IOException, FacesException, FaceletException, ELException {
        if (parent instanceof AjaxSource) {
            // only process if parent was just created
            if (parent.getParent() == null) {
                AjaxSource src = (AjaxSource) parent;
                AjaxListener listener = null;
                ValueExpression ve = null;
                if (this.binding != null) {
                    ve = this.binding.getValueExpression(ctx,
                            AjaxListener.class);
                    // TODO - handle both JSF 1.2/1.1 cases.
                    listener = new AjaxListenerHelper(new LegacyValueBinding(ve));
                }
                if (listener == null) {
                    try {
                        listener = (AjaxListener) listenerType.newInstance();
                    } catch (Exception e) {
                        throw new TagAttributeException(this.tag, this.type, e.getCause());
                    }
                    if (ve != null) {
                        ve.setValue(ctx, ve);
                    }
                }
                src.addAjaxListener(listener);
            }
        } else {
            throw new TagException(this.tag, Messages.getMessage(Messages.NOT_PARENT_AJAX_COMPONENT_ERROR, parent));
        }
    }
}
