/**
 * License Agreement.
 *
 * Ajax4jsf 1.1 - Natural Ajax for Java Server Faces (JSF)
 *
 * Copyright (C) 2007 Exadel, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 */

package org.ajax4jsf.framework.util.command;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.faces.component.UICommand;
import javax.faces.component.UIComponent;
import javax.faces.component.UIParameter;

import org.ajax4jsf.framework.renderer.RendererUtils.HTML;
import org.ajax4jsf.framework.resource.ClientScript;
import org.ajax4jsf.framework.util.InputUtils;
import org.ajax4jsf.framework.util.javascript.JSFunction;
import org.ajax4jsf.framework.util.javascript.JSReference;
import org.ajax4jsf.framework.util.javascript.ScriptString;
/**
 * Utility class to build command links, buttons etc.
 * Introduces correct parameter escaping
 * @author Maksim Kaszynski
 *
 */
public class CommandScriptBuilder {
	public static final ClientScript SCRIPT = new CommandScript();

	private static final ScriptString JSR_EVENT =  new JSReference("(event || window.event)");
	
	private boolean checkActionAndListener = true;
	private StringBuffer codeAfter = new StringBuffer();
	private StringBuffer codeBefore = new StringBuffer();
	private UICommand component;
	private Map parameters;
	private String target;
	private boolean useOriginalOnclick = true;
	
	public CommandScriptBuilder(UICommand command) {
		this.component = command;
		collectParams();
	}
	
	public void addCodeAfter(String codeAfter) {
		this.codeAfter.append(codeAfter);
	}
	
	public void addCodeBefore(String codeBefore) {
		this.codeBefore.append(codeBefore);
	}
	
	public void addParameter(String name, Object value) {
		if (parameters == null) {
			parameters = new LinkedHashMap();
		}
		parameters.put(name, value);
	}
	
	
	
	private void addParameter(UIParameter parameter) {
		addParameter(parameter.getName(), parameter.getValue());
	}

	private void collectParams () {
		for (Iterator iter = component.getChildren().iterator(); iter.hasNext();) {
			UIComponent kid = (UIComponent) iter.next();
			if (kid instanceof UIParameter) {
				addParameter((UIParameter) kid);
			}
		}
		
	}

	public StringBuffer getScript() {
		StringBuffer script = new StringBuffer();
		
		script.append(codeBefore).append(";");
		
		if (useOriginalOnclick) {
			Object onclick = component.getAttributes().get(HTML.onclick_ATTRIBUTE);
			
			if (onclick != null) {
				script.append(onclick).append(";");
			}
		}
		
		
		if (shouldGenerateAction()) {
			JSFunction function = new JSFunction("new A4J_Command");
			function.addParameter(JSR_EVENT);
			function.addParameter(parameters);
			function.addParameter(target);
			
			function.appendScript(script);
			script.append(";");
		}
		
		
		script.append(codeAfter).append(";");
		
		return script;
	}

	public String getTarget() {
		return target;
	}
	
	private boolean hasActionOrListenerSet() {
		return (component.getAction() != null) || (component.getActionListener() != null);
	}
	
	public boolean isCheckActionAndListener() {
		return checkActionAndListener;
	}
	
	public boolean isUseOriginalOnclick() {
		return useOriginalOnclick;
	}
	
	public void setCheckActionAndListener(boolean checkActionAndListener) {
		this.checkActionAndListener = checkActionAndListener;
	}

	public void setTarget(String target) {
		this.target = target;
	}
	
	public void setUseOriginalOnclick(boolean useOriginalOnclick) {
		this.useOriginalOnclick = useOriginalOnclick;
	}
	
	private boolean shouldGenerateAction() {
		
		return (!InputUtils.isDisabled(component)) && (checkActionAndListener ? hasActionOrListenerSet() : true);
		
	}
	
	public String toString() {
		return getScript().toString();
	}
	
	
}
