package webwork.multipart;

import java.io.Serializable;

/**
 * @author Hani Suleiman (hani@formicary.net) Date: Dec 20, 2003 Time: 11:06:59 PM
 */
public class ProgressMonitor implements Serializable
{
    private static final long serialVersionUID = 5344649008664921629L;
    private static final long MIN_SAMPLE_TIME = 150l;
    private static final int HISTORY_SIZE = 20;
    private String fileName;
    private int read;
    private long size;
    private boolean completed;
    private long startTime;
    //array of datapoint[0] = bytes in this chunk, datapoint[1] = data in this chunk
    private int[][] history = new int[HISTORY_SIZE][2];
    private boolean historyFull = false;
    private int historyIndex = 0;
    private long lastHistoryTime;
    private long lastHistoryTotal;

    public String getFileName()
    {
        return fileName;
    }

    public void setFileName(String fileName)
    {
        this.fileName = fileName;
        startTime = System.currentTimeMillis();
        lastHistoryTime = startTime;
    }

    public int getRead()
    {
        return read;
    }

    public void setRead(int read)
    {
        long now = System.currentTimeMillis();
        long timeSinceLastRead = now - lastHistoryTime;
        if (timeSinceLastRead > MIN_SAMPLE_TIME)
        {
            long diff;
            if (this.read == 0)
            {
                diff = read;
            }
            else
            {
                diff = read - lastHistoryTotal;
            }
            history[historyIndex][0] = (int) diff;
            history[historyIndex][1] = (int) timeSinceLastRead;
            historyIndex++;
            if (historyIndex == HISTORY_SIZE)
            {
                historyFull = true;
                historyIndex = 0;
            }
            lastHistoryTime = now;
            lastHistoryTotal = read;
        }
        this.read = read;
    }

    public long getStartTime()
    {
        return startTime;
    }

    public int getRemainingSeconds()
    {
        int bitrate = getBitRate();
        if (bitrate == 0)
        {
            return 0;
        }
        return (int) ((size - read) / bitrate);
    }

    /**
     * Get the number of bytes per second for this upload. If the upload has just started or is finished, the rate will
     * be 0
     *
     * @return the bit rate
     */
    public int getBitRate()
    {
        if (completed || startTime == 0)
        {
            return 0;
        }
        int totalBytes = 0;
        int totalTime = 0;
        int end = history.length;
        if (!historyFull)
        {
            end = historyIndex;
        }
        for (int i = 0; i < end; i++)
        {
            int[] values = history[i];
            totalBytes += values[0];
            totalTime += values[1];
        }
        if (totalTime == 0)
        {
            return 0;
        }
        double timeInSeconds = totalTime / (double) 1000;
        return (int) (totalBytes / timeInSeconds);
    }

    public long getRemaining()
    {
        return size - read;
    }

    public long getSize()
    {
        return size;
    }

    public void setSize(long size)
    {
        this.size = size;
    }

    public boolean isCompleted()
    {
        return completed;
    }

    public void setCompleted(boolean completed)
    {
        this.completed = completed;
    }
}
