/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.multipart;

import org.apache.commons.logging.LogFactory;
import webwork.config.Configuration;
import webwork.util.ClassLoaderUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;

/**
 * <code>MultiPartRequestWrapper</code> will parse a multipart request and provide a wrapper around the request. The
 * parse it uses depends on the "webwork.multipart.parser" setting. The present choices are "internal", the
 * webwork-multipart parser (derived from pell), "pell" for Jason Pell and "cos" for Jason Hunter.
 *
 * @author Matt Baldree
 * @author Hani Suleiman $Date: 2007/11/25 14:22:32 $
 * @version $Revision: 1.16 $
 */
public class MultiPartRequestWrapper extends HttpServletRequestWrapper
{
    MultiPartRequest multi;

    public MultiPartRequestWrapper(HttpServletRequest request, String saveDir, Integer maxSize)
            throws IOException
    {
        super(request);
        if (request instanceof MultiPartRequest)
        {
            multi = (MultiPartRequest) request;
        }
        else
        {
            String parser = Configuration.getString("webwork.multipart.parser");
            if ("pell".equals(parser))
            {
                multi = new PellMultiPartRequest(request, saveDir, maxSize.intValue());
            }
            else if ("custom".equals(parser))
            {  // allow specification of your own handler class - WW-814
                String customHandler = Configuration.getString("webwork.multipart.parser.class");
                try
                {
                    Class customHandlerClass = ClassLoaderUtils.loadClass(customHandler, this.getClass());
                    // This is probably better if the dep ever moves to a newer version of common-beanutils
                    // Object customHandlerObject = ConstructorUtils.invokeConstructor(customHandlerClass, new Object[]{request, saveDir, maxSize});
                    Object customHandlerObject = customHandlerClass.getConstructor(new Class[] { HttpServletRequest.class, String.class, Integer.TYPE })
                            .newInstance(new Object[] { request, saveDir, maxSize });
                    if (customHandlerObject instanceof MultiPartRequest)
                    {
                        multi = (MultiPartRequest) customHandlerObject;
                    }
                    else
                    {
                        throw new IOException("The provided custom multipart parser class does not implement MultiPartRequest.");
                    }
                }
                catch (Exception e)
                {
                    LogFactory.getLog(MultiPartRequestWrapper.class).error(e.getMessage(), e);
                    throw new IOException("Error instantiating webwork.multipart.parser.class: " + customHandler + "(original message: " + e.getMessage() + ')');
                }
            }
            else
            {
                throw new IOException("Not multipart parse specified in webwork.properties.");
            }
        }
    }

    public Enumeration getParameterNames()
    {
        return mergeParams(super.getParameterNames(), multi.getParameterNames());
    }

    public String getParameter(String name)
    {
        // Look in the original request first
        String superValue = super.getParameter(name);
        return (superValue != null ? superValue : multi.getParameter(name));
    }

    public String[] getParameterValues(String name)
    {
        // Look in the original request first
        String[] superValue = super.getParameterValues(name);
        return (superValue != null ? superValue : multi.getParameterValues(name));
    }

    public Map getParameterMap()
    {
        Map map = new HashMap();
        Enumeration e = getParameterNames();
        while (e.hasMoreElements())
        {
            String name = (String) e.nextElement();
            map.put(name, this.getParameterValues(name));
        }
        return map;
    }

    // Methods only in MultipartRequest
    public Enumeration getFileNames()
    {
        return multi.getFileNames();
    }

    public String getFilesystemName(String name)
    {
        return multi.getFilesystemName(name);
    }

    public String getContentType(String name)
    {
        return multi.getContentType(name);
    }

    public File getFile(String name)
    {
        return multi.getFile(name);
    }

    //private
    protected Enumeration mergeParams(Enumeration params1, Enumeration params2)
    {
        Vector temp = new Vector();
        while (params1.hasMoreElements())
        {
            temp.add(params1.nextElement());
        }
        while (params2.hasMoreElements())
        {
            temp.add(params2.nextElement());
        }
        return temp.elements();
    }

    public InputStream getMemoryFileContents(String name)
    {
        // since this is only implemented for the PellMultiPartRequest
        // check if we have an instance of it if not log an error
        // and return null
        if (multi instanceof PellMultiPartRequest)
        {
            return ((PellMultiPartRequest) multi).getMemoryFileContents(name);
        }
        else
        {
            LogFactory.getLog(MultiPartRequestWrapper.class).error("File uploading using memory is currently implemented only for the PellMultiPartRequest");
            return null;
        }
    }
}
