/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.action.factory;

import webwork.action.Action;
import webwork.config.Configuration;

/**
 * Default implementation of an action factory facade that creates the default set of action factory proxies.  This
 * class delegates to the proxies as implementation, thus acting as a facade to them. <p/> <p>The order in which factory
 * proxies are executed is as follows:</p> <ol> <li>{@link ChainingActionFactoryProxy} - Copies properties from one
 * action to the next in a chain.</li> <li>{@link ParametersActionFactoryProxy} - Sets parameters on the action.</li>
 * <li>{@link PrepareActionFactoryProxy} - Prepares or initializes the action.</li> <li>{@link
 * ContextActionFactoryProxy} - Establishes the action's context by executing methods for all implemented
 * <code>Aware*</code> interfaces.</li> <li>{@link CommandActionFactoryProxy} - Executes a specified "command" method if
 * the action implements the {@link webwork.action.CommandDriven} interface.</li> <li>{@link AliasingActionFactoryProxy}
 * - Locates an action from the configuration substituting the alias with the associated action string.</li> <li>{@link
 * CommandActionFactoryProxy} - Executes a specified "command" method again in the event the original action name was
 * originally an alias.</li> <li>{@link JspActionFactoryProxy} - Returns the JSP action if its suffix is ".jsp".</li>
 * <li>{@link PrefixActionFactoryProxy} - Returns an action using a configured list of packages to prefix its name.</li>
 * <li>{@link XMLActionFactoryProxy} - Returns the XML action if its suffix is ".xml".</li> <li>{@link
 * ScriptActionFactoryProxy} - Returns the Script action if its suffix matches that of a supported scripting
 * language.</li> <li>{@link JavaActionFactory} - Returns an action object from a fully qualified classname.</li> </ol>
 *
 * @author Rickard \u00D6berg (rickard@middleware-company.com)
 * @version $Revision: 1.27 $
 */
public class DefaultActionFactory
        extends ActionFactory
{
    // Attributes ----------------------------------------------------
    protected ActionFactory factory;

    /**
     * Initialize action factory proxy delegation chain.
     */
    public DefaultActionFactory()
    {
        // Default series of factory proxies
        // RO: There is a purpose for the duplicated
        // command proxy. DON'T REMOVE THE SECOND ONE!
        factory = getRootFactory();
        factory = new ScriptActionFactoryProxy(factory);
        factory = new XMLActionFactoryProxy(factory);
        factory = new PrefixActionFactoryProxy(factory);
        factory = new JspActionFactoryProxy(factory);
        factory = new CommandActionFactoryProxy(factory);
        factory = new AliasingActionFactoryProxy(factory);
        factory = new CommandActionFactoryProxy(factory);
        factory = new ContextActionFactoryProxy(factory);
        factory = new PrepareActionFactoryProxy(factory);
        factory = new ParametersActionFactoryProxy(factory);
        factory = new ChainingActionFactoryProxy(factory);
        try
        {
            boolean reloadEnabled = "true".equalsIgnoreCase(Configuration.getString("webwork.configuration.xml.reload"));
            if (reloadEnabled)
            {
                factory = new ReloadHelperActionFactoryProxy(factory);
            }
        }
        catch (IllegalArgumentException ex)
        {}
    }

    /**
     * Get the root action factory (usually java). This method is useful for cases where the standard action factories
     * are all required, and the only modification being the way that the Java action implementation is looked up.
     *
     * @return the root ActionFactory
     */
    protected ActionFactory getRootFactory()
    {
        ActionFactory root = null;
        try
        {
            boolean reloadEnabled = "true".equalsIgnoreCase(Configuration.getString("webwork.action.java.reload"));
            if (reloadEnabled)
            {
                root = new ReloadingJavaActionFactory();
            }
        }
        catch (IllegalArgumentException ex)
        {
        }
        if (root == null)
        {
            root = new JavaActionFactory();
        }

        return root;
    }

    /**
     * Get an action object. The name should be the fully qualified classname of the action. Returns an instance of the
     * matching action class by searching through the action factory proxy delegation chain.
     *
     * @param name classname of the action to be created
     *
     * @return the action corresponding to the given name
     *
     * @throws Exception
     */
    public Action getActionImpl(String name)
            throws Exception
    {
        return factory.getActionImpl(name);
    }

}
