package webwork.view.velocity;

import java.util.*;
import java.io.Writer;

import javax.servlet.ServletContext;
import javax.servlet.ServletResponse;
import javax.servlet.ServletRequest;

import org.apache.velocity.app.Velocity;
import org.apache.velocity.context.Context;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.Template;
import org.apache.velocity.runtime.RuntimeSingleton;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import webwork.config.Configuration;
import webwork.util.ServletValueStack;

/**
 * @author Hani Suleiman (hani@formicary.net) Date: Nov 15
 * @author 2003 Time: 11:21:14 AM
 */
public class VelocityHelper
{
    private static final Log log = LogFactory.getLog(VelocityHelper.class);

    /**
     * The HTTP request object context key.
     */
    public static final String REQUEST = "req";

    /**
     * The HTTP response object context key.
     */
    public static final String RESPONSE = "res";

    public static final String ACTION = "action";

    static final String WEBWORK_UTIL = "webwork";
    private static boolean initialized = false;
    public static final String VELO_CONTEXT = "__webwork__velocity__context";
    private static final Object INIT_MUTEX = new Object();

    /**
     * Hook up Velocity with the WebWork configuration.
     */
    public static void initVelocity(ServletContext context) throws Exception
    {
        synchronized (INIT_MUTEX)
        {
            // WebWork configuration provides main config
            final Properties conf = new Properties()
            {
                public Object get(Object key)
                {
                    return Configuration.get(key.toString());
                }

                public String getProperty(String key)
                {
                    return Configuration.getString(key.toString());
                }

                public Enumeration keys()
                {
                    final Iterator list = Configuration.list();
                    return new Enumeration()
                    {
                        public Object nextElement()
                        {
                            return list.next();
                        }

                        public boolean hasMoreElements()
                        {
                            return list.hasNext();
                        }
                    };
                }
            };

            // Set dynamic properties here
            // The properties not set here are taken from the WebWork configuration
            Properties p = new Properties(conf)
            {
                public Enumeration keys()
                {
                    return conf.keys();
                }
            };

            /*
            *  first, normalize our velocity log file to be in the
            *  webapp
            */

            String log = p.getProperty(Velocity.RUNTIME_LOG);

            if (log != null)
            {
                log = context.getRealPath(log);

                if (log != null)
                {
                    p.setProperty(Velocity.RUNTIME_LOG, log);
                }
            }


            /*
            *  If there is a file loader resource path, treat it the
            *  same way, but only if it doesn't start with /. In that case
            *  we use it as-is to allow the templates to be taken from some
            *  repository (!very useful during development!).
            */
            String path = p.getProperty(Velocity.FILE_RESOURCE_LOADER_PATH);

            if (path != null && (path.equals("/") || !path.startsWith("/")))
            {
                path = context.getRealPath(path);
                if (path != null)
                {
                    p.setProperty(Velocity.FILE_RESOURCE_LOADER_PATH, path);
                }
            }

            Velocity.setApplicationAttribute(ServletContext.class.getName(), context);
            Velocity.init(p);
            initialized = true;
        }
    }

    public static void merge(Context context, String templateName, Writer writer)
    {
        try
        {
            Template t = RuntimeSingleton.getTemplate(templateName);
            t.merge(context, writer);
            //java.io.PrintWriter out = new java.io.PrintWriter(System.out);
            //t.merge(context, out);
            //out.close();
        }
        catch (Exception e)
        {
            log.error(e.getMessage(), e);
        }
    }

    /**
     * Get the Velocity Context from the request and response. Additionally, initialize Velocity if WebWork hasn't
     * already initialized it.
     * <p/>
     * Note: This method should only be used if WebWork is responsible for initializing Velocity. If Velocity's
     * lifecycle is being managed else where, use {@link #getContextWithoutInit(ServletRequest, ServletResponse, Map)}
     * instead, and only <strong>after</strong> Velocity has been initialized.
     */
    public static Context getContext(ServletContext context, ServletRequest request, ServletResponse response)
    {
        return getContext(context, request, response, Collections.EMPTY_MAP);
    }

    /**
     * Get the Velocity Context from the request, response and any additional context parameters. Additionally,
     * initialize Velocity if WebWork hasn't already initialized it.
     * <p/>
     * Note: This method should only be used if WebWork is responsible for initializing Velocity. If Velocity's
     * lifecycle is being managed else where, use {@link #getContextWithoutInit(ServletRequest, ServletResponse, Map)}
     * instead, and only <strong>after</strong> Velocity has been initialized.
     */
    public static Context getContext(ServletContext context, ServletRequest request, ServletResponse response, Map extraContextParams)
    {
        try
        {
            checkInited(context);
        }
        catch (Exception e)
        {
            log.error(e.getMessage(), e);
            return null;
        }
        return getContextWithoutInit(request, response, extraContextParams);
    }

    /**
     * Get the Velocity Context from the request, response and any additional context parameters.
     * <p/>
     * Note: This method should only be used if <strong>after</strong> Velocity has been initialized elsewhere.
     */
    public static Context getContextWithoutInit(ServletRequest request, ServletResponse response, Map extraContextParams)
    {
        WebWorkVelocityContext ctx = (WebWorkVelocityContext) request.getAttribute(VELO_CONTEXT);
        if (ctx == null)
        {
            ctx = new WebWorkVelocityContext(ServletValueStack.getStack(request), extraContextParams);
            ctx.put(REQUEST, request);
            ctx.put(RESPONSE, response);
            ctx.put(WEBWORK_UTIL, new WebWorkUtil(ctx));
            ctx.put(ACTION, ServletValueStack.getStack(request).findValue("/"));
            request.setAttribute(VELO_CONTEXT, ctx);
        }
        return ctx;
    }

    private static void checkInited(ServletContext context) throws Exception
    {
        synchronized (INIT_MUTEX)
        {
            if (!initialized)
            {
                initVelocity(context);
            }
        }
    }

    /**
     * WebWork specific Velocity context implementation.
     */
    static class WebWorkVelocityContext extends VelocityContext
    {
        ServletValueStack stack;

        WebWorkVelocityContext(ServletValueStack aStack, Map extraContextParams)
        {
            super(extraContextParams);
            stack = aStack;
}

        public boolean internalContainsKey(java.lang.Object key)
        {
            boolean contains = super.internalContainsKey(key);
            return contains ? true : stack.test(key.toString());
        }

        public Object internalGet(String key)
        {
            return super.internalContainsKey(key) ? super.internalGet(key) : getValueFromStack(key);
        }

        private Object getValueFromStack(final String key)
        {
            // If key uses the velocity magic prefix of ".literal." then we just bail from here
            if (key.startsWith(".literal."))
            {
                return null;
            }
            return stack.findValue(key);
        }
    }
}
