/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.view.taglib;

import webwork.util.BeanUtil;
import webwork.util.TextUtil;
import webwork.util.URLCodec;

import java.io.IOException;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.PageContext;

/**
 * This tag is used to create a URL. You can use the "param" tag inside the body to provide additional request
 * parameters.
 *
 * @author Rickard \u00D6berg (rickard@dreambean.com)
 * @version $Revision: 1.27 $
 * @see ParamTag
 */
public class URLTag extends WebWorkBodyTagSupport implements ParamTag.Parametric
{
    // Attributes ----------------------------------------------------
    private String page;
    private String valueAttr;
    private String value;
    private final Map<String, Object> params = new LinkedHashMap<String, Object>();
    private boolean includeContext = true;

    // Public --------------------------------------------------------
    /**
     * The includeParams attribute may have the value 'none', 'get' or 'all'. It is used when the url tag is used
     * without a value or page attribute. Its value is looked up on the ValueStack If no includeParams is specified then
     * 'get' is used. none - include no parameters in the URL get  - include only GET parameters in the URL (default)
     * all  - include both GET and POST parameters in the URL
     */
    public static final String NONE = "none";
    public static final String GET = "get";
    public static final String ALL = "all";

    /**
     * @deprecated use value instead
     */
    @Deprecated
    public void setPage(final String aName)
    {
        page = aName;
    }

    public void setValue(final String aName)
    {
        valueAttr = aName;
    }

    public void setIncludeContext(final boolean includeContext)
    {
        this.includeContext = includeContext;
    }

    public void addParameter(final String name, final Object value)
    {
        if (value == null)
        {
            params.remove(name);
        }
        else
        {
            params.put(name, BeanUtil.toStringValue(value));
        }
    }

    // BodyTag implementation ----------------------------------------
    @Override
    public int doStartTag() throws JspException
    {
        value = resolveValue();
        // Clear the params map if it has been instantiated before
        params.clear();

        // BB
        //
        //  IN the base old days this used to be a bad XSS security hole since it would echo the URL as invoked.  Bad Ju Ju!
        //
        // So now we dont do this kinda malarky
        //
        return EVAL_BODY_TAG;
    }

    private String resolveValue()
    {
        if (page == null && valueAttr != null)
        {
            return findString(valueAttr);
        }
        else
        {
            return page;
        }
    }

    @Override
    public int doEndTag() throws JspException
    {
        try
        {
            final HttpServletRequest request = (HttpServletRequest) pageContext.getRequest();
            final StringBuilder urlFragment = new StringBuilder();

            if (value != null)
            {
                // Check if context path needs to be added
                // Add path to absolute links
                if (value.startsWith("/") && includeContext)
                {
                    urlFragment.append(request.getContextPath());
                }

                urlFragment.append(urlQuoteEncode(value));
            }
            //
            // Now we have a URL prefix fragment that has " and ' percent-encoded.  So the URL cannnot break out of an HTML attributes
            // it might be in.  For belts and braces, we later html encode so that if we happened to output the text rather than inlcude i
            // it as a href attribute, then we are still safe.  Html encoded URLS as hrefs are valid and will be followed by browsers.
            //
            //
            //
            // BB
            //
            //  IN the base old days this used to be a bad XSS security hole since it would echo the URL as invoked.  Bad Ju Ju!
            //
            // So now we don't do this kinda malarky
            //
            addDeclaredParameters(urlFragment);

            String result = htmlEscape(urlFragment.toString());

            final String id = getId();
            if (id != null)
            {
                pageContext.setAttribute(id, result);
                pageContext.setAttribute(id, result, PageContext.REQUEST_SCOPE);
            }
            else
            {
                try
                {
                    pageContext.getOut().write(result);
                }
                catch (final IOException _ioe)
                {
                    throw new JspException("IOError: " + _ioe.getMessage(), _ioe);
                }
            }

            super.doEndTag();
            return EVAL_PAGE;
        }
        finally
        {
            params.clear();
        }
    }

    private void addDeclaredParameters(final StringBuilder link)
    {
        if ((params != null) && (params.size() > 0))
        {
            if (link.toString().indexOf("?") == -1)
            {
                link.append('?');
            }
            else
            {
                link.append("&");
            }

            // Set parameters
            final Iterator<Map.Entry<String, Object>> iter = params.entrySet().iterator();
            while (iter.hasNext())
            {
                final Map.Entry<String, Object> entry = iter.next();
                final String name = entry.getKey();
                final Object value = entry.getValue();
                if (value != null)
                {
                    if (value instanceof String)
                    {
                        link.append(urlEncode(name));
                        link.append('=');
                        link.append(urlEncode((String) value));
                    }
                    else
                    {
                        final String[] values = (String[]) value;

                        //Include multiple parameter values.  Fixes WW-164
                        for (int i = 0; i < values.length; i++)
                        {
                            final String multiValue = values[i];
                            link.append(urlEncode(name));
                            link.append('=');
                            link.append(urlEncode(multiValue));
                            if (i != values.length - 1)
                            {
                                link.append("&");
                            }
                        }
                    }
                }

                if (iter.hasNext())
                {
                    link.append("&");
                }
            }
        }
    }

    /**
     * Template method to allow overriding the HTML escaping of our output
     *
     * @param value the string to HTML escape
     *
     * @return a HTML escaped value
     */
    protected String htmlEscape(final String value)
    {
        return TextUtil.escapeHTML(value);
    }

    /**
     * Template method to allow overriding of the URL encoding used.
     *
     * @param value the string to be URL encoded
     *
     * @return a URL encoded string please
     */
    protected String urlEncode(final String value)
    {
        return URLCodec.encode(value);
    }

    /**
     * This will encode double and single quotes into URL safe equivalents so that the URL fragment can be later HTML encoded.
     * This saves us from having to fully parse the URL and yet still makes it safe from XSS attacks because it will later be HTML encoded.
     *
     * @param urlFragement the string to be URL encoded
     *
     * @return a URL encoded string please
     */
    private StringBuilder urlQuoteEncode(final String urlFragement)
    {
        StringBuilder sb = new StringBuilder(urlFragement.length() + 16);
        for (char c : urlFragement.toCharArray())
        {
            if (c == '"')
            {
                sb.append("%22");
            }
            else if (c == '\'')
            {
                sb.append("%27");
            }
            else
            {
                sb.append(c);
            }
        }
        return sb;
    }
}
