/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.view.jasperreports;

import dori.jasper.engine.JRException;
import dori.jasper.engine.JRExporter;
import dori.jasper.engine.JRExporterParameter;
import dori.jasper.engine.JasperExportManager;
import dori.jasper.engine.JasperFillManager;
import dori.jasper.engine.JasperManager;
import dori.jasper.engine.JasperPrint;
import dori.jasper.engine.JasperReport;
import dori.jasper.engine.export.JRCsvExporter;
import dori.jasper.engine.export.JRHtmlExporter;
import dori.jasper.engine.export.JRXlsExporter;
import dori.jasper.engine.export.JRXmlExporter;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import webwork.util.ServletValueStack;
import webwork.util.FastByteArrayOutputStream;

import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.util.Map;


/**
 * Provide a view of a webwork action as a jasper report. The report format will be determined by the format parameter
 * passed in the view definition. This servlet should be deployed to service view requests of the form: <code>&lt;view
 * name&gt;.jr?dataSource=&lt;data source&gt;&amp;format=&lt;format string&gt;</code> Valid report format strings are
 * the following: <ul> <li>PDF</li> <li>XML</li> <li>HTML</li> <li>XLS</li> <li>CSV</li> </ul> It will then look for a
 * compiled report definition of the form <code>&lt;view name&gt;.jasper</code>, run it to the appropriate output format
 * and stream the results as the HTTP response.
 */
public class JasperReportViewServlet extends HttpServlet
{

    protected static Log log = LogFactory.getLog(JasperReportViewServlet.class);

    public static final String FORMAT_PDF = "PDF";
    public static final String FORMAT_XML = "XML";
    public static final String FORMAT_HTML = "HTML";
    public static final String FORMAT_XLS = "XLS";
    public static final String FORMAT_CSV = "CSV";

    /**
     * Service a HTTP request
     *
     * @param request  the request to service
     * @param response the response to send to the client
     */
    public void service(HttpServletRequest request, HttpServletResponse response)
            throws ServletException
    {

        //construct the data source for the report
        ServletValueStack stack = ServletValueStack.getStack(request);
        String dataSource = request.getParameter("dataSource");
        ValueStackDataSource stackDataSource = new ValueStackDataSource(stack, dataSource);

        //get the output format
        String outputFormat = request.getParameter("format");
        if (outputFormat == null)
        {
            outputFormat = FORMAT_PDF;
        }

        if (!"contype".equals(request.getHeader("User-Agent")))
        {
            // Determine the directory that the report file is in and set the reportDirectory parameter
            String systemId = getServletContext().getRealPath(request.getServletPath());
            Map parameters = new ValueStackShadowMap(stack);
            File directory = new File(systemId.substring(0, systemId.lastIndexOf(File.separator)));
            parameters.put("reportDirectory", directory);

            byte[] output;
            JasperPrint jasperPrint;

            // Fill the report and produce a print object
            try
            {
                JasperReport jasperReport = JasperManager.loadReport(systemId);
                jasperPrint = JasperFillManager.fillReport(jasperReport, parameters, stackDataSource);
            }
            catch (JRException e)
            {
                log.error("Error building report for uri " + systemId, e);
                throw new ServletException(e.getMessage(), e);
            }

            // Export the print object to the desired output format
            try
            {

                if (outputFormat.equals(FORMAT_PDF))
                {
                    response.setContentType("application/pdf");
                    // response.setHeader("Content-disposition", "inline; filename=report.pdf");

                    output = JasperExportManager.exportReportToPdf(jasperPrint);
                }
                else
                {
                    JRExporter exporter;

                    if (outputFormat.equals(FORMAT_CSV))
                    {
                        response.setContentType("text/plain");
                        exporter = new JRCsvExporter();
                    }
                    else if (outputFormat.equals(FORMAT_HTML))
                    {
                        response.setContentType("text/html");
                        exporter = new JRHtmlExporter();
                    }
                    else if (outputFormat.equals(FORMAT_XLS))
                    {
                        response.setContentType("application/vnd.ms-excel");
                        exporter = new JRXlsExporter();
                    }
                    else if (outputFormat.equals(FORMAT_XML))
                    {
                        response.setContentType("text/xml");
                        exporter = new JRXmlExporter();
                    }
                    else
                    {
                        throw new ServletException("Unknown report format: " + outputFormat);
                    }

                    output = exportReportToBytes(jasperPrint, exporter);
                }
            }
            catch (JRException e)
            {
                String message = "Error producing " + outputFormat + " report for uri " + systemId;
                log.error(message, e);
                throw new ServletException(e.getMessage(), e);
            }

            response.setContentLength(output.length);

            ServletOutputStream ouputStream;
            try
            {
                log.debug("Writing " + output.length + " bytes to output stream");
                ouputStream = response.getOutputStream();
                ouputStream.write(output);
                ouputStream.flush();
                ouputStream.close();
            }
            catch (IOException e)
            {
                log.error("Error writing report output", e);
                throw new ServletException(e.getMessage(), e);
            }
        }
        else
        {
            // Code to handle "contype" request from IE

            try
            {
                ServletOutputStream outputStream;
                response.setContentType("application/pdf");
                response.setContentLength(0);
                outputStream = response.getOutputStream();
                outputStream.close();
            }
            catch (IOException e)
            {
                log.error("Error writing report output", e);
                throw new ServletException(e.getMessage(), e);
            }
        }
    }

    /**
     * Run a Jasper report to CSV format and put the results in a byte array
     *
     * @param jasperPrint The Print object to render as CSV
     * @param exporter    The exporter to use to export the report
     *
     * @return A CSV formatted report
     *
     * @throws JRException If there is a problem running the report
     */
    private byte[] exportReportToBytes(JasperPrint jasperPrint, JRExporter exporter) throws JRException
    {
        byte[] output;
        FastByteArrayOutputStream baos = new FastByteArrayOutputStream();

        exporter.setParameter(JRExporterParameter.JASPER_PRINT, jasperPrint);
        exporter.setParameter(JRExporterParameter.OUTPUT_STREAM, baos);

        exporter.exportReport();

        output = baos.toByteArray();
        return output;
    }

}
