/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.multipart;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.logging.LogFactory;
import webwork.config.Configuration;
import webwork.multipart.parser.MultipartListener;
import webwork.multipart.parser.MultipartRequest;

public class WebworkMultiPartRequest extends MultiPartRequest
{
    private final ProgressMonitor monitor;

    private MultipartRequest multi;

    /**
     * The encoding is looked up from the configuration setting 'webwork.i18n.encoding'.  This is usually set in
     * default.properties & webwork.properties.
     */
    private String getEncoding()
    {
        try
        {
            String encoding = Configuration.getString("webwork.i18n.encoding");
            if (encoding != null)
            {
                return encoding;
            }
        }
        catch (IllegalArgumentException e)
        {
            LogFactory.getLog(WebworkMultiPartRequest.class).info(
                    "Could not get encoding property 'webwork.i18n.encoding' for file upload.  Using system default");
        }
        return null;
    }

    /**
     * @param maxSize maximum size post allowed
     * @param saveDir the directory to save off the file
     * @param req     the request containing the multipart
     */
    public WebworkMultiPartRequest(HttpServletRequest req, String saveDir, int maxSize) throws IOException
    {
        boolean memoryUpload = false;
        try
        {
            memoryUpload = "true".equalsIgnoreCase(Configuration.getString("webwork.multipart.uploadToMemory"));
        }
        catch (IllegalArgumentException e)
        {
            LogFactory.getLog(WebworkMultiPartRequest.class).info(
                    "Could not read webwork.multipart.uploadToMemory property for file upload. Defaulting to false");
        }
        monitor = new ProgressMonitor();

        MultipartListener listener = new MultipartListener()
        {
            public void fileCompleted(String fileName)
            {
                monitor.setCompleted(true);
            }

            public void fileStarted(String fileName, long size)
            {
                monitor.setFileName(fileName);
                monitor.setSize(size);
            }

            public void dataRead(String fileName, int amount)
            {
                monitor.setRead(monitor.getRead() + amount);
            }
        };

        req.getSession().setAttribute("progress", monitor);

        if (memoryUpload)
        {
            multi = new MultipartRequest(req.getContentType(),
                    req.getContentLength(),
                    req.getInputStream(),
                    maxSize,
                    MultipartRequest.IGNORE_FILES_IF_MAX_BYES_EXCEEDED,
                    getEncoding(),
                    listener);
        }
        else
        {
            multi = new MultipartRequest(req.getContentType(),
                    req.getContentLength(),
                    req.getInputStream(),
                    saveDir,
                    maxSize,
                    MultipartRequest.IGNORE_FILES_IF_MAX_BYES_EXCEEDED,
                    getEncoding(),
                    listener);
        }
        req.getSession().removeAttribute("progress");
    }

    public Enumeration getParameterNames()
    {
        return multi.getParameterNames();
    }

    public String getParameter(String name)
    {
        return multi.getURLParameter(name);
    }

    public String[] getParameterValues(String name)
    {
        Enumeration e = multi.getURLParameters(name);
        if (!e.hasMoreElements())
        {
            return null;
        }

        List values = new ArrayList();
        while (e.hasMoreElements())
        {
            values.add(e.nextElement());
        }
        return (String[]) values.toArray(new String[values.size()]);
    }

    public Enumeration getFileNames()
    {
        return multi.getFileParameterNames();
    }

    public String getFilesystemName(String name)
    {
        return multi.getBaseFilename(name);
    }

    public String getContentType(String name)
    {
        return multi.getContentType(name);
    }

    public File getFile(String name)
    {
        return multi.getFile(name);
    }

    public InputStream getMemoryFileContents(String name)
    {
        return multi.getFileContents(name);
    }
}
