/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.multipart;

import org.apache.commons.logging.LogFactory;
import webwork.config.Configuration;
import com.oreilly.servlet.MultipartRequest;

import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.io.IOException;
import java.util.Enumeration;

/* ------------------------------------------------------------ */

/**
 * Multipart Form Data request adapter for Jason Hunter's multipart utils. You need to watch his license. He requires
 * you to own his book.
 *
 * @author Matt Baldree (matt@smallleap.com) modified for WW's use
 * @author Scott Farquhar (scott@atlassian.com) added i18n handling (WW-109)
 * @version $Id: CosMultiPartRequest.java,v 1.3 2003/01/27 06:33:15 farkas Exp $
 */
public class CosMultiPartRequest extends MultiPartRequest
{
    /* ------------------------------------------------------------ */
    private MultipartRequest multi;

    /**
     * @param maxSize        maximum size post allowed
     * @param saveDir        the directory to save off the file
     * @param servletRequest the request containing the multipart
     */
    public CosMultiPartRequest(HttpServletRequest servletRequest, String saveDir, int maxSize)
            throws IOException
    {
        String encoding = getEncoding();
        if (encoding != null)
        {
            multi = new MultipartRequest(servletRequest, saveDir, maxSize, encoding);
        }
        else
        {
            multi = new MultipartRequest(servletRequest, saveDir, maxSize);
        }
    }

    /**
     * Set the encoding for the uploaded parameters.  This needs to be set if you are using character sets other than
     * ASCII.
     * <p/>
     * The encoding is looked up from the configuration setting 'webwork.i18n.encoding'.  This is usually set in
     * default.properties & webwork.properties.
     */
    private static String getEncoding()
    {
        String encoding = null;
        try
        {
            encoding = Configuration.getString("webwork.i18n.encoding");
        }
        catch (IllegalArgumentException e)
        {
            LogFactory.getLog(PellMultiPartRequest.class).info("Could not get encoding property 'webwork.i18n.encoding' for file upload.  Using system default");
        }
        return encoding;
    }

    public Enumeration getParameterNames()
    {
        return multi.getParameterNames();
    }

    public String getParameter(String name)
    {
        return multi.getParameter(name);
    }

    public String[] getParameterValues(String name)
    {
        return multi.getParameterValues(name);
    }

    // Methods only in MultipartRequest
    public Enumeration getFileNames()
    {
        return multi.getFileNames();
    }

    public String getFilesystemName(String name)
    {
        return multi.getFilesystemName(name);
    }

    public String getContentType(String name)
    {
        return multi.getContentType(name);
    }

    public File getFile(String name)
    {
        return multi.getFile(name);
    }
}
