/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.config;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.*;
import org.xml.sax.SAXException;
import webwork.config.util.XMLConfigurationReader;
import webwork.util.ClassLoaderUtils;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Access view configuration from an XML file.
 *
 * @author Rickard \u00D6berg (rickard@middleware-company.com)
 * @author Scott Farquhar (scott@atlassian.com)
 * @version $Revision: 1.20 $
 */
public class XMLActionConfiguration extends AbstractConfiguration
{
    // Attributes ----------------------------------------------------
    private XMLConfigurationReader configurationReader;
    private static final Log log = LogFactory.getLog(XMLActionConfiguration.class);
    private File file;
    private long lastModified;

    // Constructors --------------------------------------------------

    public XMLActionConfiguration(String aName)
    {
        URL fileUrl = ClassLoaderUtils.getResource(aName + ".xml", XMLActionConfiguration.class);
        if (fileUrl == null)
        {
            throw new IllegalArgumentException("No such XML resource:" + aName + ".xml");
        }
        file = new File(fileUrl.getFile());
        if (!file.exists() || !file.canRead())
        {
            file = null;
        }
        configurationReader = getMappingsFromResource(fileUrl);
        if (file != null)
        {
            lastModified = file.lastModified();
        }
    }

    protected XMLConfigurationReader getMappingsFromResource(URL url)
    {
        try
        {
            Document document = getDocument(url);

            log.debug("Found XML view configuration " + url);

            resolveIncludes(url, document);
            return new XMLConfigurationReader(document.getDocumentElement(), url.toExternalForm());

        }
        catch (DOMException e)
        {
            log.error("DOM exception", e);
            throw new IllegalArgumentException("Could not load XML action configuration");
        }
    }

    private Document getDocument(URL url)
    {
        try
        {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            return factory.newDocumentBuilder().parse(url.openStream());
        }
        catch (SAXException e)
        {
            log.error("SAX exception in " + url, e);
            throw new IllegalArgumentException("Could not parse XML action configuration: " + e);
        }
        catch (IOException e)
        {
            log.error("IO exception in " + url, e);
            throw new IllegalArgumentException("Could not load XML action configuration");
        }
        catch (ParserConfigurationException e)
        {
            log.error("Parser conf exception", e);
            throw new IllegalArgumentException("Could not load XML action configuration");
        }
    }

    private void resolveIncludes(URL parent, Document doc)
    {
        NodeList includes = doc.getDocumentElement().getElementsByTagName("include");
        List elements = new ArrayList(includes.getLength());
        for (int i = 0; i < includes.getLength(); i++)
        {
            Element element = (Element) includes.item(i);
            elements.add(element);
        }

        for (int i = 0; i < elements.size(); i++)
        {
            Element element = (Element) elements.get(i);
            String path = element.getAttribute("resource");
            URL url = null;
            if (path != null && path.length() > 0)
            {
                url = ClassLoaderUtils.getResource(path, getClass());
            }
            else
            {
                path = element.getAttribute("path");
                if (path != null && path.length() > 0)
                {
                    //hrm, disallow absolute includes? security risk?
                    if (path.charAt(0) == '/')
                    {
                        try
                        {
                            url = new File(path).toURL();
                        }
                        catch (MalformedURLException e)
                        {
                            //can't happen
                        }
                    }
                    else
                    {
                        try
                        {
                            File parentFile = new File(parent.getFile());
                            if (!parentFile.canRead() || !parentFile.exists())
                            {
                                log.warn("parentFile '" + parent + "' cannot be resolved so ignoring specified path include '" + path + "'");
                            }
                            url = new File(parentFile.getParentFile(), path).toURL();
                        }
                        catch (MalformedURLException e)
                        {
                            //can't happen
                        }
                    }
                }
            }
            if (url != null)
            {
                Document included = getDocument(url);
                resolveIncludes(url, included);
                DocumentFragment fragment = doc.createDocumentFragment();
                NodeList includedContents = included.getDocumentElement().getChildNodes();
                int includedSize = includedContents.getLength();
                for (int j = 0; j < includedSize; j++)
                {
                    Node imported = doc.importNode(includedContents.item(j), true);
                    fragment.appendChild(imported);
                }
                element.getParentNode().replaceChild(fragment, element);
            }
            else
            {
                log.warn("Included url '" + path + "' not found");
            }
        }
    }

    /**
     * Get a named setting. Note extension is stripped and replaced with extension defined in property file. This is
     * done here because of the recursive dependency if the constructor called Configuration.getString("webwork.action.extension").
     */
    public Object getImpl(String aName)
            throws IllegalArgumentException
    {
        //Possible recursion in getString, so we make sure that we shortcircuit the call for the reload property to prevent
        //a stackoverflow
        boolean reloadEnabled = false;
        if (!"webwork.configuration.xml.reload".equals(aName))
        {
            try
            {
                reloadEnabled = "true".equalsIgnoreCase(Configuration.getString("webwork.configuration.xml.reload"));
            }
            catch (Exception ex)
            {}
        }
        if (reloadEnabled && file != null && lastModified < file.lastModified())
        {
            lastModified = file.lastModified();
            log.debug("Reloading " + file);
            try
            {
                configurationReader = getMappingsFromResource(file.toURL());
            }
            catch (MalformedURLException e)
            {
                //can't really happen
                log.error("Something horrible happened", e);
            }
        }
        Object mapping = configurationReader.getActionMapping(aName);
        if (mapping == null)
        {
            throw new WebworkConfigurationNotFoundException(this.getClass(), "No such view mapping", aName);
        }

        return mapping;
    }


    public void setImpl(String aName, Object aValue)
    {
        throw new UnsupportedOperationException("May not update XML view mapping");
    }

    public Iterator listImpl()
    {
        return configurationReader.getActionMappingNames().iterator();
    }
}
