/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.view.velocity;

import org.apache.commons.logging.LogFactory;
import org.apache.velocity.app.Velocity;
import org.apache.velocity.context.Context;
import org.apache.velocity.exception.MethodInvocationException;
import org.apache.velocity.exception.ParseErrorException;
import org.apache.velocity.exception.ResourceNotFoundException;
import webwork.action.Action;
import webwork.action.factory.ActionFactory;
import webwork.util.ClassLoaderUtils;
import webwork.util.InjectionUtils;
import webwork.util.injection.ObjectFactory;

import java.io.CharArrayWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletOutputStream;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletResponse;

/**
 * WebWork utility methods for Velocity templates
 *
 * @author Rickard \u00D6berg (rickard@dreambean.com)
 * @version $Revision: 1.12 $
 */
public final class WebWorkUtil
{
    private Context ctx;
    ConcurrentMap<String,Class> classesMap = new ConcurrentHashMap<String,Class>();

    public WebWorkUtil(Context ctx)
    {
        this.ctx = ctx;
    }

    public Action action(Object aName)
            throws Exception
    {
        return ActionFactory.getAction(aName.toString());
    }

    public String execute(Action anAction)
            throws Exception
    {
        long start = System.currentTimeMillis();
        try
        {
            return anAction.execute();
        }
        finally
        {
            LogFactory.getLog(getClass()).debug("Execute:" + (System.currentTimeMillis() - start));
        }
    }

    public Object bean(Object aName)
            throws Exception
    {
        String name = aName.toString();
        Class c = classesMap.get(name);
        if (c == null)
        {
            c = ClassLoaderUtils.loadClass(name, WebWorkUtil.class);
            classesMap.put(name, c);
        }
        return ObjectFactory.instantiate(c);
    }

    public String include(Object aName, ServletRequest aRequest, ServletResponse aResponse)
            throws Exception
    {
        try
        {
            RequestDispatcher dispatcher = aRequest.getRequestDispatcher(aName.toString());
            if (dispatcher == null)
            {
                throw new IllegalArgumentException("Cannot find included file " + aName);
            }
            ServletResponseHandler responseHandler = new ServletResponseHandler(aResponse);
            Class[] interfaces = new Class[] { HttpServletResponse.class };
            HttpServletResponse response = (HttpServletResponse) Proxy.newProxyInstance(getClass().getClassLoader(), interfaces, responseHandler);

            dispatcher.include(aRequest, response);

            return responseHandler.getData();
        }
        catch (Exception e)
        {
            e.printStackTrace();
            throw e;
        }
    }

    public long toLong(int anInt)
    {
        return (long) anInt;
    }

    public long toLong(String aLong)
    {
        return Long.parseLong(aLong);
    }

    public int toInt(long aLong)
    {
        return (int) aLong;
    }

    public String toString(long aLong)
    {
        return Long.toString(aLong);
    }

    public String toString(int anInt)
    {
        return Integer.toString(anInt);
    }

    public String evaluate(String expression)
            throws IOException, ResourceNotFoundException, MethodInvocationException, ParseErrorException
    {
        CharArrayWriter writer = new CharArrayWriter();
        Velocity.evaluate(ctx, writer, "Error parsing " + expression, expression);
        return writer.toString();
    }

    static class ServletResponseHandler
            implements InvocationHandler
    {
        ServletResponse response;
        StringWriter strout;
        PrintWriter writer;
        ServletOutputStream sout;

        ServletResponseHandler(ServletResponse aResponse)
        {
            response = aResponse;
            strout = new StringWriter();
            sout = new ServletOutputStreamWrapper(strout);
            writer = new PrintWriter(strout);
        }

        public Object invoke(Object proxy,
                Method method,
                Object[] args)
                throws Throwable
        {
            if (method.getName().equals("getOutputStream"))
            {
                return getOutputStream();
            }
            else if (method.getName().equals("getWriter"))
            {
                return writer;
            }
            else
            {
                return InjectionUtils.invoke(method, response, args);
            }
        }

        ServletOutputStream getOutputStream()
        {
            return sout;
        }

        public String getData()
        {
            writer.flush();
            return strout.toString();
        }
    }

    static class ServletOutputStreamWrapper
            extends ServletOutputStream
    {
        Writer writer;

        ServletOutputStreamWrapper(Writer aWriter)
        {
            writer = aWriter;
        }

        public void write(int aByte)
        {
            try
            {
                writer.write(aByte);
            }
            catch (IOException e)
            {
                //this can/should never happen!
                e.printStackTrace();
            }
        }
    }
}