/*
* WebWork, Web Application Framework
*
* Distributable under Apache license.
* See terms of license at opensource.org
*/
package webwork.util;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import webwork.action.ActionContext;
import webwork.expr.Parser;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.beans.PropertyEditor;
import java.io.StringReader;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Value stack. A VS is used by the WebWork system as a way to make findInContext values available by using the push and
 * pop methods. They can then be accessed by using the find* methods.
 *
 * @author Rickard \u00D6berg (rickard@middleware-company.com)
 * @author Maurice C. Parker (maurice@vineyardenterprise.com)
 * @version $Revision: 1.61 $
 */
public class ValueStack implements Iterable<Object>
{
    // Static  -------------------------------------------------------
    public static final String STACK_NAME = "webwork.result";
    /**
     * A System property name that controls whether value stack exceptions  are logged or not.  By default they are
     * logged.
     */
    public static final String WEBWORK_VALUE_STACK_LOG_EXCEPTIONS = "webwork.valueStack.log.exceptions";

    protected static final Map<Class<?>, Map<String, MethodInfo[]>> classes = new ConcurrentHashMap<Class<?>, Map<String, MethodInfo[]>>(); // Method cache
    private final static Log log = LogFactory.getLog(ValueStack.class);

    /**
     * Clear the method cache. This must be called if the application is restarted.
     */
    public static void clearMethods()
    {
        classes.clear();
    }

    // Attributes ----------------------------------------------------
    private final List<Object> valueList = new ArrayList<Object>();
    private Parser parser;

    // Constructor ---------------------------------------------------
    public ValueStack()
    {}

    // Public --------------------------------------------------------
    /**
     * Push a value onto the value stack.
     *
     * @param value the value
     */
    public void pushValue(final Object value)
    {
        valueList.add(value);
    }

    /**
     * Peek at the object that is at the top of the stack. The object is returned unmodified, so if it would be a
     * ValueHolder it is not unwrapped
     */
    public Object peek()
    {
        final int size = valueList.size();
        if (size < 1)
        {
            return null;
        }
        return valueList.get(size - 1);
    }

    /**
     * Pop a value from the value stack.
     *
     * @return the popped value
     */
    public Object popValue()
    {

        final int size = valueList.size();
        if (size < 1)
        {
            return null;
        }
        return valueList.remove(size - 1);
    }

    public Iterator<Object> iterator()
    {
        return valueList.iterator();
    }

    /**
     * Returns the size of the value stack.
     *
     * @return size of value stack
     */
    public int size()
    {
        return valueList.size();
    }

    /**
     * Returns TRUE is the value stack is empty.
     *
     * @return true is value stack is empty
     */
    public boolean isEmpty()
    {
        return size() == 0;
    }

    /**
     * Resolve a WebWork expression statement.
     *
     * @param expression
     *
     * @return the boolean result of the expression
     */
    public boolean test(final String expression)
    {
        if (expression == null)
        {
            throw new NullPointerException("Expression cannot be null.");
        }
        boolean answer = false;
        try
        {
            final SimpleTest simpleTest = SimpleTest.getSimpleTest(expression);
            if (simpleTest != null)
            {
                answer = simpleTest.test(this, null, null);
            }
            else
            {
                final Parser p = getParser(expression);
                answer = p.test();
            }
        }
        catch (final Throwable pe)
        {
            log.error("An error occurred while parsing the expression: \"" + expression + "\", throwable: ", pe);
            throw new IllegalArgumentException("\n\nAn error occurred while parsing the expression: \n    \"" + expression + "\"\n" + pe.getMessage());
        }
        return answer;
    }

    /**
     * Find a value by id. This method can be overridden by subclasses to have some context concept to evaluate
     *
     * @param id the value id.
     *
     * @return value
     */
    protected Object findInContext(final String id)
    {
        return null;
    }

    protected Object unwrap(final Object value)
    {
        if (value instanceof ValueHolder)
        {
            return ((ValueHolder) value).getValue();
        }
        return value;
    }

    /**
     * Find a value for a given name.
     *
     * @param query
     *
     * @return the object corresponding to the query
     */
    public Object findValue(final String query) throws IllegalArgumentException
    {
        Query q;
        if ((query == null) || (query.length() == 0) || ((query.length() == 1) && (query.charAt(0) == '.')))
        {
            q = Query.CURRENT;
        }
        else
        {
            // The query segments and the current segment
            q = Query.getQuery(query);
        }
        return findValue(q);
    }

    public Object findValue(final Query q) throws IllegalArgumentException
    {
        // The query segments and the current segment
        final QuerySegment[] segments = q.getSegments();
        //      if (log.isDebugEnabled()) {
        //         log.debug("findValue() for: " + q);
        //      }

        QuerySegment segment = segments[0];
        int segmentIdx = 1;

        // The current stack pointer, and the current object,
        int stackIdx = 0;
        int size;
        Object value;

        /////////////////////////////////////////////////////////////////////////
        // evaluate the first element of the expression to see where to
        // get the requested value. These should be quick and easy objects
        // to find or create.
        /////////////////////////////////////////////////////////////////////////
        switch (segment.getType())
        {

            // get the top value off of the stack
            case QuerySegment.CURRENT:

                size = valueList.size();
                if (size < 1)
                {
                    return null;
                }
                // set up the stack, pointer, and current value
                value = valueList.get(size - 1);

                // always have the next segment ready to go
                segment = segments[segmentIdx++];

                // if we don't need to get other values then return the value
                if (segment == null)
                {
                    return unwrap(value);
                }

                // The stackIdx is already zero so we will not try to search the stack
                break;

            // return the id since it is the actual string
            case QuerySegment.STRING:
                return segment.getId();

            // the integer is the first value and only value in the segment
            case QuerySegment.NUMBER:
                return segment.getValues().get(0);

            // get an attribute
            case QuerySegment.ATTRIBUTE:

                // get the attribute
                value = findInContext(segment.getId());

                if (value == null)
                {
                    //               if (log.isDebugEnabled()) {
                    //                  log.debug("value for [" + q + "] is null.");
                    //               }

                    return null;

                    //throw new IllegalArgumentException("No such attribute: " + token.image);
                }

                // always have the next segment ready to go
                segment = segments[segmentIdx++];

                // if we don't need to search through this attribute simply return it
                if (segment == null)
                {
                    return unwrap(value);
                }

                // The stackIdx is already zero so we will not try to search the stack

                break;

            // return the http request parameter
            case QuerySegment.PARAMETER:
                return getParameter(segment.getId());

            // the reserved keyword "true"
            case QuerySegment.TRUE:
                return Boolean.TRUE;

            // the reserved keyword "false"
            case QuerySegment.FALSE:
                return Boolean.FALSE;

            // the reserved keyword "null"
            case QuerySegment.NULL:
                return null;
            // A concatenation query
            case QuerySegment.CONCAT:
                // Get the list of queries from the segment and lookup each of them.
                final List<?> concatList = segment.getValues();
                final StringBuilder buf = new StringBuilder();
                for (int i = 0; i < concatList.size(); i++)
                {
                    final Object concatValue = findValue((Query) concatList.get(i));
                    // Convert the found value to a String using BeanUtil and then append
                    // it to the resulting StringBuilder
                    buf.append(BeanUtil.toStringValue(concatValue));
                }
                return buf.toString();

            // get the root object off of the bottom of the stack
            case QuerySegment.ROOT:
                if (valueList.size() < 1)
                {
                    return null;
                }

                // The stackIdx is already zero so we will not try to search the stack
                // set up the stack pointer, and current value
                value = valueList.get(0);

                // always have the next segment ready to go
                segment = segments[segmentIdx++];

                // if we don't need to search through the stack then return the value
                // this is very unlikely to happen, but we have account for it anyway
                if (segment == null)
                {
                    return unwrap(value);
                }

                break;

            default:
                size = valueList.size();
                if (size < 1)
                {
                    return null;
                }
                // set up the stack pointer, and current value
                stackIdx = size - 1;
                value = valueList.get(stackIdx);
                break;
        }

        //log.debug( "first segment id: '" + segment.getId() + "' segment type: '" + segment.getType() + "'" );

        /////////////////////////////////////////////////////////////////////////
        // Now that stack has been set up and the context set (stackIdx) we will
        // begin parsing the rest of the expression and drilling down through
        // the object properties, collection elements, and method calls.
        /////////////////////////////////////////////////////////////////////////
        final int saveSegmentIdx = segmentIdx;
        boolean gotResult = false;
        while (true)
        {
            //log.debug("beginning valuestack search at level: '" + stackIdx);

            int workStackIdx = stackIdx;
            if (value != null)
            {
                objectWalk:
                do
                {
                    gotResult = false;
                    switch (segment.getType())
                    {

                        // access a classes property
                        case QuerySegment.PROPERTY:
                            // get the real value
                            value = unwrap(value);
                            // If the value is null we break out now since it
                            // would only result in an exception anyway
                            if (value == null)
                            {
                                break objectWalk;
                            }

                            //log.debug( "PROPERTY: attempting to get: " + segment.getId() );
                            try
                            {
                                final MethodInfo[] methods = getMethod(value.getClass(), segment.getId());
                                if (methods == null)
                                {
                                    //log.debug( "PROPERTY: method not found: '" + segment.getId() + "' current value: '" + value + "' (" + value.getClass() + ")");
                                    break objectWalk;
                                }
                                else
                                {
                                    if (methods[0].getNrOfParameters() == 0)
                                    {
                                        try
                                        {
                                            value = InjectionUtils.invoke(methods[0].getMethod(), value, null);
                                        }
                                        catch (final Exception e)
                                        {
                                            logValueStackException(e, q);
                                            throw e;
                                        }
                                        gotResult = true;
                                        //log.debug( "PROPERTY: found property value: " + value  + " (" + value.getClass() + ")");
                                    }
                                    else
                                    {
                                        break objectWalk;
                                    }
                                }
                            }
                            catch (final Exception e)
                            {
                                //log.debug( "PROPERTY: method called failed: " + e.getMessage() );
                                break objectWalk;
                            }
                            break;

                        // access a class method
                        case QuerySegment.METHOD:

                            // get the real value
                            value = unwrap(value);
                            // If the value is null we break out now since it
                            // would only result in an exception anyway
                            if (value == null)
                            {
                                break objectWalk;
                            }

                            //log.debug( "going after method: " + segment.getId() );

                            Object[] params = null;
                            MethodInfo target = null;
                            try
                            {

                                final MethodInfo[] methods = getMethod(value.getClass(), segment.getId());

                                if (methods == null)
                                {
                                    //log.debug( "METHOD: "  + segment.getId() + " was not found." );
                                    break objectWalk;
                                }

                                final List<?> pValueList = segment.getValues();
                                final int vsize = pValueList.size();
                                params = new Object[vsize];
                                for (int i = 0; i < vsize; i++)
                                {
                                    final Object param = findValue((Query) pValueList.get(i));
                                    params[i] = param;
                                }

                                target = findMethod(methods, params);
                                if (target == null)
                                {
                                    log.error("No method found for " + segment.getId() + " with parameters " + Arrays.asList(getParameterClasses(params)) + " in class " + value.getClass());
                                    break objectWalk;
                                }
                                // Convert if necessary
                                final Class<?>[] parameterTypes = target.getParameterTypes();
                                final int paramCount = parameterTypes.length;
                                for (int i = 0; i < paramCount; i++)
                                {
                                    final Class<?> parameterClass = parameterTypes[i];
                                    final Object parami = params[i];

                                    //log.debug(paramCount + ": " + parameterClass.getName());
                                    // If the parameter is not a String then we might try converting it
                                    if (!parameterClass.equals(String.class))
                                    {
                                        // If the parameter is null we will not try a conversion
                                        if (parami == null)
                                        {
                                            // If the parameterClass is primitive then null is not an acceptable argument
                                            if (parameterClass.isPrimitive())
                                            {
                                                break objectWalk;
                                            }
                                        }
                                        else if (!parameterClass.equals(parami.getClass()) && !parameterClass.isAssignableFrom(parami.getClass()))
                                        {
                                            // Get property editor
                                            final PropertyEditor pe = BeanUtil.getPropertyEditor(parameterClass);
                                            // Convert value
                                            if (pe != null)
                                            {
                                                final Object param = BeanUtil.getAsValue(pe, parami.toString());
                                                //replace with the converted value
                                                params[i] = param;
                                            }
                                        }
                                    }
                                }
                                value = InjectionUtils.invoke(target.getMethod(), value, params);
                                gotResult = true;
                            }
                            catch (final IllegalArgumentException e)
                            {
                                log.error("Illegal parameters invoking " + value.getClass() + "." + target.getMethod().getName() + "(" + Arrays.asList(getParameterClasses(params)) + ")");
                                break objectWalk;
                            }
                            catch (final Exception e)
                            {
                                Throwable throwable = e;
                                if (e instanceof InvocationTargetException)
                                {
                                    throwable = ((InvocationTargetException) e).getTargetException();
                                }
                                log.error("METHOD: \"" + segment.getId() + "\", exception: ", throwable);
                                break objectWalk;
                                //throw new IllegalArgumentException("Invalid single method access. " +
                                //                                   "Error accessing method \"" + token.image +
                                //                                   "\" using parameter: \"" + param + "\"");
                            }
                            break;

                        // access the current value as a collection
                        case QuerySegment.COLLECTION:
                            // get the real value
                            value = unwrap(value);
                            // If the value is null we break out now since it
                            // would result in an uncaught exception otherwise
                            if (value == null)
                            {
                                break objectWalk;
                            }

                            final Object key = findValue(segment.getQuery());
                            if (key == null)
                            {
                                break objectWalk;
                            }

                            // Map
                            if (value instanceof Map)
                            {
                                value = ((Map<?, ?>) value).get(key);
                                gotResult = true;
                                break;
                            }
                            // List
                            else if (value instanceof List)
                            {
                                value = ((List<?>) value).get(((Integer) key).intValue());
                                gotResult = true;
                                break;
                            }
                            // Array
                            else if (value.getClass().isArray())
                            {
                                value = Array.get(value, ((Integer) key).intValue());
                                gotResult = true;
                                break;
                            }
                            // Resource Bundle
                            else if (value instanceof ResourceBundle)
                            {
                                value = ((ResourceBundle) value).getObject(key.toString());
                                gotResult = true;
                                break;
                            }
                            // Collection
                            else if (value instanceof Collection)
                            {
                                // Not very efficient, but at least it works
                                value = ((Collection<?>) value).toArray()[((Integer) key).intValue()];
                                gotResult = true;
                                break;
                            }

                            // fail if the user tries to access something other than a Collection
                            break objectWalk;

                        // access the parent by going up one level on the stack
                        case QuerySegment.PARENT:
                            workStackIdx--;
                            if (workStackIdx < 0)
                            {
                                break objectWalk;
                                //throw new IllegalArgumentException("Parent object not available.");
                            }
                            else
                            {
                                value = valueList.get(workStackIdx);
                                gotResult = true;
                            }
                            break;

                        case QuerySegment.EXPAND:
                            //log.debug( "EXPAND: going after: " + segment.getId() );
                            try
                            {

                                final Object methodName = findValue(segment.getQuery());
                                if (methodName == null)
                                {
                                    break objectWalk;
                                }
                                value = findValue(methodName.toString());
                                gotResult = true;
                            }
                            catch (final Exception e)
                            {
                                break objectWalk;
                            }
                            break;
                    }
                    //log.debug( "finished case statement" );
                    // always have the next segment ready to go
                    segment = segments[segmentIdx++];
                }
                while (segment != null);
            }

            // if we didn't find the value, then move one down the stack and
            // try again.
            if (!gotResult && (stackIdx > 0))
            {
                stackIdx--;
                value = valueList.get(stackIdx);

                // reset the segment index to reset the search
                segmentIdx = saveSegmentIdx;
                segment = segments[segmentIdx - 1];
            }
            else
            {
                break;
            }
        }

        // If no result was found then return null
        if (!gotResult)
        {
            return null;
            //      if (value == null && log.isDebugEnabled())
            //         log.debug("value for [" + q + "] is null.");
        }

        // get the real value
        return unwrap(value);
    }

    /**
     * By default we log ValueStack exceptions unless  'webwork.valueStack.log.exceptions' is set to false as a System
     * property.
     *
     * @param e the exception
     * @param q the query stack
     */
    private void logValueStackException(final Exception e, Query q)
    {
        boolean logException = true;
        try
        {
            final String logExceptionValue = System.getProperty(WEBWORK_VALUE_STACK_LOG_EXCEPTIONS);
            if (logExceptionValue != null)
            {
                logException = Boolean.parseBoolean(logExceptionValue);
            }
        }
        catch (Exception t)
        {
            // needed in case of SecurityManager in play
        }
        if (logException)
        {
            log.error(q.toString(), e);
        }
    }

    private Object[] getParameterClasses(final Object[] params)
    {
        final Object[] classes = new Object[params.length];
        for (int i = 0; i < params.length; i++)
        {
            classes[i] = params[i] != null ? params[i].getClass() : null;
        }
        return classes;
    }

    /**
     * Return a string representation of the Stack
     *
     * @return the stack as a String
     */
    @Override
    public String toString()
    {
        String str = "Value stack\n";
        str += "===========\n";
        for (int i = 0; i < valueList.size(); i++)
        {
            final Object val = valueList.get(i);
            str += val == null ? "null\n" : val.toString() + "\n";
        }
        str += "===========\n";
        return str;
    }

    /**
     * Get the parser associated with this ValueStack
     *
     * @return the Parser for this ValueStack
     */
    private Parser getParser(final String expression)
    {
        if (parser == null)
        {
            parser = new Parser(new StringReader(expression));
            parser.setValueStack(this);
        }
        else
        {
            parser.ReInit(new StringReader(expression));
        }
        return parser;
    }

    /**
     * Get a method with a given name.
     *
     * @param cl   the class of the method
     * @param name the name of the method
     *
     * @return the wanted method
     *
     * @throws IntrospectionException
     */
    protected MethodInfo[] getMethod(final Class<?> cl, final String name) throws IntrospectionException
    {
        Map<String, MethodInfo[]> methods = classes.get(cl);
        if (methods == null)
        {
            // Get methods that can be invoked for this class
            methods = new ConcurrentHashMap<String, MethodInfo[]>();

            // Find get methods for properties
            final BeanInfo bi = Introspector.getBeanInfo(cl);
            final PropertyDescriptor[] pd = bi.getPropertyDescriptors();
            for (int i = 0; i < pd.length; ++i)
            {
                Method method = pd[i].getReadMethod();

                // Check if readable property
                if (method != null)
                {
                    if (!Modifier.isPublic(cl.getModifiers()))
                    {
                        // Find a method in an interface that *is* public
                        final Class<?>[] interfaces = bi.getBeanDescriptor().getBeanClass().getInterfaces();
                        for (final Class<?> interface1 : interfaces)
                        {
                            try
                            {
                                //log.debug("Try "+interfaces[j]);
                                method = interface1.getMethod(method.getName(), new Class[0]);
                                break;
                            }
                            catch (final Exception e)
                            {
                                // Ignore
                            }
                        }

                        // We're in trouble! Try to sneak through security
                        if (method.equals(pd[i].getReadMethod()))
                        {
                            AccessibleObject.setAccessible(new AccessibleObject[] { method }, true);
                        }
                    }

                    // Save method in map
                    //log.debug( "GET_METHOD: class: " + cl + " method: " + method.getName());
                    methods.put(pd[i].getName(), new MethodInfo[] { new MethodInfo(method) });
                }
            }

            // Find param methods
            final Method[] getters = cl.getMethods();
            for (Method method : getters)
            {
                if (!method.getName().startsWith("set") && !method.getReturnType().equals(Void.TYPE))
                {
                    // Valid method
                    // Check if public
                    if (!Modifier.isPublic(cl.getModifiers()))
                    {
                        // Find a method in an interface that *is* public
                        final Class<?>[] interfaces = cl.getInterfaces();
                        for (final Class<?> interface1 : interfaces)
                        {
                            try
                            {
                                method = interface1.getMethod(method.getName(), method.getParameterTypes());
                                break;
                            }
                            catch (final Exception e)
                            {
                                // Ignore
                            }
                        }

                        // We're in trouble! Try to sneak through security
                        if (method.equals(method))
                        {
                            AccessibleObject.setAccessible(new AccessibleObject[] { method }, true);
                        }
                    }

                    // Get name
                    String methodName = method.getName();
                    if (methodName.startsWith("get"))
                    {
                        methodName = Introspector.decapitalize(methodName.substring(3));
                    }
                    else if (methodName.startsWith("is"))
                    {
                        methodName = Introspector.decapitalize(methodName.substring(2));
                    }

                    // Save method in map
                    final MethodInfo[] current = methods.get(methodName);
                    MethodInfo[] newlist;
                    if (current == null)
                    {
                        newlist = new MethodInfo[] { new MethodInfo(method) };
                    }
                    else
                    {
                        newlist = new MethodInfo[current.length + 1];
                        System.arraycopy(current, 0, newlist, 0, current.length);
                        newlist[current.length] = new MethodInfo(method);
                    }
                    methods.put(methodName, newlist);
                    //log.debug( "GET_METHOD: class: " + cl + " method: " + method.getName());
                }
            }

            // Add map to class map
            classes.put(cl, methods);
            //log.debug("Added "+cl+" to class map:"+methods);
        }

        // Get named method/property getter
        final MethodInfo[] nameMethods = methods.get(name);

        //log.debug("Got "+cl+" "+name+":"+m);
        return nameMethods;
    }

    protected MethodInfo findMethod(final MethodInfo[] m, final Object[] params)
    {
        if (m.length == 1)
        {
            // Check if the method with the matching name also expects
            // the same number of parameters
            return (m[0].getNrOfParameters() == params.length ? m[0] : null);
        }

        MethodInfo oneMatch = null;
        List<MethodInfo> match = null;
        final int noOfArgument = params.length;

        // Do not create the ArrayList when it is not necessary (the most common case!)
        for (final MethodInfo element : m)
        {
            if (element.getNrOfParameters() == noOfArgument)
            {
                if (oneMatch == null)
                {
                    oneMatch = element;
                }
                else
                {
                    if (match == null)
                    {
                        match = new ArrayList<MethodInfo>();
                        match.add(oneMatch);
                    }
                    match.add(element);
                }
            }
        }

        // If there is no list then there is either 1 match or 0 matches.
        // Just return the oneMatch variable then
        if (match == null)
        {
            return oneMatch;
        }

        //log.debug("No of match for: " + m[0].getName() + " , " + noOfArgument + ", " + match.size());

        MethodInfo exact = null;
        final List<MethodInfo> close = new ArrayList<MethodInfo>();
        final List<MethodInfo> convert = new ArrayList<MethodInfo>();

        for (final MethodInfo current : match)
        {
            final Class<?>[] paramClass = current.getParameterTypes();
            boolean exactMatch = true;
            boolean closeMatch = true;
            boolean convertable = true;

            for (int j = 0; j < paramClass.length; j++)
            {
                final Class<?> p = params[j].getClass();
                //log.debug("Argument: " + j + " , parameterClass: " + paramClass[j].getName()
                //+ " , argumentClass: " + p.getName());
                if (paramClass[j].getName().equals(p.getName()))
                {
                    continue;
                }
                else if (paramClass[j].isAssignableFrom(p))
                {
                    exactMatch = false;
                    continue;
                }
                else
                {
                    exactMatch = false;
                    closeMatch = false;
                    try
                    {
                        // Get property editor
                        final PropertyEditor pe = BeanUtil.getPropertyEditor(paramClass[j]);
                        if (pe == null)
                        {
                            convertable = false;
                        }
                        else
                        {
                            // Convert value
                            BeanUtil.getAsValue(pe, params[j].toString());
                        }
                    }
                    catch (final Exception e)
                    {
                        convertable = false;
                    }

                    if (!convertable)
                    {
                        break;
                    }
                }
            }

            if (exactMatch)
            {
                exact = current;
                break;
            }
            else if (closeMatch)
            {
                close.add(current);
            }
            else if (convertable)
            {
                convert.add(current);
            }

        }

        if (exact != null)
        {
            return exact;
        }
        else if (close.size() > 0)
        {
            return close.get(0);
        }
        else if (convert.size() > 0)
        {
            return convert.get(0);
        }
        else
        {
            return null;
        }

    }

    /**
     * Class that holds information about a method and the parameter types that the method accepts, and the number of
     * parameters
     */
    private static class MethodInfo
    {
        private final Method _method;
        private final Class<?>[] _parameterTypes;
        private final int _nrOfParameters;

        public MethodInfo(final Method method)
        {
            _method = method;
            _parameterTypes = method.getParameterTypes();
            _nrOfParameters = _parameterTypes.length;
        }

        public Method getMethod()
        {
            return _method;
        }

        public Class<?>[] getParameterTypes()
        {
            return _parameterTypes;
        }

        public int getNrOfParameters()
        {
            return _nrOfParameters;
        }
    }

    protected Object getParameter(final String aName)
    {
        return ActionContext.getParameters().get(aName);
    }

    // Inner classes -------------------------------------------------
    // Value providers that want to use lazy evaluation should use this
    // interface
    public interface ValueHolder
    {
        // Public -----------------------------------------------------
        public Object getValue();
    }
}
