/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.util;

import java.util.ArrayList;
import java.util.List;

/**
 * A segment of a ValueStack Query.
 *
 * @author Maurice C. Parker (maurice@vineyardenterprise.com)
 * @version $Revision: 1.11 $
 */
public class QuerySegment
{
    // Attributes ----------------------------------------------------
    public final static int STRING = 0;
    public final static int TRUE = 1;
    public final static int FALSE = 2;
    public final static int CURRENT = 3;
    public final static int PARENT = 4;
    public final static int ROOT = 5;
    public final static int ATTRIBUTE = 6;
    public final static int PARAMETER = 7;
    public final static int PROPERTY = 8;
    public final static int COLLECTION = 9;
    public final static int METHOD = 10;
    public final static int EXPAND = 11;
    public final static int NUMBER = 12;
    public final static int NULL = 13;
    public final static int CONCAT = 14;

    private final Query query;
    private final String id;
    private final int type;
    private final List<Object> values = new ArrayList<Object>();

    // Public --------------------------------------------------------
    public QuerySegment(final int type)
    {
        this.type = type;
        id = null;
        query = null;
    }

    public QuerySegment(final String id, final int type)
    {
        this.type = type;
        this.id = id;
        query = null;
    }

    public QuerySegment(final String id, final Query query, final int type)
    {
        this.type = type;
        this.id = id;
        this.query = query;
    }

    /**
     * Add a value that is associated with this QuerySegment.
     */
    public void addValue(final Object value)
    {
        values.add(value);
    }

    public void createValues()
    {
        values.clear();
    }

    /**
     * Return the identification string.
     */
    public String getId()
    {
        return id;
    }

    public Query getQuery()
    {
        return query;
    }

    /**
     * Return this QuerySegments type.  The type is used to identify if this QuerySegment will be used to access an
     * object property, an object method, or a Collection.
     */
    public int getType()
    {
        return type;
    }

    /**
     * Returns a list of values for this QuerySegment.
     */
    public List<Object> getValues()
    {
        return values;
    }

    @Override
    public String toString()
    {
        final StringBuilder sb = new StringBuilder();
        sb.append("\"");
        if (values != null)
        {
            for (int i = 0; i < values.size(); i++)
            {
                sb.append(values.get(i));
                if ((i + 1) < values.size())
                {
                    sb.append(",");
                }
            }
        }
        sb.append("\"");
        return "[id=\"" + id + "\" type=\"" + type + "\" values=" + sb.toString() + "]";
    }

}
